//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Personnel Assistant Widget                                  ║
//  ║                    Dark Blood Theme - WidgetKit                               ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import WidgetKit
import SwiftUI
import AppIntents

// MARK: - Widget Configuration

struct PersonnelAssistantWidget: Widget {
    let kind: String = "PersonnelAssistantWidget"
    
    var body: some WidgetConfiguration {
        AppIntentConfiguration(
            kind: kind,
            intent: AssistantWidgetIntent.self,
            provider: AssistantTimelineProvider()
        ) { entry in
            AssistantWidgetEntryView(entry: entry)
                .containerBackground(for: .widget) {
                    WidgetBackground()
                }
        }
        .configurationDisplayName("Personnel Assistant")
        .description("Accès rapide à votre assistant AI CodinCloud")
        .supportedFamilies([.systemSmall, .systemMedium, .systemLarge, .accessoryCircular, .accessoryRectangular])
    }
}

// MARK: - Widget Intent

struct AssistantWidgetIntent: WidgetConfigurationIntent {
    static var title: LocalizedStringResource = "Configuration Assistant"
    static var description = IntentDescription("Configurez votre widget Personnel Assistant")
    
    @Parameter(title: "Afficher le statut", default: true)
    var showStatus: Bool
    
    @Parameter(title: "Action rapide", default: .voice)
    var quickAction: QuickActionType
}

enum QuickActionType: String, AppEnum {
    case voice = "voice"
    case chat = "chat"
    case infrastructure = "infrastructure"
    
    static var typeDisplayRepresentation = TypeDisplayRepresentation(name: "Action Rapide")
    static var caseDisplayRepresentations: [QuickActionType: DisplayRepresentation] = [
        .voice: "Commande Vocale",
        .chat: "Ouvrir Chat",
        .infrastructure: "Infrastructure"
    ]
}

// MARK: - Timeline Provider

struct AssistantTimelineProvider: AppIntentTimelineProvider {
    typealias Entry = AssistantWidgetEntry
    typealias Intent = AssistantWidgetIntent
    
    func placeholder(in context: Context) -> AssistantWidgetEntry {
        AssistantWidgetEntry(
            date: Date(),
            status: .online,
            lastMessage: "Bienvenue!",
            servicesCount: 24,
            configuration: AssistantWidgetIntent()
        )
    }
    
    func snapshot(for configuration: AssistantWidgetIntent, in context: Context) async -> AssistantWidgetEntry {
        AssistantWidgetEntry(
            date: Date(),
            status: .online,
            lastMessage: "Comment puis-je vous aider?",
            servicesCount: 24,
            configuration: configuration
        )
    }
    
    func timeline(for configuration: AssistantWidgetIntent, in context: Context) async -> Timeline<AssistantWidgetEntry> {
        let entry = AssistantWidgetEntry(
            date: Date(),
            status: .online,
            lastMessage: "Prêt à vous assister",
            servicesCount: 24,
            configuration: configuration
        )
        
        // Refresh every 15 minutes
        let nextUpdate = Calendar.current.date(byAdding: .minute, value: 15, to: Date())!
        return Timeline(entries: [entry], policy: .after(nextUpdate))
    }
}

// MARK: - Widget Entry

struct AssistantWidgetEntry: TimelineEntry {
    let date: Date
    let status: AssistantStatus
    let lastMessage: String
    let servicesCount: Int
    let configuration: AssistantWidgetIntent
    
    enum AssistantStatus {
        case online, offline, busy
        
        var color: Color {
            switch self {
            case .online: return Color(hex: "228B22")
            case .offline: return Color(hex: "FF4444")
            case .busy: return Color(hex: "FF8C00")
            }
        }
        
        var text: String {
            switch self {
            case .online: return "En ligne"
            case .offline: return "Hors ligne"
            case .busy: return "Occupé"
            }
        }
    }
}

// MARK: - Widget Background

struct WidgetBackground: View {
    var body: some View {
        ZStack {
            // Base gradient
            LinearGradient(
                colors: [
                    Color(hex: "0D0D0D"),
                    Color(hex: "1A0A0A"),
                    Color(hex: "150808")
                ],
                startPoint: .topLeading,
                endPoint: .bottomTrailing
            )
            
            // Blood glow effect
            RadialGradient(
                colors: [
                    Color(hex: "8B0000").opacity(0.3),
                    Color.clear
                ],
                center: .topTrailing,
                startRadius: 0,
                endRadius: 200
            )
        }
    }
}

// MARK: - Widget Entry View

struct AssistantWidgetEntryView: View {
    @Environment(\.widgetFamily) var family
    var entry: AssistantWidgetEntry
    
    var body: some View {
        switch family {
        case .systemSmall:
            SmallWidgetView(entry: entry)
        case .systemMedium:
            MediumWidgetView(entry: entry)
        case .systemLarge:
            LargeWidgetView(entry: entry)
        case .accessoryCircular:
            CircularAccessoryView(entry: entry)
        case .accessoryRectangular:
            RectangularAccessoryView(entry: entry)
        default:
            SmallWidgetView(entry: entry)
        }
    }
}

// MARK: - Small Widget

struct SmallWidgetView: View {
    let entry: AssistantWidgetEntry
    
    var body: some View {
        VStack(spacing: 8) {
            // Logo with glow
            ZStack {
                Circle()
                    .fill(
                        RadialGradient(
                            colors: [Color(hex: "DC143C").opacity(0.4), Color.clear],
                            center: .center,
                            startRadius: 0,
                            endRadius: 30
                        )
                    )
                    .frame(width: 50, height: 50)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 28))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [Color(hex: "FF2D2D"), Color(hex: "B22222")],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
            }
            
            Text("dmagh")
                .font(.system(size: 14, weight: .bold, design: .rounded))
                .foregroundColor(.white)
            
            // Status indicator
            HStack(spacing: 4) {
                Circle()
                    .fill(entry.status.color)
                    .frame(width: 6, height: 6)
                Text(entry.status.text)
                    .font(.system(size: 10))
                    .foregroundColor(Color(hex: "B0B0B0"))
            }
        }
        .widgetURL(URL(string: "personnelassistant://voice"))
    }
}

// MARK: - Medium Widget

struct MediumWidgetView: View {
    let entry: AssistantWidgetEntry
    
    var body: some View {
        HStack(spacing: 16) {
            // Left: Logo and status
            VStack(spacing: 8) {
                ZStack {
                    Circle()
                        .fill(
                            RadialGradient(
                                colors: [Color(hex: "DC143C").opacity(0.4), Color.clear],
                                center: .center,
                                startRadius: 0,
                                endRadius: 35
                            )
                        )
                        .frame(width: 60, height: 60)
                    
                    Image(systemName: "brain.head.profile")
                        .font(.system(size: 32))
                        .foregroundStyle(
                            LinearGradient(
                                colors: [Color(hex: "FF2D2D"), Color(hex: "B22222")],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            )
                        )
                }
                
                HStack(spacing: 4) {
                    Circle()
                        .fill(entry.status.color)
                        .frame(width: 8, height: 8)
                    Text(entry.status.text)
                        .font(.system(size: 11, weight: .medium))
                        .foregroundColor(Color(hex: "B0B0B0"))
                }
            }
            .frame(width: 80)
            
            // Right: Actions
            VStack(alignment: .leading, spacing: 10) {
                Text("Personnel Assistant")
                    .font(.system(size: 14, weight: .bold))
                    .foregroundColor(.white)
                
                // Quick actions
                HStack(spacing: 8) {
                    WidgetActionButton(
                        icon: "mic.fill",
                        label: "Vocal",
                        color: Color(hex: "DC143C"),
                        url: "personnelassistant://voice"
                    )
                    
                    WidgetActionButton(
                        icon: "message.fill",
                        label: "Chat",
                        color: Color(hex: "8B0000"),
                        url: "personnelassistant://chat"
                    )
                    
                    WidgetActionButton(
                        icon: "server.rack",
                        label: "Infra",
                        color: Color(hex: "228B22"),
                        url: "personnelassistant://infrastructure"
                    )
                }
                
                // Services count
                HStack(spacing: 4) {
                    Image(systemName: "square.stack.3d.up.fill")
                        .font(.system(size: 10))
                    Text("\(entry.servicesCount) services actifs")
                        .font(.system(size: 10))
                }
                .foregroundColor(Color(hex: "6B6B6B"))
            }
        }
        .padding(.horizontal, 4)
    }
}

struct WidgetActionButton: View {
    let icon: String
    let label: String
    let color: Color
    let url: String
    
    var body: some View {
        Link(destination: URL(string: url)!) {
            VStack(spacing: 4) {
                ZStack {
                    RoundedRectangle(cornerRadius: 8)
                        .fill(color.opacity(0.2))
                        .frame(width: 36, height: 36)
                    
                    Image(systemName: icon)
                        .font(.system(size: 16))
                        .foregroundColor(color)
                }
                
                Text(label)
                    .font(.system(size: 9, weight: .medium))
                    .foregroundColor(Color(hex: "B0B0B0"))
            }
        }
    }
}

// MARK: - Large Widget

struct LargeWidgetView: View {
    let entry: AssistantWidgetEntry
    
    var body: some View {
        VStack(spacing: 16) {
            // Header
            HStack {
                ZStack {
                    Circle()
                        .fill(
                            RadialGradient(
                                colors: [Color(hex: "DC143C").opacity(0.4), Color.clear],
                                center: .center,
                                startRadius: 0,
                                endRadius: 30
                            )
                        )
                        .frame(width: 50, height: 50)
                    
                    Image(systemName: "brain.head.profile")
                        .font(.system(size: 26))
                        .foregroundStyle(
                            LinearGradient(
                                colors: [Color(hex: "FF2D2D"), Color(hex: "B22222")],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            )
                        )
                }
                
                VStack(alignment: .leading, spacing: 2) {
                    Text("Personnel Assistant")
                        .font(.system(size: 16, weight: .bold))
                        .foregroundColor(.white)
                    
                    HStack(spacing: 4) {
                        Circle()
                            .fill(entry.status.color)
                            .frame(width: 8, height: 8)
                        Text(entry.status.text)
                            .font(.system(size: 12))
                            .foregroundColor(Color(hex: "B0B0B0"))
                    }
                }
                
                Spacer()
                
                // Say "dmagh" badge
                VStack(spacing: 2) {
                    Image(systemName: "waveform.circle.fill")
                        .font(.system(size: 20))
                        .foregroundColor(Color(hex: "DC143C"))
                    Text("\"dmagh\"")
                        .font(.system(size: 10, weight: .bold, design: .monospaced))
                        .foregroundColor(Color(hex: "FF2D2D"))
                }
                .padding(8)
                .background(Color(hex: "1F1010"))
                .cornerRadius(10)
            }
            
            // Main action buttons
            HStack(spacing: 12) {
                LargeWidgetActionButton(
                    icon: "mic.fill",
                    title: "Commande Vocale",
                    subtitle: "Dites \"dmagh\"",
                    color: Color(hex: "DC143C"),
                    url: "personnelassistant://voice"
                )
                
                LargeWidgetActionButton(
                    icon: "message.fill",
                    title: "Chat",
                    subtitle: "Conversation",
                    color: Color(hex: "8B0000"),
                    url: "personnelassistant://chat"
                )
            }
            
            // Infrastructure status
            VStack(spacing: 8) {
                HStack {
                    Text("Infrastructure")
                        .font(.system(size: 12, weight: .semibold))
                        .foregroundColor(.white)
                    Spacer()
                    Text("\(entry.servicesCount) services")
                        .font(.system(size: 11))
                        .foregroundColor(Color(hex: "6B6B6B"))
                }
                
                HStack(spacing: 8) {
                    InfraStatusPill(name: "Docker", status: .healthy)
                    InfraStatusPill(name: "VPN", status: .healthy)
                    InfraStatusPill(name: "SSO", status: .healthy)
                    InfraStatusPill(name: "API", status: .healthy)
                }
            }
            .padding(12)
            .background(Color(hex: "1F1010"))
            .cornerRadius(12)
            
            // Last message preview
            HStack {
                Image(systemName: "quote.bubble.fill")
                    .font(.system(size: 12))
                    .foregroundColor(Color(hex: "DC143C"))
                
                Text(entry.lastMessage)
                    .font(.system(size: 11))
                    .foregroundColor(Color(hex: "B0B0B0"))
                    .lineLimit(1)
                
                Spacer()
            }
            .padding(.horizontal, 8)
        }
        .padding(4)
    }
}

struct LargeWidgetActionButton: View {
    let icon: String
    let title: String
    let subtitle: String
    let color: Color
    let url: String
    
    var body: some View {
        Link(destination: URL(string: url)!) {
            HStack(spacing: 12) {
                ZStack {
                    Circle()
                        .fill(color.opacity(0.2))
                        .frame(width: 44, height: 44)
                    
                    Image(systemName: icon)
                        .font(.system(size: 20))
                        .foregroundColor(color)
                }
                
                VStack(alignment: .leading, spacing: 2) {
                    Text(title)
                        .font(.system(size: 13, weight: .semibold))
                        .foregroundColor(.white)
                    Text(subtitle)
                        .font(.system(size: 10))
                        .foregroundColor(Color(hex: "6B6B6B"))
                }
                
                Spacer()
                
                Image(systemName: "chevron.right")
                    .font(.system(size: 12))
                    .foregroundColor(Color(hex: "6B6B6B"))
            }
            .padding(12)
            .background(Color(hex: "1F1010"))
            .cornerRadius(12)
        }
    }
}

struct InfraStatusPill: View {
    let name: String
    let status: ServiceStatus
    
    enum ServiceStatus {
        case healthy, degraded, down
        
        var color: Color {
            switch self {
            case .healthy: return Color(hex: "228B22")
            case .degraded: return Color(hex: "FF8C00")
            case .down: return Color(hex: "FF4444")
            }
        }
    }
    
    var body: some View {
        HStack(spacing: 4) {
            Circle()
                .fill(status.color)
                .frame(width: 6, height: 6)
            Text(name)
                .font(.system(size: 10, weight: .medium))
                .foregroundColor(Color(hex: "B0B0B0"))
        }
        .padding(.horizontal, 8)
        .padding(.vertical, 4)
        .background(Color(hex: "2A1A1A"))
        .cornerRadius(6)
    }
}

// MARK: - Accessory Views (Lock Screen)

struct CircularAccessoryView: View {
    let entry: AssistantWidgetEntry
    
    var body: some View {
        ZStack {
            AccessoryWidgetBackground()
            
            VStack(spacing: 2) {
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 20))
                
                Text("dmagh")
                    .font(.system(size: 8, weight: .bold))
            }
        }
        .widgetURL(URL(string: "personnelassistant://voice"))
    }
}

struct RectangularAccessoryView: View {
    let entry: AssistantWidgetEntry
    
    var body: some View {
        HStack(spacing: 8) {
            Image(systemName: "brain.head.profile")
                .font(.system(size: 24))
            
            VStack(alignment: .leading, spacing: 2) {
                Text("Personnel Assistant")
                    .font(.system(size: 12, weight: .bold))
                
                HStack(spacing: 4) {
                    Circle()
                        .fill(entry.status.color)
                        .frame(width: 6, height: 6)
                    Text(entry.status.text)
                        .font(.system(size: 10))
                }
            }
        }
        .widgetURL(URL(string: "personnelassistant://voice"))
    }
}

// MARK: - Color Extension (Widget Target)

fileprivate extension Color {
    init(hex: String) {
        let hex = hex.trimmingCharacters(in: CharacterSet.alphanumerics.inverted)
        var int: UInt64 = 0
        Scanner(string: hex).scanHexInt64(&int)
        let a, r, g, b: UInt64
        switch hex.count {
        case 3:
            (a, r, g, b) = (255, (int >> 8) * 17, (int >> 4 & 0xF) * 17, (int & 0xF) * 17)
        case 6:
            (a, r, g, b) = (255, int >> 16, int >> 8 & 0xFF, int & 0xFF)
        case 8:
            (a, r, g, b) = (int >> 24, int >> 16 & 0xFF, int >> 8 & 0xFF, int & 0xFF)
        default:
            (a, r, g, b) = (255, 0, 0, 0)
        }
        self.init(
            .sRGB,
            red: Double(r) / 255,
            green: Double(g) / 255,
            blue: Double(b) / 255,
            opacity: Double(a) / 255
        )
    }
}

// MARK: - Widget Bundle

@main
struct PersonnelAssistantWidgetBundle: WidgetBundle {
    var body: some Widget {
        PersonnelAssistantWidget()
    }
}

// MARK: - Previews

#Preview("Small", as: .systemSmall) {
    PersonnelAssistantWidget()
} timeline: {
    AssistantWidgetEntry(date: .now, status: .online, lastMessage: "Prêt!", servicesCount: 24, configuration: AssistantWidgetIntent())
}

#Preview("Medium", as: .systemMedium) {
    PersonnelAssistantWidget()
} timeline: {
    AssistantWidgetEntry(date: .now, status: .online, lastMessage: "Prêt!", servicesCount: 24, configuration: AssistantWidgetIntent())
}

#Preview("Large", as: .systemLarge) {
    PersonnelAssistantWidget()
} timeline: {
    AssistantWidgetEntry(date: .now, status: .online, lastMessage: "Comment puis-je vous aider?", servicesCount: 24, configuration: AssistantWidgetIntent())
}
