//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Personnel Assistant - Apple Watch App                       ║
//  ║                    Dark Blood Theme - WatchOS                                  ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import WatchKit
import WatchConnectivity

@main
struct PersonnelAssistantWatchApp: App {
    @StateObject private var sessionManager = WatchSessionManager.shared
    @StateObject private var voiceManager = WatchVoiceManager()
    
    var body: some Scene {
        WindowGroup {
            WatchHomeView()
                .environmentObject(sessionManager)
                .environmentObject(voiceManager)
        }
    }
}

// MARK: - Watch Session Manager

class WatchSessionManager: NSObject, ObservableObject, WCSessionDelegate {
    static let shared = WatchSessionManager()
    
    @Published var isReachable = false
    @Published var lastMessage: String = "Prêt à vous aider"
    @Published var servicesStatus: [String: Bool] = [
        "docker": true,
        "vpn": true,
        "sso": true,
        "api": true
    ]
    
    private var session: WCSession?
    
    override init() {
        super.init()
        if WCSession.isSupported() {
            session = WCSession.default
            session?.delegate = self
            session?.activate()
        }
    }
    
    func sendMessage(_ message: String) {
        guard let session = session, session.isReachable else {
            lastMessage = "iPhone non connecté"
            return
        }
        
        session.sendMessage(
            ["action": "chat", "message": message],
            replyHandler: { response in
                DispatchQueue.main.async {
                    if let reply = response["reply"] as? String {
                        self.lastMessage = reply
                    }
                }
            },
            errorHandler: { error in
                DispatchQueue.main.async {
                    self.lastMessage = "Erreur: \(error.localizedDescription)"
                }
            }
        )
    }
    
    func startVoiceOnPhone() {
        guard let session = session, session.isReachable else { return }
        session.sendMessage(["action": "startVoice"], replyHandler: nil, errorHandler: nil)
    }
    
    func requestInfraStatus() {
        guard let session = session, session.isReachable else { return }
        session.sendMessage(
            ["action": "infraStatus"],
            replyHandler: { response in
                DispatchQueue.main.async {
                    if let status = response["status"] as? [String: Bool] {
                        self.servicesStatus = status
                    }
                }
            },
            errorHandler: nil
        )
    }
    
    // WCSessionDelegate
    func session(_ session: WCSession, activationDidCompleteWith activationState: WCSessionActivationState, error: Error?) {
        DispatchQueue.main.async {
            self.isReachable = session.isReachable
        }
    }
    
    func sessionReachabilityDidChange(_ session: WCSession) {
        DispatchQueue.main.async {
            self.isReachable = session.isReachable
        }
    }
    
    func session(_ session: WCSession, didReceiveMessage message: [String : Any]) {
        DispatchQueue.main.async {
            if let response = message["response"] as? String {
                self.lastMessage = response
            }
            if let status = message["infraStatus"] as? [String: Bool] {
                self.servicesStatus = status
            }
        }
    }
}

// MARK: - Watch Voice Manager

class WatchVoiceManager: ObservableObject {
    @Published var isRecording = false
    @Published var transcribedText = ""
    
    func startRecording() {
        isRecording = true
        // Use Watch's dictation
    }
    
    func stopRecording() {
        isRecording = false
    }
}

// MARK: - Watch Home View

struct WatchHomeView: View {
    @EnvironmentObject var sessionManager: WatchSessionManager
    @EnvironmentObject var voiceManager: WatchVoiceManager
    @State private var showVoiceInput = false
    
    var body: some View {
        NavigationStack {
            ScrollView {
                VStack(spacing: 12) {
                    // Header with logo
                    WatchHeaderView()
                    
                    // Quick Actions
                    VStack(spacing: 8) {
                        // Voice Button - Main Action
                        Button(action: { showVoiceInput = true }) {
                            WatchActionCard(
                                icon: "mic.fill",
                                title: "dmagh",
                                subtitle: "Commande vocale",
                                color: WatchColors.accent
                            )
                        }
                        .buttonStyle(.plain)
                        
                        // Status Row
                        HStack(spacing: 8) {
                            NavigationLink(destination: WatchInfraView()) {
                                WatchMiniCard(
                                    icon: "server.rack",
                                    value: "\(sessionManager.servicesStatus.filter { $0.value }.count)",
                                    label: "Services",
                                    color: WatchColors.success
                                )
                            }
                            .buttonStyle(.plain)
                            
                            NavigationLink(destination: WatchChatView()) {
                                WatchMiniCard(
                                    icon: "message.fill",
                                    value: "Chat",
                                    label: "Ouvrir",
                                    color: WatchColors.primary
                                )
                            }
                            .buttonStyle(.plain)
                        }
                    }
                    
                    // Last Message Preview
                    WatchMessagePreview(message: sessionManager.lastMessage)
                    
                    // Connection Status
                    HStack(spacing: 4) {
                        Circle()
                            .fill(sessionManager.isReachable ? WatchColors.success : WatchColors.error)
                            .frame(width: 6, height: 6)
                        Text(sessionManager.isReachable ? "Connecté" : "Déconnecté")
                            .font(.system(size: 10))
                            .foregroundColor(WatchColors.textMuted)
                    }
                }
                .padding(.horizontal, 4)
            }
            .background(WatchColors.background)
            .sheet(isPresented: $showVoiceInput) {
                WatchVoiceInputView()
            }
        }
    }
}

// MARK: - Watch Header

struct WatchHeaderView: View {
    var body: some View {
        HStack(spacing: 8) {
            ZStack {
                Circle()
                    .fill(
                        RadialGradient(
                            colors: [WatchColors.accent.opacity(0.4), Color.clear],
                            center: .center,
                            startRadius: 0,
                            endRadius: 20
                        )
                    )
                    .frame(width: 36, height: 36)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 18))
                    .foregroundColor(WatchColors.accent)
            }
            
            VStack(alignment: .leading, spacing: 0) {
                Text("Personnel")
                    .font(.system(size: 12, weight: .bold))
                    .foregroundColor(.white)
                Text("Assistant")
                    .font(.system(size: 10))
                    .foregroundColor(WatchColors.textSecondary)
            }
            
            Spacer()
        }
        .padding(.vertical, 8)
    }
}

// MARK: - Watch Action Card

struct WatchActionCard: View {
    let icon: String
    let title: String
    let subtitle: String
    let color: Color
    
    var body: some View {
        HStack(spacing: 12) {
            ZStack {
                Circle()
                    .fill(color.opacity(0.2))
                    .frame(width: 40, height: 40)
                
                Image(systemName: icon)
                    .font(.system(size: 18))
                    .foregroundColor(color)
            }
            
            VStack(alignment: .leading, spacing: 2) {
                Text(title)
                    .font(.system(size: 14, weight: .bold))
                    .foregroundColor(.white)
                Text(subtitle)
                    .font(.system(size: 10))
                    .foregroundColor(WatchColors.textMuted)
            }
            
            Spacer()
            
            Image(systemName: "chevron.right")
                .font(.system(size: 10))
                .foregroundColor(WatchColors.textMuted)
        }
        .padding(12)
        .background(WatchColors.surface)
        .cornerRadius(12)
    }
}

// MARK: - Watch Mini Card

struct WatchMiniCard: View {
    let icon: String
    let value: String
    let label: String
    let color: Color
    
    var body: some View {
        VStack(spacing: 4) {
            Image(systemName: icon)
                .font(.system(size: 16))
                .foregroundColor(color)
            
            Text(value)
                .font(.system(size: 12, weight: .bold))
                .foregroundColor(.white)
            
            Text(label)
                .font(.system(size: 8))
                .foregroundColor(WatchColors.textMuted)
        }
        .frame(maxWidth: .infinity)
        .padding(.vertical, 10)
        .background(WatchColors.surface)
        .cornerRadius(10)
    }
}

// MARK: - Watch Message Preview

struct WatchMessagePreview: View {
    let message: String
    
    var body: some View {
        HStack(spacing: 8) {
            Image(systemName: "quote.bubble.fill")
                .font(.system(size: 10))
                .foregroundColor(WatchColors.accent)
            
            Text(message)
                .font(.system(size: 10))
                .foregroundColor(WatchColors.textSecondary)
                .lineLimit(2)
        }
        .padding(10)
        .frame(maxWidth: .infinity, alignment: .leading)
        .background(WatchColors.surface)
        .cornerRadius(10)
    }
}

// MARK: - Watch Voice Input View

struct WatchVoiceInputView: View {
    @EnvironmentObject var sessionManager: WatchSessionManager
    @Environment(\.dismiss) var dismiss
    @State private var inputText = ""
    @State private var isListening = false
    @State private var pulseScale: CGFloat = 1.0
    
    var body: some View {
        VStack(spacing: 16) {
            // Animated mic
            ZStack {
                // Pulse rings
                ForEach(0..<3, id: \.self) { i in
                    Circle()
                        .stroke(WatchColors.accent.opacity(0.3 - Double(i) * 0.1), lineWidth: 2)
                        .frame(width: 60 + CGFloat(i) * 20, height: 60 + CGFloat(i) * 20)
                        .scaleEffect(isListening ? pulseScale : 1)
                        .animation(
                            .easeInOut(duration: 1)
                                .repeatForever(autoreverses: true)
                                .delay(Double(i) * 0.2),
                            value: isListening
                        )
                }
                
                // Mic button
                Circle()
                    .fill(
                        RadialGradient(
                            colors: [WatchColors.accent, WatchColors.primary],
                            center: .center,
                            startRadius: 0,
                            endRadius: 30
                        )
                    )
                    .frame(width: 60, height: 60)
                
                Image(systemName: isListening ? "waveform" : "mic.fill")
                    .font(.system(size: 24))
                    .foregroundColor(.white)
            }
            .onTapGesture {
                toggleListening()
            }
            
            Text(isListening ? "Parlez maintenant..." : "Appuyez pour parler")
                .font(.system(size: 12))
                .foregroundColor(WatchColors.textSecondary)
            
            // Quick phrases
            VStack(spacing: 6) {
                QuickPhraseButton(phrase: "État infrastructure") {
                    sessionManager.sendMessage("État de l'infrastructure")
                    dismiss()
                }
                
                QuickPhraseButton(phrase: "Services Docker") {
                    sessionManager.sendMessage("Liste les services Docker")
                    dismiss()
                }
            }
        }
        .padding()
        .background(WatchColors.background)
        .onAppear {
            isListening = true
            pulseScale = 1.2
        }
    }
    
    private func toggleListening() {
        isListening.toggle()
        if isListening {
            pulseScale = 1.2
            // Start listening via iPhone
            sessionManager.startVoiceOnPhone()
        } else {
            pulseScale = 1.0
        }
    }
}

struct QuickPhraseButton: View {
    let phrase: String
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            Text(phrase)
                .font(.system(size: 11))
                .foregroundColor(.white)
                .frame(maxWidth: .infinity)
                .padding(.vertical, 8)
                .background(WatchColors.surface)
                .cornerRadius(8)
        }
        .buttonStyle(.plain)
    }
}

// MARK: - Watch Infrastructure View

struct WatchInfraView: View {
    @EnvironmentObject var sessionManager: WatchSessionManager
    
    var body: some View {
        ScrollView {
            VStack(spacing: 10) {
                ForEach(Array(sessionManager.servicesStatus.keys.sorted()), id: \.self) { service in
                    HStack {
                        Circle()
                            .fill(sessionManager.servicesStatus[service] == true ? WatchColors.success : WatchColors.error)
                            .frame(width: 8, height: 8)
                        
                        Text(service.capitalized)
                            .font(.system(size: 14))
                            .foregroundColor(.white)
                        
                        Spacer()
                        
                        Text(sessionManager.servicesStatus[service] == true ? "OK" : "Down")
                            .font(.system(size: 12))
                            .foregroundColor(sessionManager.servicesStatus[service] == true ? WatchColors.success : WatchColors.error)
                    }
                    .padding(10)
                    .background(WatchColors.surface)
                    .cornerRadius(8)
                }
            }
            .padding(.horizontal)
        }
        .navigationTitle("Infrastructure")
        .background(WatchColors.background)
        .onAppear {
            sessionManager.requestInfraStatus()
        }
    }
}

// MARK: - Watch Chat View

struct WatchChatView: View {
    @EnvironmentObject var sessionManager: WatchSessionManager
    @State private var showDictation = false
    
    var body: some View {
        VStack(spacing: 12) {
            // Last message
            ScrollView {
                Text(sessionManager.lastMessage)
                    .font(.system(size: 12))
                    .foregroundColor(.white)
                    .frame(maxWidth: .infinity, alignment: .leading)
                    .padding()
                    .background(WatchColors.surface)
                    .cornerRadius(10)
            }
            
            // Send button
            Button(action: { showDictation = true }) {
                HStack {
                    Image(systemName: "mic.fill")
                    Text("Envoyer")
                }
                .font(.system(size: 14, weight: .medium))
                .foregroundColor(.white)
                .frame(maxWidth: .infinity)
                .padding(.vertical, 12)
                .background(WatchColors.accent)
                .cornerRadius(10)
            }
            .buttonStyle(.plain)
        }
        .padding(.horizontal)
        .navigationTitle("Chat")
        .background(WatchColors.background)
    }
}

// MARK: - Watch Colors

struct WatchColors {
    static let primary = Color(hex: "8B0000")
    static let accent = Color(hex: "DC143C")
    static let background = Color(hex: "0D0D0D")
    static let surface = Color(hex: "1F1010")
    static let success = Color(hex: "228B22")
    static let error = Color(hex: "FF4444")
    static let textSecondary = Color(hex: "B0B0B0")
    static let textMuted = Color(hex: "6B6B6B")
}

// MARK: - Color Extension (Watch Target)

fileprivate extension Color {
    init(hex: String) {
        let hex = hex.trimmingCharacters(in: CharacterSet.alphanumerics.inverted)
        var int: UInt64 = 0
        Scanner(string: hex).scanHexInt64(&int)
        let a, r, g, b: UInt64
        switch hex.count {
        case 3:
            (a, r, g, b) = (255, (int >> 8) * 17, (int >> 4 & 0xF) * 17, (int & 0xF) * 17)
        case 6:
            (a, r, g, b) = (255, int >> 16, int >> 8 & 0xFF, int & 0xFF)
        default:
            (a, r, g, b) = (255, 0, 0, 0)
        }
        self.init(
            .sRGB,
            red: Double(r) / 255,
            green: Double(g) / 255,
            blue: Double(b) / 255,
            opacity: Double(a) / 255
        )
    }
}
