//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Personnel Assistant - Watch Complications                   ║
//  ║                    Dark Blood Theme - ClockKit                                ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import WidgetKit

// MARK: - Watch Complications Widget

struct PersonnelAssistantComplications: Widget {
    let kind: String = "PersonnelAssistantComplications"
    
    var body: some WidgetConfiguration {
        StaticConfiguration(
            kind: kind,
            provider: ComplicationTimelineProvider()
        ) { entry in
            ComplicationEntryView(entry: entry)
        }
        .configurationDisplayName("Personnel Assistant")
        .description("Accès rapide à votre assistant")
        .supportedFamilies([
            .accessoryCircular,
            .accessoryRectangular,
            .accessoryInline,
            .accessoryCorner
        ])
    }
}

// MARK: - Timeline Provider

struct ComplicationTimelineProvider: TimelineProvider {
    typealias Entry = ComplicationEntry
    
    func placeholder(in context: Context) -> ComplicationEntry {
        ComplicationEntry(date: Date(), status: .online, servicesCount: 24)
    }
    
    func getSnapshot(in context: Context, completion: @escaping (ComplicationEntry) -> Void) {
        let entry = ComplicationEntry(date: Date(), status: .online, servicesCount: 24)
        completion(entry)
    }
    
    func getTimeline(in context: Context, completion: @escaping (Timeline<ComplicationEntry>) -> Void) {
        let entry = ComplicationEntry(date: Date(), status: .online, servicesCount: 24)
        let nextUpdate = Calendar.current.date(byAdding: .minute, value: 15, to: Date())!
        let timeline = Timeline(entries: [entry], policy: .after(nextUpdate))
        completion(timeline)
    }
}

// MARK: - Complication Entry

struct ComplicationEntry: TimelineEntry {
    let date: Date
    let status: AssistantStatus
    let servicesCount: Int
    
    enum AssistantStatus {
        case online, offline
        
        var color: Color {
            switch self {
            case .online: return Color(hex: "228B22")
            case .offline: return Color(hex: "FF4444")
            }
        }
    }
}

// MARK: - Complication Entry View

struct ComplicationEntryView: View {
    @Environment(\.widgetFamily) var family
    var entry: ComplicationEntry
    
    var body: some View {
        switch family {
        case .accessoryCircular:
            CircularComplication(entry: entry)
        case .accessoryRectangular:
            RectangularComplication(entry: entry)
        case .accessoryInline:
            InlineComplication(entry: entry)
        case .accessoryCorner:
            CornerComplication(entry: entry)
        default:
            CircularComplication(entry: entry)
        }
    }
}

// MARK: - Circular Complication

struct CircularComplication: View {
    let entry: ComplicationEntry
    
    var body: some View {
        ZStack {
            AccessoryWidgetBackground()
            
            VStack(spacing: 2) {
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 18))
                    .foregroundColor(Color(hex: "DC143C"))
                
                Text("dmagh")
                    .font(.system(size: 8, weight: .bold, design: .monospaced))
                    .foregroundColor(.white)
            }
        }
        .widgetURL(URL(string: "personnelassistant://voice"))
    }
}

// MARK: - Rectangular Complication

struct RectangularComplication: View {
    let entry: ComplicationEntry
    
    var body: some View {
        HStack(spacing: 8) {
            ZStack {
                Circle()
                    .fill(Color(hex: "DC143C").opacity(0.3))
                    .frame(width: 30, height: 30)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 16))
                    .foregroundColor(Color(hex: "DC143C"))
            }
            
            VStack(alignment: .leading, spacing: 2) {
                Text("Personnel Assistant")
                    .font(.system(size: 12, weight: .semibold))
                    .foregroundColor(.white)
                
                HStack(spacing: 4) {
                    Circle()
                        .fill(entry.status.color)
                        .frame(width: 6, height: 6)
                    
                    Text("\(entry.servicesCount) services")
                        .font(.system(size: 10))
                        .foregroundColor(.gray)
                }
            }
            
            Spacer()
        }
        .widgetURL(URL(string: "personnelassistant://home"))
    }
}

// MARK: - Inline Complication

struct InlineComplication: View {
    let entry: ComplicationEntry
    
    var body: some View {
        HStack(spacing: 4) {
            Image(systemName: "brain.head.profile")
            Text("dmagh")
            Text("•")
            Text(entry.status == .online ? "En ligne" : "Hors ligne")
        }
        .widgetURL(URL(string: "personnelassistant://voice"))
    }
}

// MARK: - Corner Complication

struct CornerComplication: View {
    let entry: ComplicationEntry
    
    var body: some View {
        ZStack {
            Image(systemName: "brain.head.profile")
                .font(.system(size: 20))
                .foregroundColor(Color(hex: "DC143C"))
        }
        .widgetLabel {
            Text("dmagh")
                .font(.system(size: 10, weight: .bold))
        }
        .widgetURL(URL(string: "personnelassistant://voice"))
    }
}

// MARK: - Color Extension (Watch Complications Target)

fileprivate extension Color {
    init(hex: String) {
        let hex = hex.trimmingCharacters(in: CharacterSet.alphanumerics.inverted)
        var int: UInt64 = 0
        Scanner(string: hex).scanHexInt64(&int)
        let a, r, g, b: UInt64
        switch hex.count {
        case 3:
            (a, r, g, b) = (255, (int >> 8) * 17, (int >> 4 & 0xF) * 17, (int & 0xF) * 17)
        case 6:
            (a, r, g, b) = (255, int >> 16, int >> 8 & 0xFF, int & 0xFF)
        default:
            (a, r, g, b) = (255, 0, 0, 0)
        }
        self.init(
            .sRGB,
            red: Double(r) / 255,
            green: Double(g) / 255,
            blue: Double(b) / 255,
            opacity: Double(a) / 255
        )
    }
}

// MARK: - Widget Bundle for Watch

struct PersonnelAssistantWatchWidgetBundle: WidgetBundle {
    var body: some Widget {
        PersonnelAssistantComplications()
    }
}
