//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Content Viewers - Multi-format Support                      ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import WebKit

// MARK: - Markdown View

struct MarkdownView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            ForEach(parseMarkdown(content), id: \.id) { element in
                element.view
            }
        }
    }
    
    private func parseMarkdown(_ content: String) -> [MarkdownElement] {
        var elements: [MarkdownElement] = []
        let lines = content.components(separatedBy: "\n")
        var codeBlock = ""
        var inCodeBlock = false
        var codeLanguage = ""
        
        for line in lines {
            if line.hasPrefix("```") {
                if inCodeBlock {
                    elements.append(MarkdownElement(type: .code, content: codeBlock, language: codeLanguage))
                    codeBlock = ""
                    inCodeBlock = false
                    codeLanguage = ""
                } else {
                    inCodeBlock = true
                    codeLanguage = String(line.dropFirst(3))
                }
            } else if inCodeBlock {
                if !codeBlock.isEmpty { codeBlock += "\n" }
                codeBlock += line
            } else if line.hasPrefix("### ") {
                elements.append(MarkdownElement(type: .h3, content: String(line.dropFirst(4))))
            } else if line.hasPrefix("## ") {
                elements.append(MarkdownElement(type: .h2, content: String(line.dropFirst(3))))
            } else if line.hasPrefix("# ") {
                elements.append(MarkdownElement(type: .h1, content: String(line.dropFirst(2))))
            } else if line.hasPrefix("- ") || line.hasPrefix("• ") {
                elements.append(MarkdownElement(type: .bullet, content: String(line.dropFirst(2))))
            } else if line.hasPrefix("> ") {
                elements.append(MarkdownElement(type: .quote, content: String(line.dropFirst(2))))
            } else if !line.isEmpty {
                elements.append(MarkdownElement(type: .paragraph, content: line))
            }
        }
        
        return elements
    }
}

struct MarkdownElement: Identifiable {
    let id = UUID()
    let type: MarkdownType
    let content: String
    var language: String = ""
    
    @ViewBuilder
    var view: some View {
        let theme = ThemeManager.shared
        
        switch type {
        case .h1:
            Text(parseInlineMarkdown(content))
                .font(.system(size: 24, weight: .bold))
                .foregroundColor(theme.textPrimary)
        case .h2:
            Text(parseInlineMarkdown(content))
                .font(.system(size: 20, weight: .semibold))
                .foregroundColor(theme.textPrimary)
        case .h3:
            Text(parseInlineMarkdown(content))
                .font(.system(size: 17, weight: .semibold))
                .foregroundColor(theme.textPrimary)
        case .paragraph:
            Text(parseInlineMarkdown(content))
                .font(theme.fontBody)
                .foregroundColor(theme.textPrimary)
        case .bullet:
            HStack(alignment: .top, spacing: 8) {
                Text("•")
                    .foregroundColor(theme.accentColor)
                Text(parseInlineMarkdown(content))
                    .font(theme.fontBody)
                    .foregroundColor(theme.textPrimary)
            }
        case .quote:
            HStack(spacing: 12) {
                Rectangle()
                    .fill(theme.accentColor)
                    .frame(width: 3)
                Text(parseInlineMarkdown(content))
                    .font(.system(size: 15, weight: .regular, design: .serif))
                    .foregroundColor(theme.textSecondary)
                    .italic()
            }
            .padding(.vertical, 4)
        case .code:
            CodeBlockView(code: content, language: language)
        }
    }
    
    private func parseInlineMarkdown(_ text: String) -> AttributedString {
        var result = AttributedString(text)
        
        // Bold: **text**
        let boldPattern = /\*\*(.+?)\*\*/
        for match in text.matches(of: boldPattern) {
            if let range = result.range(of: String(match.0)) {
                result[range].font = .boldSystemFont(ofSize: 16)
            }
        }
        
        // Code: `text`
        let codePattern = /`(.+?)`/
        for match in text.matches(of: codePattern) {
            if let range = result.range(of: String(match.0)) {
                result[range].font = .monospacedSystemFont(ofSize: 14, weight: .regular)
                result[range].backgroundColor = ThemeManager.shared.surfaceElevated
            }
        }
        
        return result
    }
}

enum MarkdownType {
    case h1, h2, h3, paragraph, bullet, quote, code
}

// MARK: - Code Block View

struct CodeBlockView: View {
    @EnvironmentObject var theme: ThemeManager
    let code: String
    let language: String
    @State private var isCopied = false
    
    var body: some View {
        VStack(alignment: .leading, spacing: 0) {
            // Header
            HStack {
                Text(language.isEmpty ? "code" : language)
                    .font(.system(size: 12, weight: .medium, design: .monospaced))
                    .foregroundColor(theme.textMuted)
                
                Spacer()
                
                Button(action: copyCode) {
                    HStack(spacing: 4) {
                        Image(systemName: isCopied ? "checkmark" : "doc.on.doc")
                            .font(.system(size: 12))
                        Text(isCopied ? "Copié!" : "Copier")
                            .font(.system(size: 12))
                    }
                    .foregroundColor(isCopied ? theme.successColor : theme.textMuted)
                }
            }
            .padding(.horizontal, 12)
            .padding(.vertical, 8)
            .background(theme.surfaceElevated)
            
            // Code
            ScrollView(.horizontal, showsIndicators: false) {
                Text(code)
                    .font(theme.fontCode)
                    .foregroundColor(theme.textPrimary)
                    .padding(12)
            }
            .background(theme.surfaceColor)
        }
        .cornerRadius(theme.radiusSmall)
        .overlay(
            RoundedRectangle(cornerRadius: theme.radiusSmall)
                .stroke(theme.surfaceElevated, lineWidth: 1)
        )
    }
    
    private func copyCode() {
        UIPasteboard.general.string = code
        withAnimation {
            isCopied = true
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 2) {
            withAnimation {
                isCopied = false
            }
        }
    }
}

// MARK: - JSON View

struct JSONView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    @State private var isExpanded = true
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            // Header
            HStack {
                Image(systemName: "curlybraces")
                    .foregroundColor(theme.accentColor)
                Text("JSON")
                    .font(.system(size: 14, weight: .medium))
                    .foregroundColor(theme.textSecondary)
                
                Spacer()
                
                Button(action: { isExpanded.toggle() }) {
                    Image(systemName: isExpanded ? "chevron.up" : "chevron.down")
                        .foregroundColor(theme.textMuted)
                }
            }
            
            if isExpanded {
                JSONTreeView(content: content)
            }
        }
        .padding(12)
        .background(theme.surfaceColor)
        .cornerRadius(theme.radiusMedium)
    }
}

struct JSONTreeView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        ScrollView(.horizontal, showsIndicators: false) {
            Text(formatJSON(content))
                .font(theme.fontCode)
                .foregroundColor(theme.textPrimary)
        }
    }
    
    private func formatJSON(_ json: String) -> String {
        guard let data = json.data(using: .utf8),
              let object = try? JSONSerialization.jsonObject(with: data),
              let prettyData = try? JSONSerialization.data(withJSONObject: object, options: [.prettyPrinted, .sortedKeys]),
              let pretty = String(data: prettyData, encoding: .utf8)
        else {
            return json
        }
        return pretty
    }
}

// MARK: - Code View (for inline code)

struct CodeView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            // Extract code blocks
            ForEach(extractCodeBlocks(from: content), id: \.id) { block in
                if block.isCode {
                    CodeBlockView(code: block.content, language: block.language)
                } else {
                    Text(block.content)
                        .font(theme.fontBody)
                        .foregroundColor(theme.textPrimary)
                }
            }
        }
    }
    
    private func extractCodeBlocks(from content: String) -> [CodeBlock] {
        var blocks: [CodeBlock] = []
        var currentText = ""
        var inCode = false
        var currentCode = ""
        var language = ""
        
        for line in content.components(separatedBy: "\n") {
            if line.hasPrefix("```") {
                if inCode {
                    blocks.append(CodeBlock(content: currentCode, isCode: true, language: language))
                    currentCode = ""
                    inCode = false
                    language = ""
                } else {
                    if !currentText.isEmpty {
                        blocks.append(CodeBlock(content: currentText, isCode: false, language: ""))
                        currentText = ""
                    }
                    inCode = true
                    language = String(line.dropFirst(3))
                }
            } else if inCode {
                if !currentCode.isEmpty { currentCode += "\n" }
                currentCode += line
            } else {
                if !currentText.isEmpty { currentText += "\n" }
                currentText += line
            }
        }
        
        if !currentText.isEmpty {
            blocks.append(CodeBlock(content: currentText, isCode: false, language: ""))
        }
        
        return blocks
    }
}

struct CodeBlock: Identifiable {
    let id = UUID()
    let content: String
    let isCode: Bool
    let language: String
}

// MARK: - PDF Viewer

struct PDFViewer: UIViewRepresentable {
    let data: Data
    
    func makeUIView(context: Context) -> WKWebView {
        let webView = WKWebView()
        webView.load(data, mimeType: "application/pdf", characterEncodingName: "utf-8", baseURL: URL(fileURLWithPath: "/"))
        return webView
    }
    
    func updateUIView(_ uiView: WKWebView, context: Context) {}
}

// MARK: - Document Preview

struct DocumentPreview: View {
    @EnvironmentObject var theme: ThemeManager
    let attachment: Attachment
    
    var body: some View {
        NavigationStack {
            VStack {
                switch attachment.type {
                case .pdf:
                    PDFViewer(data: attachment.data)
                case .image:
                    if let uiImage = UIImage(data: attachment.data) {
                        Image(uiImage: uiImage)
                            .resizable()
                            .aspectRatio(contentMode: .fit)
                    }
                default:
                    Text("Preview not available")
                        .foregroundColor(theme.textMuted)
                }
            }
            .navigationTitle(attachment.name)
            .navigationBarTitleDisplayMode(.inline)
        }
    }
}

// MARK: - Sheet Viewer (Spreadsheet)

struct SheetViewer: View {
    @EnvironmentObject var theme: ThemeManager
    let data: [[String]]
    
    var body: some View {
        ScrollView([.horizontal, .vertical]) {
            VStack(alignment: .leading, spacing: 0) {
                ForEach(data.indices, id: \.self) { rowIndex in
                    HStack(spacing: 0) {
                        ForEach(data[rowIndex].indices, id: \.self) { colIndex in
                            Text(data[rowIndex][colIndex])
                                .font(theme.fontBody)
                                .foregroundColor(rowIndex == 0 ? theme.textPrimary : theme.textSecondary)
                                .fontWeight(rowIndex == 0 ? .semibold : .regular)
                                .padding(8)
                                .frame(minWidth: 100)
                                .background(rowIndex == 0 ? theme.surfaceElevated : theme.surfaceColor)
                                .border(theme.surfaceElevated, width: 0.5)
                        }
                    }
                }
            }
        }
        .cornerRadius(theme.radiusMedium)
    }
}
