//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Voice Overlay - Full Screen Recording Interface             ║
//  ║                              Dark Blood Theme                                  ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

struct VoiceOverlay: View {
    @EnvironmentObject var theme: ThemeManager
    @ObservedObject var voiceManager: VoiceManager
    @Binding var isPresented: Bool
    let onComplete: (String) -> Void
    
    @State private var pulseScale: CGFloat = 1.0
    @State private var waveAmplitudes: [CGFloat] = Array(repeating: 0.3, count: 40)
    @State private var bloodDrops: [BloodDroplet] = []
    @State private var showTranscript = true
    @State private var isPaused = false
    
    var body: some View {
        ZStack {
            // ═══════════════════════════════════════════════════════════════════════
            // Background with blood effect
            // ═══════════════════════════════════════════════════════════════════════
            
            theme.backgroundColor
                .ignoresSafeArea()
            
            // Animated blood gradient background
            RadialGradient(
                colors: [
                    theme.primaryDark.opacity(0.8),
                    theme.backgroundColor.opacity(0.9),
                    Color.black
                ],
                center: .center,
                startRadius: 50,
                endRadius: UIScreen.main.bounds.height
            )
            .ignoresSafeArea()
            .scaleEffect(pulseScale)
            .animation(
                .easeInOut(duration: 1.5).repeatForever(autoreverses: true),
                value: pulseScale
            )
            
            // Blood drops falling animation
            BloodRainEffect(drops: $bloodDrops, color: theme.accentColor)
                .ignoresSafeArea()
                .opacity(0.4)
            
            // Floating particles
            ParticleSystem(color: theme.accentGlow, count: 30)
                .ignoresSafeArea()
                .opacity(0.3)
            
            // ═══════════════════════════════════════════════════════════════════════
            // Main Content
            // ═══════════════════════════════════════════════════════════════════════
            
            VStack(spacing: 40) {
                Spacer()
                
                // Status indicator
                VStack(spacing: 12) {
                    if voiceManager.isRecording && !isPaused {
                        HStack(spacing: 8) {
                            Circle()
                                .fill(theme.accentGlow)
                                .frame(width: 12, height: 12)
                                .shadow(color: theme.accentGlow, radius: 10)
                                .opacity(pulseScale > 1.05 ? 1 : 0.5)
                            
                            Text("ÉCOUTE EN COURS")
                                .font(.system(size: 14, weight: .bold, design: .monospaced))
                                .foregroundColor(theme.accentGlow)
                                .tracking(3)
                        }
                    } else if isPaused {
                        HStack(spacing: 8) {
                            Image(systemName: "pause.circle.fill")
                                .foregroundColor(theme.warningColor)
                            Text("EN PAUSE")
                                .font(.system(size: 14, weight: .bold, design: .monospaced))
                                .foregroundColor(theme.warningColor)
                                .tracking(3)
                        }
                    }
                }
                .animation(.easeInOut, value: isPaused)
                
                // ═══════════════════════════════════════════════════════════════════
                // Central Waveform Visualization
                // ═══════════════════════════════════════════════════════════════════
                
                ZStack {
                    // Outer glow rings
                    ForEach(0..<3, id: \.self) { i in
                        Circle()
                            .stroke(
                                theme.accentColor.opacity(0.3 - Double(i) * 0.1),
                                lineWidth: 2
                            )
                            .frame(
                                width: 200 + CGFloat(i) * 60 + (voiceManager.isRecording ? CGFloat(voiceManager.audioLevel) * 50 : 0),
                                height: 200 + CGFloat(i) * 60 + (voiceManager.isRecording ? CGFloat(voiceManager.audioLevel) * 50 : 0)
                            )
                            .scaleEffect(pulseScale + CGFloat(i) * 0.05)
                            .animation(.easeOut(duration: 0.1), value: voiceManager.audioLevel)
                    }
                    
                    // Circular waveform
                    CircularWaveform(
                        amplitudes: waveAmplitudes,
                        color: theme.accentGlow,
                        audioLevel: voiceManager.audioLevel
                    )
                    .frame(width: 220, height: 220)
                    
                    // Central microphone button
                    ZStack {
                        Circle()
                            .fill(
                                RadialGradient(
                                    colors: [
                                        theme.accentGlow,
                                        theme.primaryColor,
                                        theme.primaryDark
                                    ],
                                    center: .center,
                                    startRadius: 0,
                                    endRadius: 60
                                )
                            )
                            .frame(width: 120, height: 120)
                            .shadow(color: theme.accentGlow.opacity(0.8), radius: 30)
                        
                        Image(systemName: voiceManager.isRecording ? "waveform" : "mic.fill")
                            .font(.system(size: 44, weight: .medium))
                            .foregroundColor(.white)
                            .symbolEffect(.bounce, value: voiceManager.audioLevel)
                    }
                    .scaleEffect(1 + CGFloat(voiceManager.audioLevel) * 0.1)
                    .animation(.easeOut(duration: 0.05), value: voiceManager.audioLevel)
                }
                
                // ═══════════════════════════════════════════════════════════════════
                // Transcription Display
                // ═══════════════════════════════════════════════════════════════════
                
                VStack(spacing: 16) {
                    if showTranscript {
                        Text(voiceManager.transcribedText.isEmpty ? "Parlez maintenant..." : voiceManager.transcribedText)
                            .font(.system(size: 20, weight: .medium))
                            .foregroundColor(voiceManager.transcribedText.isEmpty ? theme.textMuted : theme.textPrimary)
                            .multilineTextAlignment(.center)
                            .lineLimit(4)
                            .padding(.horizontal, 40)
                            .animation(.easeInOut, value: voiceManager.transcribedText)
                    }
                    
                    // Audio level indicator bar
                    AudioLevelBar(level: voiceManager.audioLevel, color: theme.accentColor)
                        .frame(width: 200, height: 4)
                }
                .frame(height: 120)
                
                Spacer()
                
                // ═══════════════════════════════════════════════════════════════════
                // Control Buttons
                // ═══════════════════════════════════════════════════════════════════
                
                HStack(spacing: 60) {
                    // Cancel button
                    VoiceControlButton(
                        icon: "xmark",
                        label: "Annuler",
                        color: theme.textMuted,
                        size: 56
                    ) {
                        cancelRecording()
                    }
                    
                    // Pause/Resume button
                    VoiceControlButton(
                        icon: isPaused ? "play.fill" : "pause.fill",
                        label: isPaused ? "Reprendre" : "Pause",
                        color: theme.warningColor,
                        size: 56
                    ) {
                        togglePause()
                    }
                    
                    // Stop and send button
                    VoiceControlButton(
                        icon: "checkmark",
                        label: "Envoyer",
                        color: theme.successColor,
                        size: 56,
                        isPrimary: true
                    ) {
                        stopAndSend()
                    }
                }
                .padding(.bottom, 60)
            }
        }
        .onAppear {
            startRecording()
            startAnimations()
        }
        .onDisappear {
            if voiceManager.isRecording {
                voiceManager.stopRecording()
            }
        }
        .onChange(of: voiceManager.audioLevel) { _, newLevel in
            updateWaveAmplitudes(with: newLevel)
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Actions
    // ═══════════════════════════════════════════════════════════════════════════
    
    private func startRecording() {
        voiceManager.startRecording()
    }
    
    private func togglePause() {
        withAnimation(.spring()) {
            isPaused.toggle()
        }
        
        if isPaused {
            voiceManager.stopRecording()
        } else {
            voiceManager.startRecording()
        }
    }
    
    private func cancelRecording() {
        withAnimation(.easeOut(duration: 0.3)) {
            voiceManager.stopRecording()
            isPresented = false
        }
    }
    
    private func stopAndSend() {
        voiceManager.stopRecording()
        
        let text = voiceManager.transcribedText
        
        withAnimation(.easeOut(duration: 0.3)) {
            isPresented = false
        }
        
        if !text.isEmpty {
            onComplete(text)
        }
    }
    
    private func startAnimations() {
        withAnimation(.easeInOut(duration: 1.5).repeatForever(autoreverses: true)) {
            pulseScale = 1.1
        }
        
        // Start blood drops
        Timer.scheduledTimer(withTimeInterval: 0.5, repeats: true) { _ in
            if bloodDrops.count < 20 {
                bloodDrops.append(BloodDroplet())
            }
        }
    }
    
    private func updateWaveAmplitudes(with level: Float) {
        // Shift existing values and add new one
        waveAmplitudes.removeFirst()
        waveAmplitudes.append(CGFloat(level) * 0.8 + 0.2)
    }
}

// MARK: - Circular Waveform

struct CircularWaveform: View {
    let amplitudes: [CGFloat]
    let color: Color
    let audioLevel: Float
    
    var body: some View {
        GeometryReader { geometry in
            let center = CGPoint(x: geometry.size.width / 2, y: geometry.size.height / 2)
            let radius = min(geometry.size.width, geometry.size.height) / 2
            
            ZStack {
                // Base circle
                Circle()
                    .stroke(color.opacity(0.3), lineWidth: 2)
                
                // Waveform
                Path { path in
                    for (index, amplitude) in amplitudes.enumerated() {
                        let angle = (Double(index) / Double(amplitudes.count)) * 2 * .pi - .pi / 2
                        let adjustedRadius = radius * (0.7 + amplitude * 0.3)
                        
                        let x = center.x + cos(angle) * adjustedRadius
                        let y = center.y + sin(angle) * adjustedRadius
                        
                        if index == 0 {
                            path.move(to: CGPoint(x: x, y: y))
                        } else {
                            path.addLine(to: CGPoint(x: x, y: y))
                        }
                    }
                    path.closeSubpath()
                }
                .stroke(
                    LinearGradient(
                        colors: [color, color.opacity(0.5)],
                        startPoint: .top,
                        endPoint: .bottom
                    ),
                    style: StrokeStyle(lineWidth: 3, lineCap: .round, lineJoin: .round)
                )
                .shadow(color: color.opacity(0.5), radius: 5)
            }
        }
    }
}

// MARK: - Audio Level Bar

struct AudioLevelBar: View {
    let level: Float
    let color: Color
    
    var body: some View {
        GeometryReader { geometry in
            ZStack(alignment: .leading) {
                // Background
                RoundedRectangle(cornerRadius: 2)
                    .fill(color.opacity(0.2))
                
                // Level indicator
                RoundedRectangle(cornerRadius: 2)
                    .fill(
                        LinearGradient(
                            colors: [color.opacity(0.8), color],
                            startPoint: .leading,
                            endPoint: .trailing
                        )
                    )
                    .frame(width: geometry.size.width * CGFloat(level))
                    .shadow(color: color, radius: 5)
                    .animation(.easeOut(duration: 0.05), value: level)
            }
        }
    }
}

// MARK: - Voice Control Button

struct VoiceControlButton: View {
    let icon: String
    let label: String
    let color: Color
    let size: CGFloat
    var isPrimary: Bool = false
    let action: () -> Void
    
    @State private var isPressed = false
    
    var body: some View {
        Button(action: action) {
            VStack(spacing: 8) {
                ZStack {
                    Circle()
                        .fill(
                            isPrimary
                            ? color
                            : Color.white.opacity(0.1)
                        )
                        .frame(width: size, height: size)
                        .overlay(
                            Circle()
                                .stroke(color, lineWidth: 2)
                        )
                        .shadow(color: isPrimary ? color.opacity(0.5) : .clear, radius: 10)
                    
                    Image(systemName: icon)
                        .font(.system(size: size * 0.4, weight: .semibold))
                        .foregroundColor(isPrimary ? .white : color)
                }
                .scaleEffect(isPressed ? 0.9 : 1.0)
                .animation(.spring(response: 0.3), value: isPressed)
                
                Text(label)
                    .font(.system(size: 12, weight: .medium))
                    .foregroundColor(color.opacity(0.8))
            }
        }
        .buttonStyle(PlainButtonStyle())
        .simultaneousGesture(
            DragGesture(minimumDistance: 0)
                .onChanged { _ in isPressed = true }
                .onEnded { _ in isPressed = false }
        )
    }
}

// MARK: - Blood Rain Effect

struct BloodRainEffect: View {
    @Binding var drops: [BloodDroplet]
    let color: Color
    
    var body: some View {
        GeometryReader { geometry in
            ZStack {
                ForEach(drops) { drop in
                    BloodDropView(drop: drop, color: color, screenHeight: geometry.size.height)
                }
            }
            .onAppear {
                // Cleanup old drops periodically
                Timer.scheduledTimer(withTimeInterval: 3, repeats: true) { _ in
                    drops.removeAll { $0.startTime.timeIntervalSinceNow < -3 }
                }
            }
        }
    }
}

struct BloodDropView: View {
    let drop: BloodDroplet
    let color: Color
    let screenHeight: CGFloat
    
    @State private var yOffset: CGFloat = -50
    @State private var opacity: Double = 1
    
    var body: some View {
        Circle()
            .fill(
                RadialGradient(
                    colors: [color, color.opacity(0.3)],
                    center: .center,
                    startRadius: 0,
                    endRadius: drop.size
                )
            )
            .frame(width: drop.size, height: drop.size)
            .position(x: drop.x, y: yOffset)
            .opacity(opacity)
            .blur(radius: drop.blur)
            .onAppear {
                withAnimation(.easeIn(duration: drop.duration)) {
                    yOffset = screenHeight + 50
                    opacity = 0
                }
            }
    }
}

struct BloodDroplet: Identifiable {
    let id = UUID()
    let x: CGFloat = CGFloat.random(in: 0...UIScreen.main.bounds.width)
    let size: CGFloat = CGFloat.random(in: 4...12)
    let duration: Double = Double.random(in: 2...4)
    let blur: CGFloat = CGFloat.random(in: 0...2)
    let startTime = Date()
}

// MARK: - Preview

#Preview {
    VoiceOverlay(
        voiceManager: VoiceManager.shared,
        isPresented: .constant(true)
    ) { text in
        print("Received: \(text)")
    }
    .environmentObject(ThemeManager.shared)
}
