//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Infrastructure View                                         ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

struct InfrastructureView: View {
    @EnvironmentObject var theme: ThemeManager
    @State private var isLoading = false
    @State private var healthStatus: [ComponentHealth] = []
    @State private var services: [DockerService] = []
    
    var body: some View {
        NavigationStack {
            ZStack {
                theme.backgroundGradient
                    .ignoresSafeArea()
                
                ScrollView {
                    VStack(spacing: 20) {
                        // Health Overview
                        HealthOverviewCard()
                        
                        // Quick Actions
                        QuickActionsCard()
                        
                        // Services Grid
                        ServicesSection()
                    }
                    .padding()
                }
            }
            .navigationTitle("Infrastructure")
            .toolbar {
                ToolbarItem(placement: .topBarTrailing) {
                    Button(action: refreshData) {
                        Image(systemName: "arrow.clockwise")
                            .foregroundColor(theme.accentColor)
                    }
                }
            }
        }
    }
    
    private func refreshData() {
        isLoading = true
        // TODO: Fetch real data
        DispatchQueue.main.asyncAfter(deadline: .now() + 1) {
            isLoading = false
        }
    }
}

// MARK: - Health Overview Card

struct HealthOverviewCard: View {
    @EnvironmentObject var theme: ThemeManager
    @State private var isAnimating = false
    
    var body: some View {
        VStack(spacing: 16) {
            HStack {
                Text("État du système")
                    .font(.system(size: 18, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                HStack(spacing: 6) {
                    Circle()
                        .fill(theme.successColor)
                        .frame(width: 10, height: 10)
                        .shadow(color: theme.successColor, radius: isAnimating ? 8 : 4)
                    Text("Healthy")
                        .font(.system(size: 14, weight: .medium))
                        .foregroundColor(theme.successColor)
                }
                .onAppear {
                    withAnimation(.easeInOut(duration: 1).repeatForever(autoreverses: true)) {
                        isAnimating = true
                    }
                }
            }
            
            // Stats grid
            LazyVGrid(columns: [
                GridItem(.flexible()),
                GridItem(.flexible()),
                GridItem(.flexible())
            ], spacing: 16) {
                StatCard(title: "Services", value: "24", icon: "cube.box.fill", color: theme.infoColor)
                StatCard(title: "Nodes", value: "3", icon: "server.rack", color: theme.warningColor)
                StatCard(title: "VPN", value: "12", icon: "network", color: theme.successColor)
            }
        }
        .padding(20)
        .background(theme.surfaceColor)
        .cornerRadius(theme.radiusLarge)
        .shadow(color: theme.primaryDark.opacity(0.3), radius: 10, x: 0, y: 5)
    }
}

struct StatCard: View {
    @EnvironmentObject var theme: ThemeManager
    let title: String
    let value: String
    let icon: String
    let color: Color
    
    var body: some View {
        VStack(spacing: 8) {
            Image(systemName: icon)
                .font(.system(size: 24))
                .foregroundColor(color)
            
            Text(value)
                .font(.system(size: 28, weight: .bold, design: .rounded))
                .foregroundColor(theme.textPrimary)
            
            Text(title)
                .font(.system(size: 12))
                .foregroundColor(theme.textMuted)
        }
        .frame(maxWidth: .infinity)
        .padding(.vertical, 16)
        .background(theme.surfaceElevated)
        .cornerRadius(theme.radiusMedium)
    }
}

// MARK: - Quick Actions Card

struct QuickActionsCard: View {
    @EnvironmentObject var theme: ThemeManager
    
    var body: some View {
        VStack(alignment: .leading, spacing: 16) {
            Text("Actions rapides")
                .font(.system(size: 18, weight: .semibold))
                .foregroundColor(theme.textPrimary)
            
            LazyVGrid(columns: [
                GridItem(.flexible()),
                GridItem(.flexible())
            ], spacing: 12) {
                QuickActionButton(
                    icon: "arrow.clockwise",
                    title: "Restart Service",
                    color: theme.warningColor
                ) {
                    // Action
                }
                
                QuickActionButton(
                    icon: "arrow.up.arrow.down",
                    title: "Scale Service",
                    color: theme.infoColor
                ) {
                    // Action
                }
                
                QuickActionButton(
                    icon: "doc.text.magnifyingglass",
                    title: "View Logs",
                    color: theme.textSecondary
                ) {
                    // Action
                }
                
                QuickActionButton(
                    icon: "chart.line.uptrend.xyaxis",
                    title: "Metrics",
                    color: theme.successColor
                ) {
                    // Action
                }
            }
        }
        .padding(20)
        .background(theme.surfaceColor)
        .cornerRadius(theme.radiusLarge)
    }
}

struct QuickActionButton: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let title: String
    let color: Color
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            VStack(spacing: 10) {
                Image(systemName: icon)
                    .font(.system(size: 22))
                    .foregroundColor(color)
                
                Text(title)
                    .font(.system(size: 12, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                    .multilineTextAlignment(.center)
            }
            .frame(maxWidth: .infinity)
            .padding(.vertical, 16)
            .background(theme.surfaceElevated)
            .cornerRadius(theme.radiusMedium)
        }
    }
}

// MARK: - Services Section

struct ServicesSection: View {
    @EnvironmentObject var theme: ThemeManager
    
    let services = [
        ("traefik_traefik", "1/1", "healthy"),
        ("prometheus_prometheus", "1/1", "healthy"),
        ("grafana_grafana", "1/1", "healthy"),
        ("headscale_headscale", "1/1", "healthy"),
        ("mcp_mcp", "1/1", "healthy"),
        ("authentik_server", "2/2", "healthy"),
    ]
    
    var body: some View {
        VStack(alignment: .leading, spacing: 16) {
            HStack {
                Text("Services Docker")
                    .font(.system(size: 18, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                Text("\(services.count) actifs")
                    .font(.system(size: 14))
                    .foregroundColor(theme.textMuted)
            }
            
            VStack(spacing: 8) {
                ForEach(services, id: \.0) { service in
                    ServiceRow(
                        name: service.0,
                        replicas: service.1,
                        status: service.2
                    )
                }
            }
        }
        .padding(20)
        .background(theme.surfaceColor)
        .cornerRadius(theme.radiusLarge)
    }
}

struct ServiceRow: View {
    @EnvironmentObject var theme: ThemeManager
    let name: String
    let replicas: String
    let status: String
    
    var body: some View {
        HStack(spacing: 12) {
            Circle()
                .fill(status == "healthy" ? theme.successColor : theme.warningColor)
                .frame(width: 8, height: 8)
            
            Text(name)
                .font(.system(size: 14, design: .monospaced))
                .foregroundColor(theme.textPrimary)
                .lineLimit(1)
            
            Spacer()
            
            Text(replicas)
                .font(.system(size: 13, weight: .medium))
                .foregroundColor(theme.textSecondary)
                .padding(.horizontal, 10)
                .padding(.vertical, 4)
                .background(theme.surfaceElevated)
                .cornerRadius(theme.radiusSmall)
        }
        .padding(.vertical, 8)
    }
}

// MARK: - Models

struct ComponentHealth: Identifiable {
    let id = UUID()
    let name: String
    let status: String
}

struct DockerService: Identifiable {
    let id = UUID()
    let name: String
    let replicas: String
    let status: String
}

// MARK: - Preview

#Preview {
    InfrastructureView()
        .environmentObject(ThemeManager.shared)
}
