//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Dark Blood Modals - Sophisticated UI Components             ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

// MARK: - Blood Alert Modal

struct BloodAlertModal: View {
    @EnvironmentObject var theme: ThemeManager
    @Binding var isPresented: Bool
    
    let title: String
    let message: String
    let icon: String
    let iconColor: Color?
    let primaryAction: ModalAction?
    let secondaryAction: ModalAction?
    
    @State private var showContent = false
    @State private var pulseGlow = false
    
    init(
        isPresented: Binding<Bool>,
        title: String,
        message: String,
        icon: String = "exclamationmark.triangle.fill",
        iconColor: Color? = nil,
        primaryAction: ModalAction? = nil,
        secondaryAction: ModalAction? = nil
    ) {
        self._isPresented = isPresented
        self.title = title
        self.message = message
        self.icon = icon
        self.iconColor = iconColor
        self.primaryAction = primaryAction
        self.secondaryAction = secondaryAction
    }
    
    var body: some View {
        ZStack {
            // Backdrop
            Color.black.opacity(0.7)
                .ignoresSafeArea()
                .onTapGesture {
                    dismiss()
                }
            
            // Blood effect overlay
            RadialGradient(
                colors: [
                    theme.primaryDark.opacity(0.4),
                    Color.clear
                ],
                center: .top,
                startRadius: 0,
                endRadius: 400
            )
            .ignoresSafeArea()
            
            // Modal content
            VStack(spacing: 0) {
                // Icon header with glow
                ZStack {
                    // Glow effect
                    Circle()
                        .fill((iconColor ?? theme.accentColor).opacity(0.3))
                        .frame(width: 100, height: 100)
                        .blur(radius: pulseGlow ? 30 : 20)
                        .scaleEffect(pulseGlow ? 1.2 : 1.0)
                    
                    // Icon background
                    Circle()
                        .fill(
                            RadialGradient(
                                colors: [
                                    (iconColor ?? theme.accentColor),
                                    theme.primaryDark
                                ],
                                center: .center,
                                startRadius: 0,
                                endRadius: 40
                            )
                        )
                        .frame(width: 80, height: 80)
                        .shadow(color: (iconColor ?? theme.accentColor).opacity(0.5), radius: 15)
                    
                    // Icon
                    Image(systemName: icon)
                        .font(.system(size: 36, weight: .semibold))
                        .foregroundColor(.white)
                }
                .padding(.top, 30)
                .padding(.bottom, 20)
                
                // Title
                Text(title)
                    .font(.system(size: 22, weight: .bold))
                    .foregroundColor(theme.textPrimary)
                    .multilineTextAlignment(.center)
                    .padding(.horizontal, 20)
                
                // Message
                Text(message)
                    .font(.system(size: 15))
                    .foregroundColor(theme.textSecondary)
                    .multilineTextAlignment(.center)
                    .padding(.horizontal, 30)
                    .padding(.top, 12)
                    .padding(.bottom, 30)
                
                // Divider with blood gradient
                Rectangle()
                    .fill(
                        LinearGradient(
                            colors: [
                                Color.clear,
                                theme.primaryColor.opacity(0.5),
                                Color.clear
                            ],
                            startPoint: .leading,
                            endPoint: .trailing
                        )
                    )
                    .frame(height: 1)
                
                // Actions
                HStack(spacing: 0) {
                    if let secondary = secondaryAction {
                        Button(action: {
                            secondary.action()
                            dismiss()
                        }) {
                            Text(secondary.title)
                                .font(.system(size: 17, weight: .medium))
                                .foregroundColor(theme.textSecondary)
                                .frame(maxWidth: .infinity)
                                .padding(.vertical, 16)
                        }
                        
                        // Vertical divider
                        Rectangle()
                            .fill(theme.primaryColor.opacity(0.3))
                            .frame(width: 1)
                    }
                    
                    if let primary = primaryAction {
                        Button(action: {
                            primary.action()
                            dismiss()
                        }) {
                            Text(primary.title)
                                .font(.system(size: 17, weight: .semibold))
                                .foregroundColor(primary.isDestructive ? theme.errorColor : theme.accentColor)
                                .frame(maxWidth: .infinity)
                                .padding(.vertical, 16)
                        }
                    } else {
                        Button(action: dismiss) {
                            Text("OK")
                                .font(.system(size: 17, weight: .semibold))
                                .foregroundColor(theme.accentColor)
                                .frame(maxWidth: .infinity)
                                .padding(.vertical, 16)
                        }
                    }
                }
            }
            .background(
                RoundedRectangle(cornerRadius: 20)
                    .fill(theme.surfaceColor)
                    .overlay(
                        RoundedRectangle(cornerRadius: 20)
                            .stroke(
                                LinearGradient(
                                    colors: [
                                        theme.primaryColor.opacity(0.5),
                                        theme.accentColor.opacity(0.3),
                                        theme.primaryColor.opacity(0.5)
                                    ],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                ),
                                lineWidth: 1
                            )
                    )
                    .shadow(color: theme.primaryDark.opacity(0.5), radius: 30)
            )
            .frame(maxWidth: 320)
            .scaleEffect(showContent ? 1 : 0.8)
            .opacity(showContent ? 1 : 0)
        }
        .onAppear {
            withAnimation(.spring(response: 0.4, dampingFraction: 0.7)) {
                showContent = true
            }
            withAnimation(.easeInOut(duration: 2).repeatForever(autoreverses: true)) {
                pulseGlow = true
            }
        }
    }
    
    private func dismiss() {
        withAnimation(.spring(response: 0.3)) {
            showContent = false
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.3) {
            isPresented = false
        }
    }
}

struct ModalAction {
    let title: String
    let isDestructive: Bool
    let action: () -> Void
    
    init(title: String, isDestructive: Bool = false, action: @escaping () -> Void) {
        self.title = title
        self.isDestructive = isDestructive
        self.action = action
    }
}

// MARK: - Blood Confirmation Sheet

struct BloodConfirmationSheet: View {
    @EnvironmentObject var theme: ThemeManager
    @Binding var isPresented: Bool
    
    let title: String
    let message: String
    let confirmTitle: String
    let confirmColor: Color
    let onConfirm: () -> Void
    
    @State private var offset: CGFloat = 300
    
    var body: some View {
        ZStack(alignment: .bottom) {
            // Backdrop
            Color.black.opacity(0.5)
                .ignoresSafeArea()
                .onTapGesture {
                    dismiss()
                }
            
            // Sheet content
            VStack(spacing: 20) {
                // Handle
                RoundedRectangle(cornerRadius: 3)
                    .fill(theme.textMuted.opacity(0.5))
                    .frame(width: 40, height: 5)
                    .padding(.top, 12)
                
                // Content
                VStack(spacing: 12) {
                    Text(title)
                        .font(.system(size: 20, weight: .bold))
                        .foregroundColor(theme.textPrimary)
                    
                    Text(message)
                        .font(.system(size: 15))
                        .foregroundColor(theme.textSecondary)
                        .multilineTextAlignment(.center)
                        .padding(.horizontal)
                }
                .padding(.vertical, 10)
                
                // Buttons
                VStack(spacing: 12) {
                    // Confirm button
                    Button(action: {
                        onConfirm()
                        dismiss()
                    }) {
                        Text(confirmTitle)
                            .font(.system(size: 17, weight: .semibold))
                            .foregroundColor(.white)
                            .frame(maxWidth: .infinity)
                            .padding(.vertical, 16)
                            .background(
                                LinearGradient(
                                    colors: [confirmColor, confirmColor.opacity(0.7)],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                            .cornerRadius(14)
                            .shadow(color: confirmColor.opacity(0.4), radius: 10)
                    }
                    
                    // Cancel button
                    Button(action: dismiss) {
                        Text("Annuler")
                            .font(.system(size: 17, weight: .medium))
                            .foregroundColor(theme.textSecondary)
                            .frame(maxWidth: .infinity)
                            .padding(.vertical, 16)
                            .background(theme.surfaceElevated)
                            .cornerRadius(14)
                    }
                }
                .padding(.horizontal, 20)
                .padding(.bottom, 30)
            }
            .background(
                RoundedRectangle(cornerRadius: 24)
                    .fill(theme.surfaceColor)
                    .overlay(
                        RoundedRectangle(cornerRadius: 24)
                            .stroke(theme.primaryColor.opacity(0.3), lineWidth: 1)
                    )
                    .shadow(color: theme.primaryDark.opacity(0.5), radius: 20)
            )
            .offset(y: offset)
        }
        .onAppear {
            withAnimation(.spring(response: 0.4, dampingFraction: 0.8)) {
                offset = 0
            }
        }
    }
    
    private func dismiss() {
        withAnimation(.spring(response: 0.3)) {
            offset = 300
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.3) {
            isPresented = false
        }
    }
}

// MARK: - Blood Toast Notification

struct BloodToast: View {
    @EnvironmentObject var theme: ThemeManager
    
    let message: String
    let type: ToastType
    let duration: TimeInterval
    @Binding var isShowing: Bool
    
    @State private var offset: CGFloat = -100
    @State private var opacity: Double = 0
    
    enum ToastType {
        case success, error, warning, info
        
        var icon: String {
            switch self {
            case .success: return "checkmark.circle.fill"
            case .error: return "xmark.circle.fill"
            case .warning: return "exclamationmark.triangle.fill"
            case .info: return "info.circle.fill"
            }
        }
    }
    
    var iconColor: Color {
        switch type {
        case .success: return theme.successColor
        case .error: return theme.errorColor
        case .warning: return theme.warningColor
        case .info: return theme.accentColor
        }
    }
    
    var body: some View {
        HStack(spacing: 12) {
            // Icon with glow
            ZStack {
                Circle()
                    .fill(iconColor.opacity(0.2))
                    .frame(width: 36, height: 36)
                
                Image(systemName: type.icon)
                    .font(.system(size: 20))
                    .foregroundColor(iconColor)
            }
            
            Text(message)
                .font(.system(size: 14, weight: .medium))
                .foregroundColor(theme.textPrimary)
                .lineLimit(2)
            
            Spacer()
            
            // Dismiss button
            Button(action: {
                hideToast()
            }) {
                Image(systemName: "xmark")
                    .font(.system(size: 12, weight: .bold))
                    .foregroundColor(theme.textMuted)
                    .padding(6)
            }
        }
        .padding(.horizontal, 16)
        .padding(.vertical, 14)
        .background(
            RoundedRectangle(cornerRadius: 16)
                .fill(theme.surfaceElevated)
                .overlay(
                    RoundedRectangle(cornerRadius: 16)
                        .stroke(iconColor.opacity(0.3), lineWidth: 1)
                )
                .shadow(color: Color.black.opacity(0.3), radius: 15, y: 5)
        )
        .padding(.horizontal, 20)
        .offset(y: offset)
        .opacity(opacity)
        .onAppear {
            showToast()
        }
    }
    
    private func showToast() {
        withAnimation(.spring(response: 0.4, dampingFraction: 0.7)) {
            offset = 0
            opacity = 1
        }
        
        // Auto-hide after duration
        DispatchQueue.main.asyncAfter(deadline: .now() + duration) {
            hideToast()
        }
    }
    
    private func hideToast() {
        withAnimation(.easeOut(duration: 0.3)) {
            offset = -100
            opacity = 0
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.3) {
            isShowing = false
        }
    }
}

// MARK: - Blood Loading Modal

struct BloodLoadingModal: View {
    @EnvironmentObject var theme: ThemeManager
    @Binding var isPresented: Bool
    
    let message: String
    
    @State private var rotation: Double = 0
    @State private var scale: CGFloat = 1.0
    
    var body: some View {
        ZStack {
            // Backdrop
            Color.black.opacity(0.7)
                .ignoresSafeArea()
            
            VStack(spacing: 24) {
                // Blood spinner
                ZStack {
                    // Outer ring
                    Circle()
                        .stroke(theme.primaryColor.opacity(0.3), lineWidth: 4)
                        .frame(width: 60, height: 60)
                    
                    // Spinning arc
                    Circle()
                        .trim(from: 0, to: 0.7)
                        .stroke(
                            LinearGradient(
                                colors: [theme.accentGlow, theme.primaryColor],
                                startPoint: .leading,
                                endPoint: .trailing
                            ),
                            style: StrokeStyle(lineWidth: 4, lineCap: .round)
                        )
                        .frame(width: 60, height: 60)
                        .rotationEffect(.degrees(rotation))
                    
                    // Center blood drop
                    Image(systemName: "drop.fill")
                        .font(.system(size: 20))
                        .foregroundColor(theme.accentColor)
                        .scaleEffect(scale)
                }
                
                Text(message)
                    .font(.system(size: 15, weight: .medium))
                    .foregroundColor(theme.textPrimary)
            }
            .padding(40)
            .background(
                RoundedRectangle(cornerRadius: 20)
                    .fill(theme.surfaceColor)
                    .overlay(
                        RoundedRectangle(cornerRadius: 20)
                            .stroke(theme.primaryColor.opacity(0.3), lineWidth: 1)
                    )
                    .shadow(color: theme.primaryDark.opacity(0.5), radius: 20)
            )
        }
        .onAppear {
            withAnimation(.linear(duration: 1).repeatForever(autoreverses: false)) {
                rotation = 360
            }
            withAnimation(.easeInOut(duration: 0.8).repeatForever(autoreverses: true)) {
                scale = 1.2
            }
        }
    }
}

// MARK: - Blood Action Sheet

struct BloodActionSheet: View {
    @EnvironmentObject var theme: ThemeManager
    @Binding var isPresented: Bool
    
    let title: String?
    let actions: [ActionSheetItem]
    
    @State private var offset: CGFloat = 400
    
    var body: some View {
        ZStack(alignment: .bottom) {
            // Backdrop
            Color.black.opacity(0.5)
                .ignoresSafeArea()
                .onTapGesture {
                    dismiss()
                }
            
            VStack(spacing: 8) {
                // Actions group
                VStack(spacing: 0) {
                    // Title if provided
                    if let title = title {
                        Text(title)
                            .font(.system(size: 13, weight: .medium))
                            .foregroundColor(theme.textMuted)
                            .padding(.vertical, 16)
                        
                        Divider()
                            .background(theme.primaryColor.opacity(0.3))
                    }
                    
                    // Action buttons
                    ForEach(Array(actions.enumerated()), id: \.offset) { index, action in
                        Button(action: {
                            action.action()
                            dismiss()
                        }) {
                            HStack(spacing: 12) {
                                if let icon = action.icon {
                                    Image(systemName: icon)
                                        .font(.system(size: 20))
                                        .foregroundColor(action.isDestructive ? theme.errorColor : theme.accentColor)
                                        .frame(width: 28)
                                }
                                
                                Text(action.title)
                                    .font(.system(size: 17, weight: action.isDestructive ? .semibold : .regular))
                                    .foregroundColor(action.isDestructive ? theme.errorColor : theme.textPrimary)
                                
                                Spacer()
                            }
                            .padding(.horizontal, 20)
                            .padding(.vertical, 16)
                        }
                        
                        if index < actions.count - 1 {
                            Divider()
                                .background(theme.primaryColor.opacity(0.3))
                        }
                    }
                }
                .background(
                    RoundedRectangle(cornerRadius: 16)
                        .fill(theme.surfaceColor)
                )
                
                // Cancel button
                Button(action: dismiss) {
                    Text("Annuler")
                        .font(.system(size: 17, weight: .semibold))
                        .foregroundColor(theme.accentColor)
                        .frame(maxWidth: .infinity)
                        .padding(.vertical, 16)
                        .background(
                            RoundedRectangle(cornerRadius: 16)
                                .fill(theme.surfaceColor)
                        )
                }
            }
            .padding(.horizontal, 8)
            .padding(.bottom, 30)
            .offset(y: offset)
        }
        .onAppear {
            withAnimation(.spring(response: 0.4, dampingFraction: 0.8)) {
                offset = 0
            }
        }
    }
    
    private func dismiss() {
        withAnimation(.spring(response: 0.3)) {
            offset = 400
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.3) {
            isPresented = false
        }
    }
}

struct ActionSheetItem {
    let title: String
    let icon: String?
    let isDestructive: Bool
    let action: () -> Void
    
    init(title: String, icon: String? = nil, isDestructive: Bool = false, action: @escaping () -> Void) {
        self.title = title
        self.icon = icon
        self.isDestructive = isDestructive
        self.action = action
    }
}

// MARK: - View Extension for Modals

extension View {
    func bloodAlert(
        isPresented: Binding<Bool>,
        title: String,
        message: String,
        icon: String = "exclamationmark.triangle.fill",
        iconColor: Color? = nil,
        primaryAction: ModalAction? = nil,
        secondaryAction: ModalAction? = nil
    ) -> some View {
        ZStack {
            self
            
            if isPresented.wrappedValue {
                BloodAlertModal(
                    isPresented: isPresented,
                    title: title,
                    message: message,
                    icon: icon,
                    iconColor: iconColor,
                    primaryAction: primaryAction,
                    secondaryAction: secondaryAction
                )
                .transition(.opacity)
                .zIndex(100)
            }
        }
    }
    
    func bloodToast(
        isPresented: Binding<Bool>,
        message: String,
        type: BloodToast.ToastType,
        duration: TimeInterval = 3.0
    ) -> some View {
        ZStack(alignment: .top) {
            self
            
            if isPresented.wrappedValue {
                BloodToast(
                    message: message,
                    type: type,
                    duration: duration,
                    isShowing: isPresented
                )
                .padding(.top, 50)
                .zIndex(100)
            }
        }
    }
    
    func bloodLoading(
        isPresented: Binding<Bool>,
        message: String = "Chargement..."
    ) -> some View {
        ZStack {
            self
            
            if isPresented.wrappedValue {
                BloodLoadingModal(isPresented: isPresented, message: message)
                    .transition(.opacity)
                    .zIndex(100)
            }
        }
    }
}

// MARK: - Previews

#Preview("Alert") {
    BloodAlertModal(
        isPresented: .constant(true),
        title: "Confirmation",
        message: "Êtes-vous sûr de vouloir effectuer cette action ?",
        icon: "exclamationmark.triangle.fill",
        primaryAction: ModalAction(title: "Confirmer") {},
        secondaryAction: ModalAction(title: "Annuler") {}
    )
    .environmentObject(ThemeManager.shared)
}

#Preview("Loading") {
    BloodLoadingModal(
        isPresented: .constant(true),
        message: "Connexion en cours..."
    )
    .environmentObject(ThemeManager.shared)
}
