//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Chat View - Main Conversation Interface                     ║
//  ║                    With Voice Overlay & Dark Blood Theme                       ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

struct ChatView: View {
    @EnvironmentObject var chatManager: ChatManager
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var appState: AppState
    @StateObject private var voiceManager = VoiceManager.shared
    
    @State private var inputText = ""
    @State private var showVoiceOverlay = false
    @State private var showProviderPicker = false
    @State private var showAttachmentPicker = false
    @State private var showClearConfirmation = false
    @State private var showToast = false
    @State private var toastMessage = ""
    @State private var toastType: BloodToast.ToastType = .info
    @FocusState private var isInputFocused: Bool
    
    var body: some View {
        NavigationStack {
            ZStack {
                // Background
                theme.backgroundGradient
                    .ignoresSafeArea()
                
                VStack(spacing: 0) {
                    // Messages list
                    ScrollViewReader { proxy in
                        ScrollView {
                            LazyVStack(spacing: 16) {
                                // Welcome message if empty
                                if chatManager.messages.isEmpty {
                                    WelcomeView()
                                        .transition(.opacity.combined(with: .scale))
                                }
                                
                                ForEach(Array(chatManager.messages.enumerated()), id: \.element.id) { index, message in
                                    MessageBubble(message: message)
                                        .messageAppear(delay: Double(index) * 0.05)
                                        .id(message.id)
                                        .contextMenu {
                                            Button {
                                                UIPasteboard.general.string = message.content
                                                showNotification("Copié!", type: .success)
                                            } label: {
                                                Label("Copier", systemImage: "doc.on.doc")
                                            }
                                            
                                            if message.role == .assistant {
                                                Button {
                                                    voiceManager.speak(message.content)
                                                } label: {
                                                    Label("Lire à voix haute", systemImage: "speaker.wave.3")
                                                }
                                            }
                                        }
                                }
                                
                                // Streaming content
                                if !chatManager.streamingContent.isEmpty {
                                    StreamingBubble(content: chatManager.streamingContent)
                                }
                                
                                // Typing indicator
                                if chatManager.isTyping && chatManager.streamingContent.isEmpty {
                                    HStack {
                                        TypingIndicator()
                                        Spacer()
                                    }
                                    .padding(.horizontal)
                                }
                            }
                            .padding()
                        }
                        .onChange(of: chatManager.messages.count) { _, _ in
                            withAnimation(.spring(response: 0.3)) {
                                if let lastId = chatManager.messages.last?.id {
                                    proxy.scrollTo(lastId, anchor: .bottom)
                                }
                            }
                        }
                        .onChange(of: chatManager.streamingContent) { _, _ in
                            withAnimation {
                                if let lastId = chatManager.messages.last?.id {
                                    proxy.scrollTo(lastId, anchor: .bottom)
                                }
                            }
                        }
                    }
                    
                    // Input area
                    EnhancedChatInputBar(
                        text: $inputText,
                        isInputFocused: $isInputFocused,
                        onSend: sendMessage,
                        onVoice: { showVoiceOverlay = true },
                        onAttachment: { showAttachmentPicker = true }
                    )
                }
                
                // Voice Overlay (Full Screen)
                if showVoiceOverlay {
                    VoiceOverlay(
                        voiceManager: voiceManager,
                        isPresented: $showVoiceOverlay
                    ) { transcribedText in
                        Task {
                            await chatManager.sendMessage(transcribedText, isVoice: true)
                        }
                    }
                    .transition(.opacity.combined(with: .scale(scale: 0.95)))
                    .zIndex(50)
                }
                
                // Speaking indicator
                if voiceManager.isSpeaking {
                    SpeakingIndicator {
                        voiceManager.stopSpeaking()
                    }
                    .transition(.move(edge: .bottom).combined(with: .opacity))
                    .zIndex(40)
                }
            }
            .navigationTitle("Assistant")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .topBarLeading) {
                    Button(action: {
                        if !chatManager.messages.isEmpty {
                            showClearConfirmation = true
                        }
                    }) {
                        Image(systemName: "plus.message.fill")
                            .foregroundColor(theme.accentColor)
                    }
                }
                
                ToolbarItem(placement: .topBarTrailing) {
                    Menu {
                        ForEach(AIProvider.allCases, id: \.self) { provider in
                            Button(action: {
                                withAnimation(.spring()) {
                                    chatManager.selectedProvider = provider
                                }
                                showNotification("Provider: \(provider.displayName)", type: .info)
                            }) {
                                Label(provider.displayName, systemImage: provider.icon)
                                if chatManager.selectedProvider == provider {
                                    Image(systemName: "checkmark")
                                }
                            }
                        }
                    } label: {
                        HStack(spacing: 4) {
                            Image(systemName: chatManager.selectedProvider.icon)
                            Image(systemName: "chevron.down")
                                .font(.system(size: 10))
                        }
                        .foregroundColor(theme.accentColor)
                    }
                }
            }
            .animation(.spring(response: 0.4), value: showVoiceOverlay)
        }
        .sheet(isPresented: $showAttachmentPicker) {
            AttachmentPickerView()
        }
        .bloodAlert(
            isPresented: $showClearConfirmation,
            title: "Nouvelle conversation",
            message: "Voulez-vous effacer la conversation actuelle et en démarrer une nouvelle ?",
            icon: "trash.fill",
            iconColor: theme.warningColor,
            primaryAction: ModalAction(title: "Effacer", isDestructive: true) {
                withAnimation(.spring()) {
                    chatManager.clearConversation()
                }
            },
            secondaryAction: ModalAction(title: "Annuler") {}
        )
        .bloodToast(isPresented: $showToast, message: toastMessage, type: toastType)
        .onAppear {
            setupVoiceCallbacks()
        }
        .onChange(of: appState.triggerVoiceInput) { _, newValue in
            if newValue {
                showVoiceOverlay = true
            }
        }
    }
    
    private func sendMessage() {
        guard !inputText.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty else { return }
        
        let message = inputText
        inputText = ""
        
        Task {
            await chatManager.sendMessage(message, isVoice: false)
        }
    }
    
    private func showNotification(_ message: String, type: BloodToast.ToastType) {
        toastMessage = message
        toastType = type
        showToast = true
    }
    
    private func setupVoiceCallbacks() {
        voiceManager.onAutoStop = { text in
            Task {
                await chatManager.sendMessage(text, isVoice: true)
            }
            showVoiceOverlay = false
        }
    }
}

// MARK: - Welcome View

struct WelcomeView: View {
    @EnvironmentObject var theme: ThemeManager
    @State private var showContent = false
    
    var body: some View {
        VStack(spacing: 24) {
            // Logo
            ZStack {
                Circle()
                    .fill(
                        RadialGradient(
                            colors: [theme.accentColor.opacity(0.3), Color.clear],
                            center: .center,
                            startRadius: 20,
                            endRadius: 80
                        )
                    )
                    .frame(width: 120, height: 120)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 50))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [theme.accentGlow, theme.primaryLight],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
                    .shadow(color: theme.accentColor.opacity(0.5), radius: 15)
            }
            .scaleEffect(showContent ? 1 : 0.5)
            .opacity(showContent ? 1 : 0)
            
            VStack(spacing: 12) {
                Text("Personnel Assistant")
                    .font(.system(size: 28, weight: .bold))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [theme.primaryLight, theme.accentColor],
                            startPoint: .leading,
                            endPoint: .trailing
                        )
                    )
                
                Text("Votre orchestrateur AI personnel")
                    .font(.system(size: 16))
                    .foregroundColor(theme.textSecondary)
            }
            .offset(y: showContent ? 0 : 20)
            .opacity(showContent ? 1 : 0)
            
            // Quick actions
            VStack(spacing: 12) {
                WelcomeActionButton(
                    icon: "mic.fill",
                    title: "Commencer une conversation vocale",
                    color: theme.accentColor
                )
                
                WelcomeActionButton(
                    icon: "server.rack",
                    title: "Vérifier l'état de l'infrastructure",
                    color: theme.successColor
                )
                
                WelcomeActionButton(
                    icon: "wand.and.stars",
                    title: "Demander une assistance",
                    color: theme.warningColor
                )
            }
            .padding(.top, 20)
            .offset(y: showContent ? 0 : 30)
            .opacity(showContent ? 1 : 0)
        }
        .padding(.vertical, 40)
        .onAppear {
            withAnimation(.spring(response: 0.6, dampingFraction: 0.7).delay(0.2)) {
                showContent = true
            }
        }
    }
}

struct WelcomeActionButton: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let title: String
    let color: Color
    
    var body: some View {
        HStack(spacing: 12) {
            Image(systemName: icon)
                .font(.system(size: 18))
                .foregroundColor(color)
                .frame(width: 36, height: 36)
                .background(color.opacity(0.15))
                .cornerRadius(10)
            
            Text(title)
                .font(.system(size: 14, weight: .medium))
                .foregroundColor(theme.textPrimary)
            
            Spacer()
            
            Image(systemName: "chevron.right")
                .font(.system(size: 12))
                .foregroundColor(theme.textMuted)
        }
        .padding(.horizontal, 16)
        .padding(.vertical, 14)
        .background(theme.surfaceColor)
        .cornerRadius(14)
        .overlay(
            RoundedRectangle(cornerRadius: 14)
                .stroke(theme.surfaceElevated, lineWidth: 1)
        )
    }
}

// MARK: - Enhanced Chat Input Bar

struct EnhancedChatInputBar: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    @Binding var text: String
    var isInputFocused: FocusState<Bool>.Binding
    let onSend: () -> Void
    let onVoice: () -> Void
    let onAttachment: () -> Void
    
    @State private var isExpanded = false
    
    var body: some View {
        VStack(spacing: 0) {
            // Top divider with gradient
            Rectangle()
                .fill(
                    LinearGradient(
                        colors: [Color.clear, theme.primaryColor.opacity(0.3), Color.clear],
                        startPoint: .leading,
                        endPoint: .trailing
                    )
                )
                .frame(height: 1)
            
            HStack(spacing: 12) {
                // Attachment button
                Button(action: onAttachment) {
                    Image(systemName: "plus.circle.fill")
                        .font(.system(size: 28))
                        .foregroundStyle(
                            LinearGradient(
                                colors: [theme.textMuted, theme.textSecondary],
                                startPoint: .top,
                                endPoint: .bottom
                            )
                        )
                }
                
                // Text input with glow when focused
                HStack(spacing: 8) {
                    TextField("Message...", text: $text, axis: .vertical)
                        .focused(isInputFocused)
                        .lineLimit(1...5)
                        .font(.system(size: 16))
                        .foregroundColor(theme.textPrimary)
                    
                    // Voice button
                    Button(action: onVoice) {
                        ZStack {
                            Circle()
                                .fill(theme.primaryColor.opacity(0.2))
                                .frame(width: 36, height: 36)
                            
                            Image(systemName: "mic.fill")
                                .font(.system(size: 16))
                                .foregroundColor(theme.accentColor)
                        }
                    }
                }
                .padding(.horizontal, 16)
                .padding(.vertical, 10)
                .background(theme.surfaceColor)
                .cornerRadius(24)
                .overlay(
                    RoundedRectangle(cornerRadius: 24)
                        .stroke(
                            isInputFocused.wrappedValue
                            ? theme.accentColor.opacity(0.5)
                            : theme.surfaceElevated,
                            lineWidth: 1
                        )
                )
                .shadow(
                    color: isInputFocused.wrappedValue ? theme.accentColor.opacity(0.2) : .clear,
                    radius: 10
                )
                .animation(.easeOut(duration: 0.2), value: isInputFocused.wrappedValue)
                
                // Send button
                Button(action: onSend) {
                    ZStack {
                        Circle()
                            .fill(
                                canSend
                                ? LinearGradient(
                                    colors: [theme.primaryLight, theme.primaryColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                                : LinearGradient(
                                    colors: [theme.surfaceElevated, theme.surfaceColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                            .frame(width: 44, height: 44)
                        
                        Image(systemName: "arrow.up")
                            .font(.system(size: 18, weight: .bold))
                            .foregroundColor(canSend ? .white : theme.textMuted)
                    }
                    .shadow(color: canSend ? theme.accentColor.opacity(0.4) : .clear, radius: 10)
                    .scaleEffect(canSend ? 1.0 : 0.95)
                    .animation(.spring(response: 0.3), value: canSend)
                }
                .disabled(!canSend || chatManager.isLoading)
            }
            .padding(.horizontal, 16)
            .padding(.vertical, 12)
            .background(theme.backgroundColor)
        }
    }
    
    private var canSend: Bool {
        !text.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty && !chatManager.isLoading
    }
}

// MARK: - Speaking Indicator

struct SpeakingIndicator: View {
    @EnvironmentObject var theme: ThemeManager
    let onStop: () -> Void
    
    @State private var wavePhase: CGFloat = 0
    
    var body: some View {
        VStack {
            Spacer()
            
            HStack(spacing: 16) {
                // Waveform animation
                HStack(spacing: 3) {
                    ForEach(0..<7, id: \.self) { i in
                        Capsule()
                            .fill(theme.accentColor)
                            .frame(width: 3, height: 8 + sin(wavePhase + Double(i) * 0.5) * 8)
                    }
                }
                
                Text("Lecture en cours...")
                    .font(.system(size: 14, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                // Stop button
                Button(action: onStop) {
                    Image(systemName: "stop.fill")
                        .font(.system(size: 14))
                        .foregroundColor(.white)
                        .padding(10)
                        .background(theme.primaryColor)
                        .clipShape(Circle())
                }
            }
            .padding(.horizontal, 20)
            .padding(.vertical, 14)
            .background(
                RoundedRectangle(cornerRadius: 20)
                    .fill(theme.surfaceColor)
                    .shadow(color: theme.primaryDark.opacity(0.3), radius: 15)
            )
            .padding(.horizontal, 16)
            .padding(.bottom, 100)
        }
        .onAppear {
            withAnimation(.linear(duration: 0.5).repeatForever(autoreverses: false)) {
                wavePhase = .pi * 2
            }
        }
    }
}

// MARK: - Chat Input Bar

struct ChatInputBar: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    @Binding var text: String
    @Binding var isRecording: Bool
    @Binding var showProviderPicker: Bool
    @Binding var showAttachmentPicker: Bool
    var isInputFocused: FocusState<Bool>.Binding
    let onSend: () -> Void
    
    var body: some View {
        VStack(spacing: 0) {
            Divider()
                .background(theme.surfaceElevated)
            
            HStack(spacing: 12) {
                // Attachment button
                Button(action: {
                    showAttachmentPicker = true
                }) {
                    Image(systemName: "paperclip")
                        .font(.system(size: 20))
                        .foregroundColor(theme.textSecondary)
                }
                
                // Text input
                HStack(spacing: 8) {
                    TextField("Message...", text: $text, axis: .vertical)
                        .focused(isInputFocused)
                        .lineLimit(1...5)
                        .font(theme.fontBody)
                        .foregroundColor(theme.textPrimary)
                    
                    // Voice button
                    Button(action: {
                        isRecording.toggle()
                    }) {
                        Image(systemName: isRecording ? "waveform.circle.fill" : "mic.fill")
                            .font(.system(size: 20))
                            .foregroundColor(isRecording ? theme.accentGlow : theme.textSecondary)
                            .animation(.easeInOut, value: isRecording)
                    }
                }
                .padding(.horizontal, 16)
                .padding(.vertical, 10)
                .background(theme.surfaceColor)
                .cornerRadius(theme.radiusLarge)
                
                // Send button
                Button(action: onSend) {
                    ZStack {
                        Circle()
                            .fill(
                                LinearGradient(
                                    colors: canSend ? [theme.primaryLight, theme.primaryColor] : [theme.surfaceElevated, theme.surfaceColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                            .frame(width: 40, height: 40)
                        
                        Image(systemName: "arrow.up")
                            .font(.system(size: 16, weight: .bold))
                            .foregroundColor(canSend ? .white : theme.textMuted)
                    }
                    .shadow(color: canSend ? theme.accentColor.opacity(0.4) : .clear, radius: 8)
                }
                .disabled(!canSend || chatManager.isLoading)
            }
            .padding(.horizontal, 16)
            .padding(.vertical, 12)
            .background(theme.backgroundColor)
        }
    }
    
    private var canSend: Bool {
        !text.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty && !chatManager.isLoading
    }
}

// MARK: - Message Bubble

struct MessageBubble: View {
    @EnvironmentObject var theme: ThemeManager
    let message: ChatMessage
    
    var isUser: Bool {
        message.role == .user
    }
    
    var body: some View {
        HStack(alignment: .top, spacing: 12) {
            if isUser {
                Spacer(minLength: 60)
            } else {
                // Avatar
                ZStack {
                    Circle()
                        .fill(theme.primaryGradient)
                        .frame(width: 36, height: 36)
                    
                    Image(systemName: "brain.head.profile")
                        .font(.system(size: 16))
                        .foregroundColor(.white)
                }
                .shadow(color: theme.accentColor.opacity(0.3), radius: 5)
            }
            
            VStack(alignment: isUser ? .trailing : .leading, spacing: 6) {
                // Content
                MessageContent(message: message)
                    .padding(.horizontal, 16)
                    .padding(.vertical, 12)
                    .background(
                        isUser
                        ? AnyShapeStyle(theme.primaryGradient)
                        : AnyShapeStyle(theme.messageGradient)
                    )
                    .cornerRadius(theme.radiusLarge, corners: isUser ? [.topLeft, .topRight, .bottomLeft] : [.topLeft, .topRight, .bottomRight])
                    .shadow(color: theme.primaryDark.opacity(0.2), radius: 5, x: 0, y: 2)
                
                // Metadata
                HStack(spacing: 8) {
                    if let provider = message.provider {
                        HStack(spacing: 4) {
                            Image(systemName: provider.icon)
                                .font(.system(size: 10))
                            Text(provider.rawValue)
                                .font(.system(size: 10))
                        }
                        .foregroundColor(theme.textMuted)
                    }
                    
                    Text(formatTime(message.createdAt))
                        .font(.system(size: 10))
                        .foregroundColor(theme.textMuted)
                    
                    if let time = message.processingTimeMs {
                        Text("(\(time)ms)")
                            .font(.system(size: 10))
                            .foregroundColor(theme.textMuted)
                    }
                }
            }
            
            if !isUser {
                Spacer(minLength: 60)
            }
        }
    }
    
    private func formatTime(_ date: Date) -> String {
        let formatter = DateFormatter()
        formatter.timeStyle = .short
        return formatter.string(from: date)
    }
}

// MARK: - Message Content

struct MessageContent: View {
    @EnvironmentObject var theme: ThemeManager
    let message: ChatMessage
    
    var body: some View {
        switch message.contentType {
        case .markdown:
            MarkdownView(content: message.content)
        case .json:
            JSONView(content: message.content)
        case .code:
            CodeView(content: message.content)
        default:
            Text(message.content)
                .font(theme.fontBody)
                .foregroundColor(message.role == .user ? .white : theme.textPrimary)
        }
    }
}

// MARK: - Streaming Bubble

struct StreamingBubble: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        HStack(alignment: .top, spacing: 12) {
            // Avatar with glow
            ZStack {
                GlowPulse(color: theme.accentColor, size: 36)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 16))
                    .foregroundColor(.white)
            }
            
            VStack(alignment: .leading, spacing: 6) {
                MarkdownView(content: content)
                    .padding(.horizontal, 16)
                    .padding(.vertical, 12)
                    .background(theme.messageGradient)
                    .cornerRadius(theme.radiusLarge, corners: [.topLeft, .topRight, .bottomRight])
                    .overlay(
                        RoundedRectangle(cornerRadius: theme.radiusLarge)
                            .stroke(theme.accentColor.opacity(0.3), lineWidth: 1)
                    )
                
                HStack(spacing: 4) {
                    ProgressView()
                        .scaleEffect(0.7)
                    Text("Streaming...")
                        .font(.system(size: 10))
                }
                .foregroundColor(theme.textMuted)
            }
            
            Spacer(minLength: 60)
        }
    }
}

// MARK: - Attachment Picker

struct AttachmentPickerView: View {
    @EnvironmentObject var theme: ThemeManager
    @Environment(\.dismiss) var dismiss
    
    var body: some View {
        NavigationStack {
            List {
                Section("Documents") {
                    AttachmentOption(icon: "doc.fill", title: "Document", type: .document)
                    AttachmentOption(icon: "doc.richtext.fill", title: "PDF", type: .pdf)
                    AttachmentOption(icon: "tablecells.fill", title: "Tableur", type: .spreadsheet)
                    AttachmentOption(icon: "play.rectangle.fill", title: "Présentation", type: .presentation)
                }
                
                Section("Autres") {
                    AttachmentOption(icon: "photo.fill", title: "Image", type: .image)
                    AttachmentOption(icon: "doc.text.fill", title: "JSON", type: .json)
                    AttachmentOption(icon: "chevron.left.forwardslash.chevron.right", title: "Code", type: .code)
                }
            }
            .navigationTitle("Joindre un fichier")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .cancellationAction) {
                    Button("Annuler") {
                        dismiss()
                    }
                }
            }
        }
    }
}

struct AttachmentOption: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let title: String
    let type: AttachmentType
    
    var body: some View {
        Button(action: {
            // Handle attachment selection
        }) {
            HStack {
                Image(systemName: icon)
                    .foregroundColor(theme.accentColor)
                Text(title)
            }
        }
    }
}

// MARK: - Corner Radius Extension

extension View {
    func cornerRadius(_ radius: CGFloat, corners: UIRectCorner) -> some View {
        clipShape(RoundedCorner(radius: radius, corners: corners))
    }
}

struct RoundedCorner: Shape {
    var radius: CGFloat = .infinity
    var corners: UIRectCorner = .allCorners
    
    func path(in rect: CGRect) -> Path {
        let path = UIBezierPath(
            roundedRect: rect,
            byRoundingCorners: corners,
            cornerRadii: CGSize(width: radius, height: radius)
        )
        return Path(path.cgPath)
    }
}

// MARK: - Preview

#Preview {
    ChatView()
        .environmentObject(ChatManager.shared)
        .environmentObject(ThemeManager.shared)
}
