//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Login View - Dark Blood Theme                               ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

struct LoginView: View {
    @EnvironmentObject var authManager: AuthManager
    @EnvironmentObject var theme: ThemeManager
    @State private var isAnimating = false
    @State private var showParticles = true
    
    var body: some View {
        ZStack {
            // Background
            theme.backgroundGradient
                .ignoresSafeArea()
            
            // Particle effect
            if showParticles {
                ParticleSystem(color: theme.accentColor, count: 30)
                    .opacity(0.3)
            }
            
            // Content
            VStack(spacing: 40) {
                Spacer()
                
                // Logo & Title
                VStack(spacing: 20) {
                    // Animated logo
                    ZStack {
                        // Glow circles
                        ForEach(0..<3, id: \.self) { i in
                            Circle()
                                .stroke(
                                    theme.accentColor.opacity(0.3 - Double(i) * 0.1),
                                    lineWidth: 2
                                )
                                .frame(width: 120 + CGFloat(i) * 30, height: 120 + CGFloat(i) * 30)
                                .scaleEffect(isAnimating ? 1.1 : 0.9)
                                .animation(
                                    Animation.easeInOut(duration: 2)
                                        .repeatForever(autoreverses: true)
                                        .delay(Double(i) * 0.2),
                                    value: isAnimating
                                )
                        }
                        
                        // Main icon
                        Image(systemName: "brain.head.profile")
                            .font(.system(size: 60))
                            .foregroundStyle(
                                LinearGradient(
                                    colors: [theme.primaryLight, theme.accentColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                            .shadow(color: theme.accentGlow.opacity(0.5), radius: 20)
                    }
                    
                    // Title
                    VStack(spacing: 8) {
                        Text("Personnel Assistant")
                            .font(theme.fontTitle)
                            .foregroundColor(theme.textPrimary)
                        
                        Text("CodinCloud AI Orchestrator")
                            .font(theme.fontCaption)
                            .foregroundColor(theme.textSecondary)
                    }
                }
                
                Spacer()
                
                // Features preview
                VStack(spacing: 16) {
                    FeatureRow(icon: "server.rack", title: "Infrastructure", description: "Docker Swarm, VPN, Monitoring")
                    FeatureRow(icon: "brain", title: "AI Intelligent", description: "Claude, GPT-4, Ollama")
                    FeatureRow(icon: "waveform", title: "Voice Control", description: "Commandes vocales intégrées")
                }
                .padding(.horizontal, 32)
                
                Spacer()
                
                // Login button
                VStack(spacing: 16) {
                    Button(action: {
                        authManager.startOAuthFlow()
                    }) {
                        HStack(spacing: 12) {
                            if authManager.isLoading {
                                ProgressView()
                                    .progressViewStyle(CircularProgressViewStyle(tint: .white))
                            } else {
                                Image(systemName: "person.badge.key.fill")
                            }
                            Text("Se connecter avec Authentik")
                                .fontWeight(.semibold)
                        }
                        .frame(maxWidth: .infinity)
                        .padding(.vertical, 16)
                        .background(
                            LinearGradient(
                                colors: [theme.primaryLight, theme.primaryColor],
                                startPoint: .leading,
                                endPoint: .trailing
                            )
                        )
                        .foregroundColor(.white)
                        .cornerRadius(theme.radiusLarge)
                        .shadow(color: theme.accentColor.opacity(0.4), radius: 15, x: 0, y: 5)
                    }
                    .disabled(authManager.isLoading)
                    .padding(.horizontal, 32)
                    
                    // Error message
                    if let error = authManager.error {
                        Text(error)
                            .font(theme.fontCaption)
                            .foregroundColor(theme.errorColor)
                            .multilineTextAlignment(.center)
                            .padding(.horizontal)
                    }
                    
                    // SSO info
                    HStack(spacing: 4) {
                        Image(systemName: "lock.shield.fill")
                            .font(.system(size: 12))
                        Text("Authentification sécurisée SSO")
                            .font(.system(size: 12))
                    }
                    .foregroundColor(theme.textMuted)
                }
                
                Spacer()
                    .frame(height: 40)
            }
        }
        .onAppear {
            isAnimating = true
        }
    }
}

// MARK: - Feature Row

struct FeatureRow: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let title: String
    let description: String
    
    var body: some View {
        HStack(spacing: 16) {
            ZStack {
                Circle()
                    .fill(theme.surfaceElevated)
                    .frame(width: 44, height: 44)
                
                Image(systemName: icon)
                    .font(.system(size: 18))
                    .foregroundColor(theme.accentColor)
            }
            
            VStack(alignment: .leading, spacing: 2) {
                Text(title)
                    .font(.system(size: 16, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
                
                Text(description)
                    .font(.system(size: 13))
                    .foregroundColor(theme.textSecondary)
            }
            
            Spacer()
        }
        .padding(.horizontal, 16)
        .padding(.vertical, 12)
        .background(theme.surfaceColor.opacity(0.5))
        .cornerRadius(theme.radiusMedium)
    }
}

// MARK: - Preview

#Preview {
    LoginView()
        .environmentObject(AuthManager.shared)
        .environmentObject(ThemeManager.shared)
}
