//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Chat Manager - Real-time Communication                      ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import Combine

class ChatManager: ObservableObject {
    static let shared = ChatManager()
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Published State
    // ═══════════════════════════════════════════════════════════════════════════
    
    @Published var messages: [ChatMessage] = []
    @Published var isLoading = false
    @Published var isTyping = false
    @Published var streamingContent = ""
    @Published var currentConversationId: String?
    @Published var error: String?
    @Published var selectedProvider: AIProvider = .auto
    
    // ═══════════════════════════════════════════════════════════════════════════
    // WebSocket
    // ═══════════════════════════════════════════════════════════════════════════
    
    private var webSocketTask: URLSessionWebSocketTask?
    private var isConnected = false
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Configuration
    // ═══════════════════════════════════════════════════════════════════════════
    
    private let baseURL = "https://assistant.dmagh.me"
    private let wsURL = "wss://assistant.dmagh.me"
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Initialization
    // ═══════════════════════════════════════════════════════════════════════════
    
    init() {
        // Add welcome message
        let welcome = ChatMessage(
            id: UUID().uuidString,
            role: .assistant,
            content: """
            👋 Bienvenue dans votre Assistant Personnel CodinCloud!
            
            Je suis votre orchestrateur AI intelligent, prêt à gérer votre infrastructure:
            
            • 🐳 **Docker Swarm** - Services, logs, scaling
            • 🌐 **VPN Headscale** - Nodes, utilisateurs, routes
            • 🔐 **Authentik SSO** - Utilisateurs, applications
            • 📊 **Prometheus/Grafana** - Métriques, alertes
            • 🚦 **Traefik** - Routers, middlewares
            
            Comment puis-je vous aider aujourd'hui?
            """,
            contentType: .markdown,
            provider: .anthropic,
            model: "claude-sonnet-4-20250514",
            thinking: nil,
            toolsUsed: nil,
            audioUrl: nil,
            createdAt: Date(),
            processingTimeMs: nil
        )
        messages.append(welcome)
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Send Message (REST)
    // ═══════════════════════════════════════════════════════════════════════════
    
    func sendMessage(_ content: String, isVoice: Bool = false, attachments: [Attachment] = []) async {
        guard !content.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty else { return }
        
        let userMessage = ChatMessage(
            id: UUID().uuidString,
            role: .user,
            content: content,
            contentType: .text,
            provider: nil,
            model: nil,
            thinking: nil,
            toolsUsed: nil,
            audioUrl: nil,
            createdAt: Date(),
            processingTimeMs: nil
        )
        
        await MainActor.run {
            messages.append(userMessage)
            isLoading = true
            isTyping = true
            error = nil
        }
        
        do {
            // Use streaming endpoint
            let response = try await sendStreamingRequest(content: content, isVoice: isVoice)
            
            await MainActor.run {
                messages.append(response)
                isLoading = false
                isTyping = false
                streamingContent = ""
            }
            
        } catch {
            await MainActor.run {
                self.error = error.localizedDescription
                self.isLoading = false
                self.isTyping = false
            }
        }
    }
    
    private func sendStreamingRequest(content: String, isVoice: Bool) async throws -> ChatMessage {
        // Try direct REST API if streaming fails or no token
        let token = AuthManager.shared.accessToken
        
        // Use direct chat endpoint for reliability
        let url = URL(string: "\(baseURL)/api/v1/chat")!
        var request = URLRequest(url: url)
        request.httpMethod = "POST"
        if let token = token {
            request.setValue("Bearer \(token)", forHTTPHeaderField: "Authorization")
        }
        request.setValue("application/json", forHTTPHeaderField: "Content-Type")
        
        let body: [String: Any] = [
            "content": content,
            "content_type": "text",
            "is_voice": isVoice,
            "provider": selectedProvider.rawValue
        ]
        request.httpBody = try JSONSerialization.data(withJSONObject: body)
        request.timeoutInterval = 60
        
        let (data, response) = try await URLSession.shared.data(for: request)
        
        guard let httpResponse = response as? HTTPURLResponse else {
            throw ChatError.requestFailed
        }
        
        if httpResponse.statusCode != 200 {
            if let errorJson = try? JSONSerialization.jsonObject(with: data) as? [String: Any],
               let detail = errorJson["detail"] as? String {
                throw NSError(domain: "ChatError", code: httpResponse.statusCode, userInfo: [NSLocalizedDescriptionKey: detail])
            }
            throw ChatError.requestFailed
        }
        
        // Parse response
        guard let json = try? JSONSerialization.jsonObject(with: data) as? [String: Any],
              let responseContent = json["content"] as? String else {
            throw ChatError.decodingFailed
        }
        
        let contentTypeStr = json["content_type"] as? String ?? "text"
        let providerStr = json["provider"] as? String
        let model = json["model"] as? String
        let thinking = json["thinking"] as? String
        let processingTime = json["processing_time_ms"] as? Int
        
        return ChatMessage(
            id: json["id"] as? String ?? UUID().uuidString,
            role: .assistant,
            content: responseContent,
            contentType: ContentType(rawValue: contentTypeStr) ?? .text,
            provider: providerStr.flatMap { AIProvider(rawValue: $0) } ?? selectedProvider,
            model: model,
            thinking: thinking,
            toolsUsed: nil,
            audioUrl: nil,
            createdAt: Date(),
            processingTimeMs: processingTime
        )
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // WebSocket Connection
    // ═══════════════════════════════════════════════════════════════════════════
    
    func connectWebSocket() {
        guard let token = AuthManager.shared.accessToken else { return }
        
        let url = URL(string: "\(wsURL)/api/v1/ws/\(token)")!
        webSocketTask = URLSession.shared.webSocketTask(with: url)
        webSocketTask?.resume()
        isConnected = true
        
        receiveMessage()
    }
    
    func disconnectWebSocket() {
        webSocketTask?.cancel(with: .goingAway, reason: nil)
        isConnected = false
    }
    
    private func receiveMessage() {
        webSocketTask?.receive { [weak self] result in
            switch result {
            case .success(let message):
                switch message {
                case .string(let text):
                    self?.handleWebSocketMessage(text)
                case .data(let data):
                    if let text = String(data: data, encoding: .utf8) {
                        self?.handleWebSocketMessage(text)
                    }
                @unknown default:
                    break
                }
                // Continue receiving
                self?.receiveMessage()
                
            case .failure(let error):
                print("WebSocket error: \(error)")
                DispatchQueue.main.async {
                    self?.isConnected = false
                }
            }
        }
    }
    
    private func handleWebSocketMessage(_ text: String) {
        guard let data = text.data(using: .utf8),
              let message = try? JSONDecoder().decode(WSMessage.self, from: data)
        else { return }
        
        DispatchQueue.main.async {
            switch message.type {
            case .thinking:
                self.isTyping = true
                
            case .chunk:
                if let content = message.payload?["content"] {
                    self.streamingContent += content
                }
                
            case .complete:
                if let content = message.payload?["content"] {
                    let response = ChatMessage(
                        id: UUID().uuidString,
                        role: .assistant,
                        content: content,
                        contentType: self.detectContentType(content),
                        provider: self.selectedProvider,
                        model: nil,
                        thinking: nil,
                        toolsUsed: nil,
                        audioUrl: nil,
                        createdAt: Date(),
                        processingTimeMs: nil
                    )
                    self.messages.append(response)
                    self.streamingContent = ""
                    self.isTyping = false
                }
                
            case .error:
                self.error = message.payload?["message"]
                self.isTyping = false
                
            default:
                break
            }
        }
    }
    
    func sendViaWebSocket(_ content: String) {
        let message: [String: Any] = [
            "type": "chat",
            "payload": [
                "content": content,
                "provider": selectedProvider.rawValue
            ]
        ]
        
        guard let data = try? JSONSerialization.data(withJSONObject: message),
              let text = String(data: data, encoding: .utf8)
        else { return }
        
        webSocketTask?.send(.string(text)) { error in
            if let error = error {
                print("WebSocket send error: \(error)")
            }
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Utilities
    // ═══════════════════════════════════════════════════════════════════════════
    
    private func detectContentType(_ content: String) -> ContentType {
        if content.contains("```json") {
            return .json
        } else if content.contains("```") {
            return .code
        } else if content.contains("#") || content.contains("**") || content.contains("- ") {
            return .markdown
        }
        return .text
    }
    
    func clearConversation() {
        messages.removeAll()
        currentConversationId = nil
        
        // Re-add welcome message
        let welcome = ChatMessage(
            id: UUID().uuidString,
            role: .assistant,
            content: "🔄 Nouvelle conversation démarrée. Comment puis-je vous aider?",
            contentType: .markdown,
            provider: .anthropic,
            model: nil,
            thinking: nil,
            toolsUsed: nil,
            audioUrl: nil,
            createdAt: Date(),
            processingTimeMs: nil
        )
        messages.append(welcome)
    }
}

// MARK: - Errors

enum ChatError: LocalizedError {
    case notAuthenticated
    case requestFailed
    case decodingFailed
    
    var errorDescription: String? {
        switch self {
        case .notAuthenticated:
            return "Not authenticated"
        case .requestFailed:
            return "Request failed"
        case .decodingFailed:
            return "Failed to decode response"
        }
    }
}
