//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Auth Manager - Authentik SSO                               ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import AuthenticationServices
import Security
import CommonCrypto

class AuthManager: NSObject, ObservableObject {
    static let shared = AuthManager()
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Published State
    // ═══════════════════════════════════════════════════════════════════════════
    
    @Published var isAuthenticated = false
    @Published var isLoading = false
    @Published var currentUser: User?
    @Published var error: String?
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Configuration - Authentik OAuth2
    // ═══════════════════════════════════════════════════════════════════════════
    
    private let baseURL = "https://assistant.dmagh.me"
    private let authentikURL = "https://sso.codincloud.com"
    private let clientId = "personnel-assistant-ios"
    private let redirectScheme = "personnelassistant"
    private let redirectURI = "personnelassistant://callback"
    
    // PKCE (for public clients)
    private var codeVerifier: String?
    
    // Keychain keys
    private let accessTokenKey = "access_token"
    private let refreshTokenKey = "refresh_token"
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Initialization
    // ═══════════════════════════════════════════════════════════════════════════
    
    override init() {
        super.init()
        checkStoredCredentials()
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // OAuth Flow
    // ═══════════════════════════════════════════════════════════════════════════
    
    func startOAuthFlow() {
        isLoading = true
        error = nil
        
        // Generate PKCE code verifier and challenge
        codeVerifier = generateCodeVerifier()
        guard let verifier = codeVerifier,
              let codeChallenge = generateCodeChallenge(from: verifier) else {
            error = "Failed to generate PKCE challenge"
            isLoading = false
            return
        }
        
        // Build authorization URL with PKCE
        var components = URLComponents(string: "\(authentikURL)/application/o/authorize/")!
        components.queryItems = [
            URLQueryItem(name: "client_id", value: clientId),
            URLQueryItem(name: "response_type", value: "code"),
            URLQueryItem(name: "redirect_uri", value: redirectURI),
            URLQueryItem(name: "scope", value: "openid profile email offline_access"),
            URLQueryItem(name: "state", value: UUID().uuidString),
            URLQueryItem(name: "code_challenge", value: codeChallenge),
            URLQueryItem(name: "code_challenge_method", value: "S256")
        ]
        
        guard let authURL = components.url else {
            error = "Invalid authorization URL"
            isLoading = false
            return
        }
        
        // Use ASWebAuthenticationSession
        let session = ASWebAuthenticationSession(
            url: authURL,
            callbackURLScheme: redirectScheme
        ) { [weak self] callbackURL, error in
            DispatchQueue.main.async {
                self?.handleCallback(callbackURL: callbackURL, error: error)
            }
        }
        
        session.presentationContextProvider = self
        session.prefersEphemeralWebBrowserSession = false
        session.start()
    }
    
    // PKCE Helper Methods
    private func generateCodeVerifier() -> String {
        var bytes = [UInt8](repeating: 0, count: 32)
        _ = SecRandomCopyBytes(kSecRandomDefault, bytes.count, &bytes)
        return Data(bytes).base64EncodedString()
            .replacingOccurrences(of: "+", with: "-")
            .replacingOccurrences(of: "/", with: "_")
            .replacingOccurrences(of: "=", with: "")
    }
    
    private func generateCodeChallenge(from verifier: String) -> String? {
        guard let data = verifier.data(using: .utf8) else { return nil }
        var hash = [UInt8](repeating: 0, count: Int(CC_SHA256_DIGEST_LENGTH))
        data.withUnsafeBytes {
            _ = CC_SHA256($0.baseAddress, CC_LONG(data.count), &hash)
        }
        return Data(hash).base64EncodedString()
            .replacingOccurrences(of: "+", with: "-")
            .replacingOccurrences(of: "/", with: "_")
            .replacingOccurrences(of: "=", with: "")
    }
    
    private func handleCallback(callbackURL: URL?, error: Error?) {
        if let error = error {
            self.error = error.localizedDescription
            self.isLoading = false
            return
        }
        
        guard let url = callbackURL,
              let components = URLComponents(url: url, resolvingAgainstBaseURL: false),
              let code = components.queryItems?.first(where: { $0.name == "code" })?.value
        else {
            self.error = "Invalid callback URL"
            self.isLoading = false
            return
        }
        
        // Exchange code for tokens
        Task {
            await exchangeCode(code)
        }
    }
    
    private func exchangeCode(_ code: String) async {
        guard let verifier = codeVerifier else {
            await MainActor.run {
                self.error = "Missing PKCE verifier"
                self.isLoading = false
            }
            return
        }
        
        do {
            // Exchange code directly with Authentik (public client with PKCE)
            let url = URL(string: "\(authentikURL)/application/o/token/")!
            var request = URLRequest(url: url)
            request.httpMethod = "POST"
            request.setValue("application/x-www-form-urlencoded", forHTTPHeaderField: "Content-Type")
            
            let bodyParams = [
                "grant_type=authorization_code",
                "client_id=\(clientId)",
                "code=\(code)",
                "redirect_uri=\(redirectURI.addingPercentEncoding(withAllowedCharacters: .urlQueryAllowed) ?? redirectURI)",
                "code_verifier=\(verifier)"
            ].joined(separator: "&")
            
            request.httpBody = bodyParams.data(using: .utf8)
            
            let (data, response) = try await URLSession.shared.data(for: request)
            
            guard let httpResponse = response as? HTTPURLResponse,
                  httpResponse.statusCode == 200
            else {
                let errorBody = String(data: data, encoding: .utf8) ?? "Unknown error"
                print("Token exchange failed: \(errorBody)")
                throw AuthError.tokenExchangeFailed
            }
            
            let decoder = JSONDecoder()
            decoder.keyDecodingStrategy = .convertFromSnakeCase
            let oauthResponse = try decoder.decode(OAuthTokenResponse.self, from: data)
            
            // Store tokens
            saveToken(oauthResponse.accessToken, forKey: accessTokenKey)
            if let refreshToken = oauthResponse.refreshToken {
                saveToken(refreshToken, forKey: refreshTokenKey)
            }
            
            // Fetch user info
            await fetchUserInfo(accessToken: oauthResponse.accessToken)
            
            await MainActor.run {
                self.isAuthenticated = true
                self.isLoading = false
            }
            
        } catch {
            await MainActor.run {
                self.error = error.localizedDescription
                self.isLoading = false
            }
        }
    }
    
    private func fetchUserInfo(accessToken: String) async {
        do {
            let url = URL(string: "\(authentikURL)/application/o/userinfo/")!
            var request = URLRequest(url: url)
            request.setValue("Bearer \(accessToken)", forHTTPHeaderField: "Authorization")
            
            let (data, response) = try await URLSession.shared.data(for: request)
            
            guard let httpResponse = response as? HTTPURLResponse,
                  httpResponse.statusCode == 200
            else { return }
            
            let userInfo = try JSONDecoder().decode(AuthentikUserInfo.self, from: data)
            
            await MainActor.run {
                self.currentUser = User(
                    id: userInfo.sub,
                    username: userInfo.preferredUsername ?? userInfo.sub,
                    email: userInfo.email ?? "",
                    name: userInfo.name ?? userInfo.preferredUsername ?? "User",
                    avatar: nil,
                    groups: userInfo.groups ?? []
                )
            }
        } catch {
            print("Failed to fetch user info: \(error)")
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Token Management
    // ═══════════════════════════════════════════════════════════════════════════
    
    var accessToken: String? {
        getToken(forKey: accessTokenKey)
    }
    
    func refreshTokenIfNeeded() async throws {
        guard let refreshToken = getToken(forKey: refreshTokenKey) else {
            throw AuthError.noRefreshToken
        }
        
        let url = URL(string: "\(authentikURL)/application/o/token/")!
        var request = URLRequest(url: url)
        request.httpMethod = "POST"
        request.setValue("application/x-www-form-urlencoded", forHTTPHeaderField: "Content-Type")
        
        let bodyParams = [
            "grant_type=refresh_token",
            "client_id=\(clientId)",
            "refresh_token=\(refreshToken)"
        ].joined(separator: "&")
        
        request.httpBody = bodyParams.data(using: .utf8)
        
        let (data, response) = try await URLSession.shared.data(for: request)
        
        guard let httpResponse = response as? HTTPURLResponse,
              httpResponse.statusCode == 200
        else {
            throw AuthError.tokenRefreshFailed
        }
        
        let decoder = JSONDecoder()
        decoder.keyDecodingStrategy = .convertFromSnakeCase
        let oauthResponse = try decoder.decode(OAuthTokenResponse.self, from: data)
        
        saveToken(oauthResponse.accessToken, forKey: accessTokenKey)
        
        if let newRefreshToken = oauthResponse.refreshToken {
            saveToken(newRefreshToken, forKey: refreshTokenKey)
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Keychain Operations
    // ═══════════════════════════════════════════════════════════════════════════
    
    private func saveToken(_ token: String, forKey key: String) {
        let data = token.data(using: .utf8)!
        
        let query: [String: Any] = [
            kSecClass as String: kSecClassGenericPassword,
            kSecAttrAccount as String: key,
            kSecValueData as String: data
        ]
        
        SecItemDelete(query as CFDictionary)
        SecItemAdd(query as CFDictionary, nil)
    }
    
    private func getToken(forKey key: String) -> String? {
        let query: [String: Any] = [
            kSecClass as String: kSecClassGenericPassword,
            kSecAttrAccount as String: key,
            kSecReturnData as String: true
        ]
        
        var result: AnyObject?
        let status = SecItemCopyMatching(query as CFDictionary, &result)
        
        guard status == errSecSuccess,
              let data = result as? Data,
              let token = String(data: data, encoding: .utf8)
        else {
            return nil
        }
        
        return token
    }
    
    private func deleteToken(forKey key: String) {
        let query: [String: Any] = [
            kSecClass as String: kSecClassGenericPassword,
            kSecAttrAccount as String: key
        ]
        SecItemDelete(query as CFDictionary)
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Session Management
    // ═══════════════════════════════════════════════════════════════════════════
    
    private func checkStoredCredentials() {
        if accessToken != nil {
            Task {
                await fetchCurrentUser()
            }
        }
    }
    
    private func fetchCurrentUser() async {
        guard let token = accessToken else { return }
        
        do {
            // Use Authentik userinfo endpoint
            let url = URL(string: "\(authentikURL)/application/o/userinfo/")!
            var request = URLRequest(url: url)
            request.setValue("Bearer \(token)", forHTTPHeaderField: "Authorization")
            
            let (data, response) = try await URLSession.shared.data(for: request)
            
            guard let httpResponse = response as? HTTPURLResponse,
                  httpResponse.statusCode == 200
            else {
                throw AuthError.invalidSession
            }
            
            let userInfo = try JSONDecoder().decode(AuthentikUserInfo.self, from: data)
            
            await MainActor.run {
                self.currentUser = User(
                    id: userInfo.sub,
                    username: userInfo.preferredUsername ?? userInfo.sub,
                    email: userInfo.email ?? "",
                    name: userInfo.name ?? userInfo.preferredUsername ?? "User",
                    avatar: nil,
                    groups: userInfo.groups ?? []
                )
                self.isAuthenticated = true
            }
            
        } catch {
            // Token might be expired, try refresh
            do {
                try await refreshTokenIfNeeded()
                await fetchCurrentUser()
            } catch {
                await MainActor.run {
                    self.logout()
                }
            }
        }
    }
    
    func logout() {
        deleteToken(forKey: accessTokenKey)
        deleteToken(forKey: refreshTokenKey)
        
        DispatchQueue.main.async {
            withAnimation {
                self.isAuthenticated = false
                self.currentUser = nil
            }
        }
    }
}

// MARK: - ASWebAuthenticationPresentationContextProviding

extension AuthManager: ASWebAuthenticationPresentationContextProviding {
    func presentationAnchor(for session: ASWebAuthenticationSession) -> ASPresentationAnchor {
        guard let scene = UIApplication.shared.connectedScenes.first as? UIWindowScene,
              let window = scene.windows.first
        else {
            return UIWindow()
        }
        return window
    }
}

// MARK: - Errors

enum AuthError: LocalizedError {
    case tokenExchangeFailed
    case tokenRefreshFailed
    case noRefreshToken
    case invalidSession
    
    var errorDescription: String? {
        switch self {
        case .tokenExchangeFailed:
            return "Failed to exchange authorization code"
        case .tokenRefreshFailed:
            return "Failed to refresh access token"
        case .noRefreshToken:
            return "No refresh token available"
        case .invalidSession:
            return "Session is invalid"
        }
    }
}

// MARK: - OAuth Response Models

struct OAuthTokenResponse: Codable {
    let accessToken: String
    let tokenType: String
    let expiresIn: Int?
    let refreshToken: String?
    let idToken: String?
    let scope: String?
}

struct AuthentikUserInfo: Codable {
    let sub: String
    let email: String?
    let emailVerified: Bool?
    let name: String?
    let givenName: String?
    let preferredUsername: String?
    let nickname: String?
    let groups: [String]?
    
    enum CodingKeys: String, CodingKey {
        case sub, email, name, nickname, groups
        case emailVerified = "email_verified"
        case givenName = "given_name"
        case preferredUsername = "preferred_username"
    }
}
