//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║           🤖 Personnel Assistant - iOS App                                     ║
//  ║           Dark Blood Theme with Sophisticated Animations                       ║
//  ║           Widget, Siri & Apple Watch Support                                   ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import WatchConnectivity

@main
struct PersonnelAssistantApp: App {
    @StateObject private var authManager = AuthManager.shared
    @StateObject private var themeManager = ThemeManager.shared
    @StateObject private var chatManager = ChatManager.shared
    @StateObject private var appState = AppState.shared
    @StateObject private var watchConnector = WatchConnector.shared
    
    var body: some Scene {
        WindowGroup {
            RootView()
                .environmentObject(authManager)
                .environmentObject(themeManager)
                .environmentObject(chatManager)
                .environmentObject(appState)
                .preferredColorScheme(.dark)
                .onOpenURL { url in
                    handleDeepLink(url)
                }
                .onReceive(NotificationCenter.default.publisher(for: Notification.Name("StartVoiceRecording"))) { _ in
                    appState.triggerVoiceInput = true
                }
                .onReceive(NotificationCenter.default.publisher(for: Notification.Name("SendChatMessage"))) { notification in
                    if let message = notification.userInfo?["message"] as? String {
                        Task {
                            await chatManager.sendMessage(message, isVoice: false)
                        }
                    }
                }
        }
    }
    
    private func handleDeepLink(_ url: URL) {
        guard let host = url.host else { return }
        
        switch host {
        case "voice":
            appState.selectedTab = 0
            appState.triggerVoiceInput = true
            
        case "chat":
            appState.selectedTab = 0
            
        case "infrastructure":
            appState.selectedTab = 1
            
        case "tools":
            appState.selectedTab = 2
            
        case "settings":
            appState.selectedTab = 3
            
        case "home":
            appState.selectedTab = 0
            
        default:
            break
        }
    }
}

// MARK: - App State

class AppState: ObservableObject {
    static let shared = AppState()
    
    @Published var selectedTab: Int = 0
    @Published var triggerVoiceInput: Bool = false
    @Published var pendingMessage: String?
}

// MARK: - Watch Connector

class WatchConnector: NSObject, ObservableObject, WCSessionDelegate {
    static let shared = WatchConnector()
    
    @Published var isWatchReachable = false
    
    private var session: WCSession?
    
    override init() {
        super.init()
        if WCSession.isSupported() {
            session = WCSession.default
            session?.delegate = self
            session?.activate()
        }
    }
    
    // Send message to Watch
    func sendToWatch(_ message: [String: Any]) {
        guard let session = session, session.isReachable else { return }
        session.sendMessage(message, replyHandler: nil, errorHandler: nil)
    }
    
    // Send infrastructure status to Watch
    func sendInfraStatus() {
        let status: [String: Bool] = [
            "docker": true,
            "vpn": true,
            "sso": true,
            "api": true
        ]
        sendToWatch(["infraStatus": status])
    }
    
    // WCSessionDelegate
    func session(_ session: WCSession, activationDidCompleteWith activationState: WCSessionActivationState, error: Error?) {
        DispatchQueue.main.async {
            self.isWatchReachable = session.isReachable
        }
    }
    
    func sessionDidBecomeInactive(_ session: WCSession) {}
    
    func sessionDidDeactivate(_ session: WCSession) {
        session.activate()
    }
    
    func sessionReachabilityDidChange(_ session: WCSession) {
        DispatchQueue.main.async {
            self.isWatchReachable = session.isReachable
        }
    }
    
    func session(_ session: WCSession, didReceiveMessage message: [String : Any], replyHandler: @escaping ([String : Any]) -> Void) {
        DispatchQueue.main.async {
            if let action = message["action"] as? String {
                switch action {
                case "chat":
                    if let text = message["message"] as? String {
                        Task {
                            await ChatManager.shared.sendMessage(text, isVoice: false)
                            replyHandler(["reply": "Message envoyé"])
                        }
                    }
                    
                case "startVoice":
                    AppState.shared.triggerVoiceInput = true
                    replyHandler(["status": "ok"])
                    
                case "infraStatus":
                    replyHandler([
                        "status": [
                            "docker": true,
                            "vpn": true,
                            "sso": true,
                            "api": true
                        ]
                    ])
                    
                default:
                    replyHandler(["error": "Unknown action"])
                }
            }
        }
    }
}

// MARK: - Root View

struct RootView: View {
    @EnvironmentObject var authManager: AuthManager
    @EnvironmentObject var themeManager: ThemeManager
    @EnvironmentObject var appState: AppState
    
    var body: some View {
        ZStack {
            // Background gradient
            themeManager.backgroundGradient
                .ignoresSafeArea()
            
            // Content
            if authManager.isAuthenticated {
                MainTabView()
                    .transition(.asymmetric(
                        insertion: .opacity.combined(with: .scale(scale: 0.95)),
                        removal: .opacity
                    ))
            } else {
                LoginView()
                    .transition(.opacity)
            }
        }
        .animation(.easeInOut(duration: 0.5), value: authManager.isAuthenticated)
    }
}

// MARK: - Main Tab View

struct MainTabView: View {
    @EnvironmentObject var themeManager: ThemeManager
    @EnvironmentObject var appState: AppState
    
    var body: some View {
        TabView(selection: $appState.selectedTab) {
            ChatView()
                .tabItem {
                    Label("Chat", systemImage: "bubble.left.and.bubble.right.fill")
                }
                .tag(0)
            
            InfrastructureView()
                .tabItem {
                    Label("Infra", systemImage: "server.rack")
                }
                .tag(1)
            
            ToolsView()
                .tabItem {
                    Label("Tools", systemImage: "wrench.and.screwdriver.fill")
                }
                .tag(2)
            
            SettingsView()
                .tabItem {
                    Label("Settings", systemImage: "gearshape.fill")
                }
                .tag(3)
        }
        .tint(themeManager.accentColor)
        .onChange(of: appState.triggerVoiceInput) { _, newValue in
            if newValue {
                // The ChatView will handle this
                DispatchQueue.main.asyncAfter(deadline: .now() + 0.1) {
                    appState.triggerVoiceInput = false
                }
            }
        }
    }
}
