//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Content Viewers - Multi-format Support                      ║
//  ║         Enhanced Markdown with Tables, Full-Width Rendering                    ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import WebKit

// MARK: - Enhanced Markdown View with Table Support

struct MarkdownView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    var fullWidth: Bool = true
    
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            ForEach(parseMarkdown(content), id: \.id) { element in
                element.view
                    .frame(maxWidth: .infinity, alignment: element.alignment)
            }
        }
        .frame(maxWidth: fullWidth ? .infinity : nil, alignment: .leading)
    }
    
    private func parseMarkdown(_ content: String) -> [MarkdownElement] {
        var elements: [MarkdownElement] = []
        let lines = content.components(separatedBy: "\n")
        var codeBlock = ""
        var inCodeBlock = false
        var codeLanguage = ""
        var tableRows: [[String]] = []
        var inTable = false
        
        for (index, line) in lines.enumerated() {
            // Code blocks
            if line.hasPrefix("```") {
                if inCodeBlock {
                    elements.append(MarkdownElement(type: .code, content: codeBlock, language: codeLanguage))
                    codeBlock = ""
                    inCodeBlock = false
                    codeLanguage = ""
                } else {
                    // Flush any pending table
                    if inTable && !tableRows.isEmpty {
                        elements.append(MarkdownElement(type: .table, tableData: tableRows))
                        tableRows = []
                        inTable = false
                    }
                    inCodeBlock = true
                    codeLanguage = String(line.dropFirst(3)).trimmingCharacters(in: .whitespaces)
                }
                continue
            }
            
            if inCodeBlock {
                if !codeBlock.isEmpty { codeBlock += "\n" }
                codeBlock += line
                continue
            }
            
            // Table detection
            if line.contains("|") && line.trimmingCharacters(in: .whitespaces).hasPrefix("|") {
                // Check if it's a separator line (|---|---|)
                let isSeparator = line.contains("---") || line.contains(":-")
                
                if !isSeparator {
                    let cells = line.split(separator: "|", omittingEmptySubsequences: false)
                        .map { String($0).trimmingCharacters(in: .whitespaces) }
                        .filter { !$0.isEmpty }
                    
                    if !cells.isEmpty {
                        tableRows.append(cells)
                        inTable = true
                    }
                }
                continue
            } else if inTable && !tableRows.isEmpty {
                // End of table
                elements.append(MarkdownElement(type: .table, tableData: tableRows))
                tableRows = []
                inTable = false
            }
            
            // Headers
            if line.hasPrefix("#### ") {
                elements.append(MarkdownElement(type: .h4, content: String(line.dropFirst(5))))
            } else if line.hasPrefix("### ") {
                elements.append(MarkdownElement(type: .h3, content: String(line.dropFirst(4))))
            } else if line.hasPrefix("## ") {
                elements.append(MarkdownElement(type: .h2, content: String(line.dropFirst(3))))
            } else if line.hasPrefix("# ") {
                elements.append(MarkdownElement(type: .h1, content: String(line.dropFirst(2))))
            }
            // Horizontal rule
            else if line.trimmingCharacters(in: .whitespaces).hasPrefix("---") ||
                    line.trimmingCharacters(in: .whitespaces).hasPrefix("***") {
                elements.append(MarkdownElement(type: .horizontalRule, content: ""))
            }
            // Numbered list
            else if let match = line.firstMatch(of: /^(\d+)\.\s+(.+)/) {
                elements.append(MarkdownElement(type: .numberedList, content: String(match.2), listNumber: Int(match.1) ?? 1))
            }
            // Bullet list
            else if line.hasPrefix("- ") || line.hasPrefix("• ") || line.hasPrefix("* ") {
                let content = line.dropFirst(2)
                elements.append(MarkdownElement(type: .bullet, content: String(content)))
            }
            // Checkbox
            else if line.hasPrefix("- [ ] ") {
                elements.append(MarkdownElement(type: .checkbox, content: String(line.dropFirst(6)), checked: false))
            } else if line.hasPrefix("- [x] ") || line.hasPrefix("- [X] ") {
                elements.append(MarkdownElement(type: .checkbox, content: String(line.dropFirst(6)), checked: true))
            }
            // Blockquote
            else if line.hasPrefix("> ") {
                elements.append(MarkdownElement(type: .quote, content: String(line.dropFirst(2))))
            }
            // Empty line
            else if line.trimmingCharacters(in: .whitespaces).isEmpty {
                // Add spacing
                if !elements.isEmpty && elements.last?.type != .spacing {
                    elements.append(MarkdownElement(type: .spacing, content: ""))
                }
            }
            // Paragraph
            else if !line.isEmpty {
                elements.append(MarkdownElement(type: .paragraph, content: line))
            }
        }
        
        // Flush remaining table
        if inTable && !tableRows.isEmpty {
            elements.append(MarkdownElement(type: .table, tableData: tableRows))
        }
        
        return elements
    }
}

// MARK: - Markdown Element

struct MarkdownElement: Identifiable {
    let id = UUID()
    let type: MarkdownType
    var content: String = ""
    var language: String = ""
    var tableData: [[String]] = []
    var listNumber: Int = 1
    var checked: Bool = false
    
    var alignment: Alignment {
        switch type {
        case .table:
            return .center
        default:
            return .leading
        }
    }
    
    @ViewBuilder
    var view: some View {
        let theme = ThemeManager.shared
        
        switch type {
        case .h1:
            HStack(spacing: 8) {
                Rectangle()
                    .fill(theme.goldAccent)
                    .frame(width: 4)
                    .cornerRadius(2)
                
                Text(parseInlineMarkdown(content))
                    .font(.system(size: 26, weight: .bold))
                    .foregroundColor(theme.textPrimary)
            }
            .padding(.vertical, 4)
            
        case .h2:
            Text(parseInlineMarkdown(content))
                .font(.system(size: 22, weight: .bold))
                .foregroundStyle(
                    LinearGradient(
                        colors: [theme.textPrimary, theme.textPrimary.opacity(0.8)],
                        startPoint: .leading,
                        endPoint: .trailing
                    )
                )
                .padding(.top, 8)
                .padding(.bottom, 4)
            
        case .h3:
            Text(parseInlineMarkdown(content))
                .font(.system(size: 18, weight: .semibold))
                .foregroundColor(theme.textPrimary)
                .padding(.top, 6)
            
        case .h4:
            Text(parseInlineMarkdown(content))
                .font(.system(size: 16, weight: .semibold))
                .foregroundColor(theme.textSecondary)
                .padding(.top, 4)
            
        case .paragraph:
            Text(parseInlineMarkdown(content))
                .font(.system(size: 15, weight: .regular))
                .foregroundColor(theme.textPrimary)
                .lineSpacing(4)
                .fixedSize(horizontal: false, vertical: true)
            
        case .bullet:
            HStack(alignment: .top, spacing: 10) {
                Circle()
                    .fill(theme.accentColor)
                    .frame(width: 6, height: 6)
                    .padding(.top, 7)
                
                Text(parseInlineMarkdown(content))
                    .font(.system(size: 15))
                    .foregroundColor(theme.textPrimary)
                    .lineSpacing(3)
                    .fixedSize(horizontal: false, vertical: true)
            }
            .padding(.leading, 4)
            
        case .numberedList:
            HStack(alignment: .top, spacing: 10) {
                Text("\(listNumber).")
                    .font(.system(size: 15, weight: .semibold))
                    .foregroundColor(theme.accentColor)
                    .frame(width: 24, alignment: .trailing)
                
                Text(parseInlineMarkdown(content))
                    .font(.system(size: 15))
                    .foregroundColor(theme.textPrimary)
                    .lineSpacing(3)
                    .fixedSize(horizontal: false, vertical: true)
            }
            
        case .checkbox:
            HStack(alignment: .top, spacing: 10) {
                Image(systemName: checked ? "checkmark.square.fill" : "square")
                    .font(.system(size: 16))
                    .foregroundColor(checked ? theme.successColor : theme.textMuted)
                
                Text(parseInlineMarkdown(content))
                    .font(.system(size: 15))
                    .foregroundColor(checked ? theme.textSecondary : theme.textPrimary)
                    .strikethrough(checked)
                    .lineSpacing(3)
            }
            .padding(.leading, 4)
            
        case .quote:
            HStack(spacing: 12) {
                RoundedRectangle(cornerRadius: 2)
                    .fill(
                        LinearGradient(
                            colors: [theme.accentColor, theme.accentColor.opacity(0.5)],
                            startPoint: .top,
                            endPoint: .bottom
                        )
                    )
                    .frame(width: 4)
                
                Text(parseInlineMarkdown(content))
                    .font(.system(size: 15, weight: .regular, design: .serif))
                    .foregroundColor(theme.textSecondary)
                    .italic()
                    .lineSpacing(3)
            }
            .padding(.vertical, 6)
            .padding(.horizontal, 8)
            .background(theme.surfaceElevated.opacity(0.5))
            .cornerRadius(8)
            
        case .code:
            CodeBlockView(code: content, language: language)
            
        case .table:
            MarkdownTableView(rows: tableData)
            
        case .horizontalRule:
            HStack {
                Rectangle()
                    .fill(
                        LinearGradient(
                            colors: [.clear, theme.textMuted.opacity(0.3), .clear],
                            startPoint: .leading,
                            endPoint: .trailing
                        )
                    )
                    .frame(height: 1)
            }
            .padding(.vertical, 12)
            
        case .spacing:
            Color.clear.frame(height: 8)
        }
    }
    
    private func parseInlineMarkdown(_ text: String) -> AttributedString {
        var result = AttributedString(text)
        let theme = ThemeManager.shared
        
        // Process bold: **text** or __text__
        if let boldPattern = try? Regex(#"\*\*(.+?)\*\*"#) {
            for match in text.matches(of: boldPattern) {
                let matchedText = String(match.0)
                if let range = result.range(of: matchedText) {
                    let innerText = String(matchedText.dropFirst(2).dropLast(2))
                    var replacement = AttributedString(innerText)
                    replacement.font = .boldSystemFont(ofSize: 15)
                    result.replaceSubrange(range, with: replacement)
                }
            }
        }
        
        // Process italic: *text* or _text_
        if let italicPattern = try? Regex(#"(?<!\*)\*([^*]+)\*(?!\*)"#) {
            for match in text.matches(of: italicPattern) {
                let matchedText = String(match.0)
                if let range = result.range(of: matchedText) {
                    let innerText = String(matchedText.dropFirst(1).dropLast(1))
                    var replacement = AttributedString(innerText)
                    replacement.font = .italicSystemFont(ofSize: 15)
                    result.replaceSubrange(range, with: replacement)
                }
            }
        }
        
        // Process inline code: `text`
        if let codePattern = try? Regex(#"`([^`]+)`"#) {
            for match in text.matches(of: codePattern) {
                let matchedText = String(match.0)
                if let range = result.range(of: matchedText) {
                    let innerText = String(matchedText.dropFirst(1).dropLast(1))
                    var replacement = AttributedString(innerText)
                    replacement.font = .monospacedSystemFont(ofSize: 14, weight: .regular)
                    replacement.backgroundColor = theme.surfaceElevated
                    replacement.foregroundColor = theme.accentColor
                    result.replaceSubrange(range, with: replacement)
                }
            }
        }
        
        return result
    }
}

enum MarkdownType {
    case h1, h2, h3, h4
    case paragraph
    case bullet, numberedList, checkbox
    case quote
    case code
    case table
    case horizontalRule
    case spacing
}

// MARK: - Markdown Table View

struct MarkdownTableView: View {
    @EnvironmentObject var theme: ThemeManager
    let rows: [[String]]
    
    var body: some View {
        if rows.isEmpty { return AnyView(EmptyView()) }
        
        return AnyView(
            ScrollView(.horizontal, showsIndicators: false) {
                VStack(spacing: 0) {
                    ForEach(rows.indices, id: \.self) { rowIndex in
                        let isHeader = rowIndex == 0
                        
                        HStack(spacing: 0) {
                            ForEach(rows[rowIndex].indices, id: \.self) { colIndex in
                                let cell = rows[rowIndex][colIndex]
                                
                                Text(cell)
                                    .font(.system(size: isHeader ? 13 : 14, weight: isHeader ? .bold : .regular))
                                    .foregroundColor(isHeader ? theme.textPrimary : theme.textSecondary)
                                    .lineLimit(3)
                                    .padding(.horizontal, 12)
                                    .padding(.vertical, 10)
                                    .frame(minWidth: cellWidth(for: colIndex), alignment: .leading)
                                    .background(
                                        isHeader
                                            ? theme.surfaceElevated
                                            : (rowIndex % 2 == 0 ? theme.surfaceColor.opacity(0.3) : Color.clear)
                                    )
                            }
                        }
                        
                        if isHeader {
                            Rectangle()
                                .fill(theme.accentColor.opacity(0.5))
                                .frame(height: 2)
                        }
                    }
                }
                .cornerRadius(10)
                .overlay(
                    RoundedRectangle(cornerRadius: 10)
                        .stroke(theme.surfaceElevated, lineWidth: 1)
                )
            }
        )
    }
    
    private func cellWidth(for column: Int) -> CGFloat {
        // Calculate optimal width based on content
        // Guard against empty rows or invalid column
        guard !rows.isEmpty, column >= 0 else { return 100 }
        
        let maxContent = rows.compactMap { row in
            column < row.count ? row[column] : nil
        }.max(by: { $0.count < $1.count }) ?? ""
        
        let minWidth: CGFloat = 80
        let maxWidth: CGFloat = 200
        let charWidth: CGFloat = 8
        
        // Ensure we never return NaN or Infinity
        let contentLength = max(0, maxContent.count)
        let calculatedWidth = CGFloat(contentLength) * charWidth + 24
        
        // Final safety check
        let result = min(max(calculatedWidth, minWidth), maxWidth)
        return result.isNaN || result.isInfinite ? minWidth : result
    }
}

// MARK: - Code Block View

struct CodeBlockView: View {
    @EnvironmentObject var theme: ThemeManager
    let code: String
    let language: String
    @State private var isCopied = false
    
    var body: some View {
        VStack(alignment: .leading, spacing: 0) {
            // Header
            HStack {
                HStack(spacing: 6) {
                    Circle().fill(Color.red.opacity(0.8)).frame(width: 10, height: 10)
                    Circle().fill(Color.yellow.opacity(0.8)).frame(width: 10, height: 10)
                    Circle().fill(Color.green.opacity(0.8)).frame(width: 10, height: 10)
                }
                
                Spacer()
                
                Text(language.isEmpty ? "code" : language)
                    .font(.system(size: 11, weight: .medium, design: .monospaced))
                    .foregroundColor(theme.textMuted)
                    .padding(.horizontal, 8)
                    .padding(.vertical, 3)
                    .background(Capsule().fill(theme.surfaceColor))
                
                Spacer()
                
                Button(action: copyCode) {
                    HStack(spacing: 4) {
                        Image(systemName: isCopied ? "checkmark" : "doc.on.doc")
                            .font(.system(size: 11))
                        Text(isCopied ? "Copié!" : "Copier")
                            .font(.system(size: 11))
                    }
                    .foregroundColor(isCopied ? theme.successColor : theme.textMuted)
                    .padding(.horizontal, 8)
                    .padding(.vertical, 4)
                    .background(Capsule().fill(theme.surfaceColor))
                }
            }
            .padding(.horizontal, 12)
            .padding(.vertical, 8)
            .background(theme.surfaceElevated)
            
            // Code with line numbers
            ScrollView(.horizontal, showsIndicators: false) {
                HStack(alignment: .top, spacing: 0) {
                    // Line numbers
                    VStack(alignment: .trailing, spacing: 0) {
                        ForEach(Array(code.components(separatedBy: "\n").enumerated()), id: \.offset) { index, _ in
                            Text("\(index + 1)")
                                .font(.system(size: 12, design: .monospaced))
                                .foregroundColor(theme.textMuted.opacity(0.5))
                                .frame(height: 20)
                        }
                    }
                    .padding(.horizontal, 10)
                    .padding(.vertical, 12)
                    .background(theme.surfaceElevated.opacity(0.5))
                    
                    // Code
                    Text(code)
                        .font(.system(size: 13, design: .monospaced))
                        .foregroundColor(theme.textPrimary)
                        .padding(12)
                        .textSelection(.enabled)
                }
            }
            .background(theme.surfaceColor)
        }
        .cornerRadius(10)
        .overlay(
            RoundedRectangle(cornerRadius: 10)
                .stroke(theme.surfaceElevated, lineWidth: 1)
        )
    }
    
    private func copyCode() {
        UIPasteboard.general.string = code
        withAnimation(.spring()) {
            isCopied = true
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 2) {
            withAnimation {
                isCopied = false
            }
        }
    }
}

// MARK: - JSON View

struct JSONView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    @State private var isExpanded = true
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            // Header
            HStack {
                HStack(spacing: 6) {
                    Image(systemName: "curlybraces")
                        .foregroundColor(theme.accentColor)
                    Text("JSON")
                        .font(.system(size: 14, weight: .semibold))
                        .foregroundColor(theme.textSecondary)
                }
                
                Spacer()
                
                Button(action: { withAnimation { isExpanded.toggle() } }) {
                    Image(systemName: isExpanded ? "chevron.up.circle.fill" : "chevron.down.circle.fill")
                        .font(.system(size: 18))
                        .foregroundColor(theme.textMuted)
                }
            }
            
            if isExpanded {
                JSONTreeView(content: content)
            }
        }
        .padding(14)
        .background(theme.surfaceColor)
        .cornerRadius(12)
        .overlay(
            RoundedRectangle(cornerRadius: 12)
                .stroke(theme.surfaceElevated, lineWidth: 1)
        )
    }
}

struct JSONTreeView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        ScrollView([.horizontal, .vertical], showsIndicators: false) {
            Text(formatJSON(content))
                .font(.system(size: 13, design: .monospaced))
                .foregroundColor(theme.textPrimary)
                .textSelection(.enabled)
        }
        .frame(maxHeight: 300)
    }
    
    private func formatJSON(_ json: String) -> String {
        guard let data = json.data(using: .utf8),
              let object = try? JSONSerialization.jsonObject(with: data),
              let prettyData = try? JSONSerialization.data(withJSONObject: object, options: [.prettyPrinted, .sortedKeys]),
              let pretty = String(data: prettyData, encoding: .utf8)
        else {
            return json
        }
        return pretty
    }
}

// MARK: - Code View (for inline code)

struct CodeView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            ForEach(extractCodeBlocks(from: content), id: \.id) { block in
                if block.isCode {
                    CodeBlockView(code: block.content, language: block.language)
                } else {
                    Text(block.content)
                        .font(theme.fontBody)
                        .foregroundColor(theme.textPrimary)
                }
            }
        }
    }
    
    private func extractCodeBlocks(from content: String) -> [CodeBlock] {
        var blocks: [CodeBlock] = []
        var currentText = ""
        var inCode = false
        var currentCode = ""
        var language = ""
        
        for line in content.components(separatedBy: "\n") {
            if line.hasPrefix("```") {
                if inCode {
                    blocks.append(CodeBlock(content: currentCode, isCode: true, language: language))
                    currentCode = ""
                    inCode = false
                    language = ""
                } else {
                    if !currentText.isEmpty {
                        blocks.append(CodeBlock(content: currentText, isCode: false, language: ""))
                        currentText = ""
                    }
                    inCode = true
                    language = String(line.dropFirst(3))
                }
            } else if inCode {
                if !currentCode.isEmpty { currentCode += "\n" }
                currentCode += line
            } else {
                if !currentText.isEmpty { currentText += "\n" }
                currentText += line
            }
        }
        
        if !currentText.isEmpty {
            blocks.append(CodeBlock(content: currentText, isCode: false, language: ""))
        }
        
        return blocks
    }
}

struct CodeBlock: Identifiable {
    let id = UUID()
    let content: String
    let isCode: Bool
    let language: String
}

// MARK: - PDF Viewer

struct PDFViewer: UIViewRepresentable {
    let data: Data
    
    func makeUIView(context: Context) -> WKWebView {
        let webView = WKWebView()
        webView.load(data, mimeType: "application/pdf", characterEncodingName: "utf-8", baseURL: URL(fileURLWithPath: "/"))
        return webView
    }
    
    func updateUIView(_ uiView: WKWebView, context: Context) {}
}

// MARK: - Document Preview

struct DocumentPreview: View {
    @EnvironmentObject var theme: ThemeManager
    let attachment: Attachment
    
    var body: some View {
        NavigationStack {
            VStack {
                switch attachment.type {
                case .pdf:
                    PDFViewer(data: attachment.data)
                case .image:
                    if let uiImage = UIImage(data: attachment.data) {
                        Image(uiImage: uiImage)
                            .resizable()
                            .aspectRatio(contentMode: .fit)
                    }
                default:
                    Text("Preview not available")
                        .foregroundColor(theme.textMuted)
                }
            }
            .navigationTitle(attachment.name)
            .navigationBarTitleDisplayMode(.inline)
        }
    }
}

// MARK: - Sheet Viewer (Spreadsheet)

struct SheetViewer: View {
    @EnvironmentObject var theme: ThemeManager
    let data: [[String]]
    
    var body: some View {
        ScrollView([.horizontal, .vertical]) {
            VStack(alignment: .leading, spacing: 0) {
                ForEach(data.indices, id: \.self) { rowIndex in
                    HStack(spacing: 0) {
                        ForEach(data[rowIndex].indices, id: \.self) { colIndex in
                            Text(data[rowIndex][colIndex])
                                .font(theme.fontBody)
                                .foregroundColor(rowIndex == 0 ? theme.textPrimary : theme.textSecondary)
                                .fontWeight(rowIndex == 0 ? .semibold : .regular)
                                .padding(10)
                                .frame(minWidth: 100)
                                .background(rowIndex == 0 ? theme.surfaceElevated : theme.surfaceColor)
                                .border(theme.surfaceElevated, width: 0.5)
                        }
                    }
                }
            }
        }
        .cornerRadius(theme.radiusMedium)
    }
}
