//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Settings View                                               ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

struct SettingsView: View {
    @EnvironmentObject var authManager: AuthManager
    @EnvironmentObject var chatManager: ChatManager
    @EnvironmentObject var theme: ThemeManager
    @State private var showLogoutConfirmation = false
    
    var body: some View {
        NavigationStack {
            ZStack {
                theme.backgroundGradient
                    .ignoresSafeArea()
                
                List {
                    // User Section
                    Section {
                        if let user = authManager.currentUser {
                            HStack(spacing: 16) {
                                // Avatar
                                ZStack {
                                    Circle()
                                        .fill(theme.primaryGradient)
                                        .frame(width: 60, height: 60)
                                    
                                    Text(user.name.prefix(1).uppercased())
                                        .font(.system(size: 24, weight: .bold))
                                        .foregroundColor(.white)
                                }
                                .shadow(color: theme.accentColor.opacity(0.3), radius: 5)
                                
                                VStack(alignment: .leading, spacing: 4) {
                                    Text(user.name)
                                        .font(.system(size: 18, weight: .semibold))
                                        .foregroundColor(theme.textPrimary)
                                    
                                    Text(user.email)
                                        .font(.system(size: 14))
                                        .foregroundColor(theme.textSecondary)
                                    
                                    if !user.groups.isEmpty {
                                        HStack(spacing: 4) {
                                            Image(systemName: "person.3.fill")
                                                .font(.system(size: 10))
                                            Text(user.groups.joined(separator: ", "))
                                                .font(.system(size: 11))
                                        }
                                        .foregroundColor(theme.textMuted)
                                    }
                                }
                            }
                            .padding(.vertical, 8)
                        }
                    }
                    .listRowBackground(theme.surfaceColor)
                    
                    // AI Settings
                    Section("Intelligence Artificielle") {
                        // Provider Selection
                        NavigationLink {
                            ProviderSettingsView()
                        } label: {
                            SettingsRow(
                                icon: "brain",
                                iconColor: theme.accentColor,
                                title: "Fournisseur AI",
                                value: chatManager.selectedProvider.displayName
                            )
                        }
                        
                        // Voice Settings
                        NavigationLink {
                            VoiceSettingsView()
                        } label: {
                            SettingsRow(
                                icon: "waveform",
                                iconColor: theme.successColor,
                                title: "Paramètres vocaux",
                                value: nil
                            )
                        }
                    }
                    .listRowBackground(theme.surfaceColor)
                    
                    // Infrastructure
                    Section("Infrastructure") {
                        NavigationLink {
                            InfrastructureSettingsView()
                        } label: {
                            SettingsRow(
                                icon: "server.rack",
                                iconColor: theme.infoColor,
                                title: "Clusters Docker",
                                value: "2 clusters"
                            )
                        }
                        
                        NavigationLink {
                            // VPN Settings
                        } label: {
                            SettingsRow(
                                icon: "network",
                                iconColor: theme.warningColor,
                                title: "VPN Headscale",
                                value: "Connecté"
                            )
                        }
                    }
                    .listRowBackground(theme.surfaceColor)
                    
                    // About
                    Section("À propos") {
                        SettingsRow(
                            icon: "info.circle",
                            iconColor: theme.textSecondary,
                            title: "Version",
                            value: "1.0.0"
                        )
                        
                        SettingsRow(
                            icon: "link",
                            iconColor: theme.textSecondary,
                            title: "Backend",
                            value: "assistant-api.internal"
                        )
                    }
                    .listRowBackground(theme.surfaceColor)
                    
                    // Logout
                    Section {
                        Button(action: {
                            showLogoutConfirmation = true
                        }) {
                            HStack {
                                Spacer()
                                HStack(spacing: 8) {
                                    Image(systemName: "rectangle.portrait.and.arrow.right")
                                    Text("Se déconnecter")
                                }
                                .foregroundColor(theme.errorColor)
                                Spacer()
                            }
                        }
                    }
                    .listRowBackground(theme.surfaceColor)
                }
                .scrollContentBackground(.hidden)
            }
            .navigationTitle("Paramètres")
            .bloodAlert(
                isPresented: $showLogoutConfirmation,
                title: "Se déconnecter?",
                message: "Vous serez redirigé vers l'écran de connexion Authentik SSO.",
                icon: "rectangle.portrait.and.arrow.right",
                iconColor: theme.errorColor,
                primaryAction: ModalAction(title: "Se déconnecter", isDestructive: true) {
                    withAnimation {
                        authManager.logout()
                    }
                },
                secondaryAction: ModalAction(title: "Annuler") {}
            )
        }
    }
}

// MARK: - Settings Row

struct SettingsRow: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let iconColor: Color
    let title: String
    let value: String?
    
    var body: some View {
        HStack(spacing: 12) {
            Image(systemName: icon)
                .font(.system(size: 18))
                .foregroundColor(iconColor)
                .frame(width: 28)
            
            Text(title)
                .foregroundColor(theme.textPrimary)
            
            Spacer()
            
            if let value = value {
                Text(value)
                    .foregroundColor(theme.textMuted)
            }
        }
    }
}

// MARK: - Provider Settings

struct ProviderSettingsView: View {
    @EnvironmentObject var chatManager: ChatManager
    @EnvironmentObject var theme: ThemeManager
    
    var body: some View {
        ZStack {
            theme.backgroundGradient
                .ignoresSafeArea()
            
            List {
                Section("Sélectionner le fournisseur") {
                    ForEach(AIProvider.allCases, id: \.self) { provider in
                        Button(action: {
                            chatManager.selectedProvider = provider
                        }) {
                            HStack {
                                Image(systemName: provider.icon)
                                    .foregroundColor(theme.accentColor)
                                    .frame(width: 28)
                                
                                VStack(alignment: .leading, spacing: 2) {
                                    Text(provider.displayName)
                                        .foregroundColor(theme.textPrimary)
                                    
                                    Text(providerDescription(provider))
                                        .font(.caption)
                                        .foregroundColor(theme.textMuted)
                                }
                                
                                Spacer()
                                
                                if chatManager.selectedProvider == provider {
                                    Image(systemName: "checkmark.circle.fill")
                                        .foregroundColor(theme.successColor)
                                }
                            }
                        }
                    }
                }
                .listRowBackground(theme.surfaceColor)
            }
            .scrollContentBackground(.hidden)
        }
        .navigationTitle("Fournisseur AI")
    }
    
    private func providerDescription(_ provider: AIProvider) -> String {
        switch provider {
        case .auto: return "Sélection automatique optimale"
        case .anthropic: return "Claude - Excellent pour le code et l'analyse"
        case .openai: return "GPT-4 - Polyvalent et puissant"
        case .ollama: return "Local - Rapide et privé"
        case .system: return "Commandes infrastructure"
        case .mock: return "Mode démonstration"
        }
    }
}

// MARK: - Voice Settings

struct VoiceSettingsView: View {
    @EnvironmentObject var theme: ThemeManager
    @StateObject private var voiceManager = VoiceManager.shared
    
    @AppStorage("voiceEnabled") private var voiceEnabled = true
    @AppStorage("autoStopEnabled") private var autoStopEnabled = true
    @AppStorage("silenceDuration") private var silenceDuration = 2.0
    @AppStorage("voiceSpeed") private var voiceSpeed = 0.5
    @AppStorage("autoSpeak") private var autoSpeak = false
    
    @State private var isTestingSpeech = false
    
    var body: some View {
        ZStack {
            theme.backgroundGradient
                .ignoresSafeArea()
            
            List {
                // Permission Status
                Section {
                    HStack {
                        Image(systemName: voiceManager.hasPermission ? "checkmark.circle.fill" : "xmark.circle.fill")
                            .foregroundColor(voiceManager.hasPermission ? theme.successColor : theme.errorColor)
                        
                        VStack(alignment: .leading, spacing: 2) {
                            Text("Permissions")
                                .foregroundColor(theme.textPrimary)
                            Text(voiceManager.hasPermission ? "Microphone et reconnaissance vocale autorisés" : "Permissions requises")
                                .font(.caption)
                                .foregroundColor(theme.textMuted)
                        }
                        
                        Spacer()
                        
                        if !voiceManager.hasPermission {
                            Button("Autoriser") {
                                voiceManager.requestPermissions()
                            }
                            .font(.system(size: 14, weight: .medium))
                            .foregroundColor(theme.accentColor)
                        }
                    }
                }
                .listRowBackground(theme.surfaceColor)
                
                Section("Reconnaissance vocale") {
                    Toggle(isOn: $voiceEnabled) {
                        SettingsToggleLabel(
                            icon: "mic.fill",
                            title: "Activer la voix",
                            subtitle: "Commandes et dictée vocale"
                        )
                    }
                    .tint(theme.accentColor)
                    
                    Toggle(isOn: $autoStopEnabled) {
                        SettingsToggleLabel(
                            icon: "timer",
                            title: "Arrêt automatique",
                            subtitle: "Stoppe après détection de silence"
                        )
                    }
                    .tint(theme.accentColor)
                    
                    if autoStopEnabled {
                        VStack(alignment: .leading, spacing: 8) {
                            HStack {
                                Text("Délai de silence")
                                    .foregroundColor(theme.textPrimary)
                                Spacer()
                                Text("\(silenceDuration, specifier: "%.1f")s")
                                    .foregroundColor(theme.textMuted)
                            }
                            
                            Slider(value: $silenceDuration, in: 1.0...5.0, step: 0.5)
                                .tint(theme.accentColor)
                        }
                    }
                }
                .listRowBackground(theme.surfaceColor)
                
                Section("Synthèse vocale") {
                    Toggle(isOn: $autoSpeak) {
                        SettingsToggleLabel(
                            icon: "speaker.wave.3.fill",
                            title: "Lecture automatique",
                            subtitle: "Lit les réponses de l'assistant"
                        )
                    }
                    .tint(theme.accentColor)
                    
                    VStack(alignment: .leading, spacing: 8) {
                        HStack {
                            Text("Vitesse de lecture")
                                .foregroundColor(theme.textPrimary)
                            Spacer()
                            Text("\(voiceSpeed, specifier: "%.1f")x")
                                .foregroundColor(theme.textMuted)
                        }
                        
                        Slider(value: $voiceSpeed, in: 0.3...1.0, step: 0.1)
                            .tint(theme.accentColor)
                    }
                    
                    // Test button
                    Button(action: testSpeech) {
                        HStack {
                            Image(systemName: isTestingSpeech ? "stop.fill" : "play.fill")
                                .foregroundColor(theme.accentColor)
                            Text(isTestingSpeech ? "Arrêter" : "Tester la voix")
                                .foregroundColor(theme.textPrimary)
                            Spacer()
                        }
                    }
                }
                .listRowBackground(theme.surfaceColor)
            }
            .scrollContentBackground(.hidden)
        }
        .navigationTitle("Paramètres vocaux")
    }
    
    private func testSpeech() {
        if isTestingSpeech {
            voiceManager.stopSpeaking()
            isTestingSpeech = false
        } else {
            isTestingSpeech = true
            voiceManager.speak("Bonjour, je suis votre assistant personnel CodinCloud. Comment puis-je vous aider aujourd'hui?", rate: Float(voiceSpeed))
            
            // Auto-reset after speaking
            DispatchQueue.main.asyncAfter(deadline: .now() + 5) {
                isTestingSpeech = false
            }
        }
    }
}

struct SettingsToggleLabel: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let title: String
    let subtitle: String
    
    var body: some View {
        HStack(spacing: 12) {
            Image(systemName: icon)
                .font(.system(size: 16))
                .foregroundColor(theme.accentColor)
                .frame(width: 24)
            
            VStack(alignment: .leading, spacing: 2) {
                Text(title)
                    .foregroundColor(theme.textPrimary)
                Text(subtitle)
                    .font(.caption)
                    .foregroundColor(theme.textMuted)
            }
        }
    }
}

// MARK: - Infrastructure Settings

struct InfrastructureSettingsView: View {
    @EnvironmentObject var theme: ThemeManager
    
    var body: some View {
        ZStack {
            theme.backgroundGradient
                .ignoresSafeArea()
            
            List {
                Section("Cluster Principal") {
                    InfraClusterRow(
                        name: "CodinCloud Swarm",
                        host: "localhost",
                        status: .healthy,
                        services: 24
                    )
                }
                .listRowBackground(theme.surfaceColor)
                
                Section("Cluster Secondaire") {
                    InfraClusterRow(
                        name: "Production Hetzner",
                        host: "157.90.228.104",
                        status: .healthy,
                        services: 12
                    )
                }
                .listRowBackground(theme.surfaceColor)
            }
            .scrollContentBackground(.hidden)
        }
        .navigationTitle("Clusters Docker")
    }
}

struct InfraClusterRow: View {
    @EnvironmentObject var theme: ThemeManager
    let name: String
    let host: String
    let status: ClusterStatus
    let services: Int
    
    enum ClusterStatus {
        case healthy, degraded, unhealthy
        
        var color: Color {
            switch self {
            case .healthy: return ThemeManager.shared.successColor
            case .degraded: return ThemeManager.shared.warningColor
            case .unhealthy: return ThemeManager.shared.errorColor
            }
        }
        
        var icon: String {
            switch self {
            case .healthy: return "checkmark.circle.fill"
            case .degraded: return "exclamationmark.triangle.fill"
            case .unhealthy: return "xmark.circle.fill"
            }
        }
    }
    
    var body: some View {
        HStack(spacing: 12) {
            Image(systemName: "server.rack")
                .font(.system(size: 24))
                .foregroundColor(theme.infoColor)
            
            VStack(alignment: .leading, spacing: 4) {
                Text(name)
                    .font(.system(size: 16, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                
                Text(host)
                    .font(.system(size: 13))
                    .foregroundColor(theme.textMuted)
            }
            
            Spacer()
            
            VStack(alignment: .trailing, spacing: 4) {
                HStack(spacing: 4) {
                    Image(systemName: status.icon)
                        .foregroundColor(status.color)
                    Text("Healthy")
                        .font(.system(size: 12))
                        .foregroundColor(status.color)
                }
                
                Text("\(services) services")
                    .font(.system(size: 11))
                    .foregroundColor(theme.textMuted)
            }
        }
        .padding(.vertical, 4)
    }
}

// MARK: - Previews

#Preview {
    SettingsView()
        .environmentObject(AuthManager.shared)
        .environmentObject(ChatManager.shared)
        .environmentObject(ThemeManager.shared)
}
