//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Universal Content Panel                                      ║
//  ║         Dynamic Multi-Format Display System for Assistant Responses            ║
//  ║         Supports: Markdown, JSON, PDF, Video, Sheet, Graph, Timeline, etc.     ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import Charts
import WebKit
import AVKit
import QuickLook

// MARK: - Content Types

enum PanelContentType: String, Codable {
    case text
    case markdown
    case json
    case table
    case graph
    case timeline
    case kanban
    case pdf
    case video
    case sheet
    case code
    case diagram
    case dashboard
    case form
    case image
    case mixed
    
    var icon: String {
        switch self {
        case .text: return "doc.text"
        case .markdown: return "doc.richtext"
        case .json: return "curlybraces"
        case .table: return "tablecells"
        case .graph: return "chart.pie"
        case .timeline: return "timeline.selection"
        case .kanban: return "rectangle.3.group"
        case .pdf: return "doc.fill"
        case .video: return "play.rectangle"
        case .sheet: return "tablecells.badge.ellipsis"
        case .code: return "chevron.left.forwardslash.chevron.right"
        case .diagram: return "arrow.triangle.branch"
        case .dashboard: return "square.grid.2x2"
        case .form: return "list.bullet.rectangle"
        case .image: return "photo"
        case .mixed: return "square.stack.3d.up"
        }
    }
    
    var displayName: String {
        switch self {
        case .text: return "Texte"
        case .markdown: return "Document"
        case .json: return "JSON"
        case .table: return "Tableau"
        case .graph: return "Graphique"
        case .timeline: return "Timeline"
        case .kanban: return "Kanban"
        case .pdf: return "PDF"
        case .video: return "Vidéo"
        case .sheet: return "Tableur"
        case .code: return "Code"
        case .diagram: return "Diagramme"
        case .dashboard: return "Dashboard"
        case .form: return "Formulaire"
        case .image: return "Image"
        case .mixed: return "Multiple"
        }
    }
}

// MARK: - Panel Attachment Model

struct PanelAttachment: Identifiable {
    let id: String
    let type: PanelContentType
    let title: String
    let data: Any  // Store as Any, not Codable
    let mimeType: String?
    let interactive: Bool
    
    init(id: String, type: PanelContentType, title: String, data: Any, mimeType: String? = nil, interactive: Bool = false) {
        self.id = id
        self.type = type
        self.title = title
        self.data = data
        self.mimeType = mimeType
        self.interactive = interactive
    }
    
    // Helper to get data as dictionary
    var dataAsDict: [String: Any]? {
        return data as? [String: Any]
    }
    
    // Helper to get data as string
    var dataAsString: String? {
        if let str = data as? String {
            return str
        }
        if let dict = data as? [String: Any],
           let jsonData = try? JSONSerialization.data(withJSONObject: dict),
           let str = String(data: jsonData, encoding: .utf8) {
            return str
        }
        return nil
    }
}

// MARK: - Universal Content Panel

struct UniversalContentPanel: View {
    @EnvironmentObject var theme: ThemeManager
    let attachment: PanelAttachment
    @Binding var isExpanded: Bool
    let onDismiss: () -> Void
    
    @State private var showFullScreen = false
    @State private var selectedTab = 0
    
    var body: some View {
        VStack(spacing: 0) {
            // Header
            PanelHeader(
                title: attachment.title,
                type: attachment.type,
                isExpanded: $isExpanded,
                onFullScreen: { showFullScreen = true },
                onDismiss: onDismiss
            )
            
            // Content
            ScrollView {
                contentView
                    .padding()
            }
            .frame(maxHeight: isExpanded ? .infinity : 300)
        }
        .background(theme.surfaceColor)
        .cornerRadius(16)
        .overlay(
            RoundedRectangle(cornerRadius: 16)
                .stroke(theme.primaryColor.opacity(0.3), lineWidth: 1)
        )
        .shadow(color: theme.primaryDark.opacity(0.3), radius: 15)
        .fullScreenCover(isPresented: $showFullScreen) {
            FullScreenPanelView(attachment: attachment)
        }
    }
    
    @ViewBuilder
    var contentView: some View {
        switch attachment.type {
        case .table:
            TableContentView(data: attachment.data)
        case .graph:
            GraphContentView(data: attachment.data)
        case .timeline:
            TimelineContentView(data: attachment.data)
        case .kanban:
            KanbanContentView(data: attachment.data)
        case .dashboard:
            DashboardContentView(data: attachment.data)
        case .json:
            JSONContentView(data: attachment.data)
        case .code:
            CodeContentView(data: attachment.data)
        case .markdown:
            MarkdownContentView(content: String(describing: attachment.data))
        case .pdf:
            PDFContentView(data: attachment.data)
        case .video:
            VideoContentView(data: attachment.data)
        case .image:
            ImageContentView(data: attachment.data)
        case .form:
            FormContentView(data: attachment.data)
        case .diagram:
            DiagramContentView(data: attachment.data)
        case .sheet:
            SheetContentView(data: attachment.data)
        default:
            TextContentView(content: String(describing: attachment.data))
        }
    }
}

// MARK: - Panel Header

struct PanelHeader: View {
    @EnvironmentObject var theme: ThemeManager
    let title: String
    let type: PanelContentType
    @Binding var isExpanded: Bool
    let onFullScreen: () -> Void
    let onDismiss: () -> Void
    
    var body: some View {
        HStack(spacing: 12) {
            // Type Icon
            ZStack {
                Circle()
                    .fill(theme.primaryColor.opacity(0.2))
                    .frame(width: 36, height: 36)
                
                Image(systemName: type.icon)
                    .font(.system(size: 16))
                    .foregroundColor(theme.accentColor)
            }
            
            // Title
            VStack(alignment: .leading, spacing: 2) {
                Text(title)
                    .font(.system(size: 14, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
                    .lineLimit(1)
                
                Text(type.displayName)
                    .font(.system(size: 11))
                    .foregroundColor(theme.textMuted)
            }
            
            Spacer()
            
            // Actions
            HStack(spacing: 8) {
                // Expand/Collapse
                Button(action: { withAnimation(.spring()) { isExpanded.toggle() } }) {
                    Image(systemName: isExpanded ? "chevron.down" : "chevron.up")
                        .font(.system(size: 14))
                        .foregroundColor(theme.textSecondary)
                        .frame(width: 32, height: 32)
                        .background(theme.surfaceElevated)
                        .cornerRadius(8)
                }
                
                // Full Screen
                Button(action: onFullScreen) {
                    Image(systemName: "arrow.up.left.and.arrow.down.right")
                        .font(.system(size: 14))
                        .foregroundColor(theme.textSecondary)
                        .frame(width: 32, height: 32)
                        .background(theme.surfaceElevated)
                        .cornerRadius(8)
                }
                
                // Dismiss
                Button(action: onDismiss) {
                    Image(systemName: "xmark")
                        .font(.system(size: 14))
                        .foregroundColor(theme.textSecondary)
                        .frame(width: 32, height: 32)
                        .background(theme.surfaceElevated)
                        .cornerRadius(8)
                }
            }
        }
        .padding()
        .background(theme.surfaceElevated)
    }
}

// MARK: - Table Content View

struct TableContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var tableData: (headers: [String], rows: [[Any]])? {
        guard let dict = data as? [String: Any],
              let headers = dict["headers"] as? [String],
              let rows = dict["rows"] as? [[Any]] else {
            return nil
        }
        return (headers, rows)
    }
    
    var body: some View {
        if let tableData = tableData {
            VStack(spacing: 0) {
                // Header row
                HStack(spacing: 0) {
                    ForEach(tableData.headers, id: \.self) { header in
                        Text(header)
                            .font(.system(size: 12, weight: .semibold))
                            .foregroundColor(theme.textPrimary)
                            .frame(maxWidth: .infinity, alignment: .leading)
                            .padding(8)
                            .background(theme.primaryColor.opacity(0.2))
                    }
                }
                
                // Data rows
                ForEach(Array(tableData.rows.enumerated()), id: \.offset) { index, row in
                    HStack(spacing: 0) {
                        ForEach(Array(row.enumerated()), id: \.offset) { colIndex, cell in
                            Text(String(describing: cell))
                                .font(.system(size: 12))
                                .foregroundColor(theme.textSecondary)
                                .frame(maxWidth: .infinity, alignment: .leading)
                                .padding(8)
                                .lineLimit(2)
                        }
                    }
                    .background(index % 2 == 0 ? Color.clear : theme.surfaceElevated.opacity(0.5))
                }
            }
            .cornerRadius(8)
            .overlay(
                RoundedRectangle(cornerRadius: 8)
                    .stroke(theme.surfaceElevated, lineWidth: 1)
            )
        } else {
            Text("Format de tableau invalide")
                .foregroundColor(theme.textMuted)
        }
    }
}

// MARK: - Graph Content View

struct GraphContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var chartData: (type: String, labels: [String], values: [Double])? {
        guard let dict = data as? [String: Any],
              let type = dict["type"] as? String,
              let labels = dict["labels"] as? [String],
              let values = dict["values"] as? [Any] else {
            return nil
        }
        let doubleValues = values.compactMap { value -> Double? in
            if let d = value as? Double { return d }
            if let i = value as? Int { return Double(i) }
            return nil
        }
        return (type, labels, doubleValues)
    }
    
    var body: some View {
        if let chartData = chartData {
            VStack(alignment: .leading, spacing: 16) {
                // Chart based on type
                if chartData.type == "pie" {
                    PieChartView(labels: chartData.labels, values: chartData.values)
                } else if chartData.type == "bar" {
                    BarChartView(labels: chartData.labels, values: chartData.values)
                } else {
                    LineChartView(labels: chartData.labels, values: chartData.values)
                }
                
                // Legend
                LazyVGrid(columns: [GridItem(.adaptive(minimum: 100))], spacing: 8) {
                    ForEach(Array(chartData.labels.enumerated()), id: \.offset) { index, label in
                        HStack(spacing: 6) {
                            Circle()
                                .fill(chartColors[index % chartColors.count])
                                .frame(width: 10, height: 10)
                            
                            Text(label)
                                .font(.system(size: 11))
                                .foregroundColor(theme.textSecondary)
                            
                            Spacer()
                            
                            if index < chartData.values.count {
                                Text("\(Int(chartData.values[index]))")
                                    .font(.system(size: 11, weight: .medium))
                                    .foregroundColor(theme.textPrimary)
                            }
                        }
                    }
                }
            }
        } else {
            Text("Format de graphique invalide")
                .foregroundColor(theme.textMuted)
        }
    }
    
    var chartColors: [Color] {
        [
            Color(hex: "DC143C"),
            Color(hex: "FF6B6B"),
            Color(hex: "4ECDC4"),
            Color(hex: "45B7D1"),
            Color(hex: "96CEB4"),
            Color(hex: "FFEAA7")
        ]
    }
}

// MARK: - Pie Chart View

struct PieChartView: View {
    let labels: [String]
    let values: [Double]
    
    var body: some View {
        if #available(iOS 17.0, *) {
            Chart {
                ForEach(Array(zip(labels, values).enumerated()), id: \.offset) { index, item in
                    SectorMark(
                        angle: .value("Value", item.1),
                        innerRadius: .ratio(0.5),
                        angularInset: 2
                    )
                    .foregroundStyle(by: .value("Category", item.0))
                    .cornerRadius(4)
                }
            }
            .frame(height: 200)
        } else {
            // Fallback for older iOS
            SimplePieChart(values: values, labels: labels)
                .frame(height: 200)
        }
    }
}

struct SimplePieChart: View {
    let values: [Double]
    let labels: [String]
    
    var total: Double { values.reduce(0, +) }
    
    var body: some View {
        GeometryReader { geo in
            ZStack {
                ForEach(Array(values.enumerated()), id: \.offset) { index, value in
                    PieSlice(
                        startAngle: startAngle(for: index),
                        endAngle: endAngle(for: index)
                    )
                    .fill(sliceColor(for: index))
                }
            }
        }
    }
    
    func startAngle(for index: Int) -> Angle {
        let sum = values.prefix(index).reduce(0, +)
        return .degrees(sum / total * 360 - 90)
    }
    
    func endAngle(for index: Int) -> Angle {
        let sum = values.prefix(index + 1).reduce(0, +)
        return .degrees(sum / total * 360 - 90)
    }
    
    func sliceColor(for index: Int) -> Color {
        let colors: [Color] = [.red, .orange, .green, .blue, .purple, .pink]
        return colors[index % colors.count]
    }
}

struct PieSlice: Shape {
    var startAngle: Angle
    var endAngle: Angle
    
    func path(in rect: CGRect) -> Path {
        var path = Path()
        let center = CGPoint(x: rect.midX, y: rect.midY)
        let radius = min(rect.width, rect.height) / 2
        
        path.move(to: center)
        path.addArc(center: center, radius: radius, startAngle: startAngle, endAngle: endAngle, clockwise: false)
        path.closeSubpath()
        
        return path
    }
}

// MARK: - Bar Chart View

struct BarChartView: View {
    let labels: [String]
    let values: [Double]
    
    var maxValue: Double { values.max() ?? 1 }
    
    var body: some View {
        HStack(alignment: .bottom, spacing: 8) {
            ForEach(Array(zip(labels, values).enumerated()), id: \.offset) { index, item in
                VStack(spacing: 4) {
                    Text("\(Int(item.1))")
                        .font(.system(size: 10, weight: .medium))
                        .foregroundColor(.white)
                    
                    RoundedRectangle(cornerRadius: 4)
                        .fill(
                            LinearGradient(
                                colors: [Color(hex: "DC143C"), Color(hex: "8B0000")],
                                startPoint: .top,
                                endPoint: .bottom
                            )
                        )
                        .frame(height: CGFloat(item.1 / maxValue) * 150)
                    
                    Text(item.0)
                        .font(.system(size: 9))
                        .foregroundColor(.gray)
                        .lineLimit(1)
                }
                .frame(maxWidth: .infinity)
            }
        }
        .frame(height: 200)
    }
}

// MARK: - Line Chart View

struct LineChartView: View {
    let labels: [String]
    let values: [Double]
    
    var body: some View {
        if #available(iOS 17.0, *) {
            Chart {
                ForEach(Array(zip(labels, values).enumerated()), id: \.offset) { index, item in
                    LineMark(
                        x: .value("Label", item.0),
                        y: .value("Value", item.1)
                    )
                    .foregroundStyle(Color(hex: "DC143C"))
                    
                    PointMark(
                        x: .value("Label", item.0),
                        y: .value("Value", item.1)
                    )
                    .foregroundStyle(Color(hex: "FF6B6B"))
                }
            }
            .frame(height: 200)
        } else {
            Text("Line chart requires iOS 17+")
                .foregroundColor(.gray)
        }
    }
}

// MARK: - Timeline Content View

struct TimelineContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var steps: [(id: String, title: String, status: String)]? {
        guard let dict = data as? [String: Any],
              let stepsData = dict["steps"] as? [[String: Any]] else {
            return nil
        }
        return stepsData.compactMap { step in
            guard let id = step["id"] as? String,
                  let title = step["title"] as? String,
                  let status = step["status"] as? String else {
                return nil
            }
            return (id, title, status)
        }
    }
    
    var body: some View {
        if let steps = steps {
            VStack(spacing: 0) {
                ForEach(Array(steps.enumerated()), id: \.element.id) { index, step in
                    HStack(alignment: .top, spacing: 16) {
                        // Timeline indicator
                        VStack(spacing: 0) {
                            Circle()
                                .fill(statusColor(step.status))
                                .frame(width: 16, height: 16)
                                .overlay(
                                    Image(systemName: statusIcon(step.status))
                                        .font(.system(size: 8))
                                        .foregroundColor(.white)
                                )
                            
                            if index < steps.count - 1 {
                                Rectangle()
                                    .fill(theme.surfaceElevated)
                                    .frame(width: 2)
                                    .frame(maxHeight: .infinity)
                            }
                        }
                        .frame(width: 16)
                        
                        // Step content
                        VStack(alignment: .leading, spacing: 4) {
                            Text("Étape \(index + 1)")
                                .font(.system(size: 11, weight: .medium))
                                .foregroundColor(theme.textMuted)
                            
                            Text(step.title)
                                .font(.system(size: 14, weight: .medium))
                                .foregroundColor(theme.textPrimary)
                            
                            Text(statusLabel(step.status))
                                .font(.system(size: 11))
                                .foregroundColor(statusColor(step.status))
                        }
                        .padding(.bottom, 20)
                        
                        Spacer()
                    }
                }
            }
        } else {
            Text("Format de timeline invalide")
                .foregroundColor(theme.textMuted)
        }
    }
    
    func statusColor(_ status: String) -> Color {
        switch status.lowercased() {
        case "completed": return .green
        case "in_progress": return .orange
        case "failed": return .red
        default: return .gray
        }
    }
    
    func statusIcon(_ status: String) -> String {
        switch status.lowercased() {
        case "completed": return "checkmark"
        case "in_progress": return "arrow.clockwise"
        case "failed": return "xmark"
        default: return "circle"
        }
    }
    
    func statusLabel(_ status: String) -> String {
        switch status.lowercased() {
        case "completed": return "✅ Complété"
        case "in_progress": return "🔄 En cours"
        case "failed": return "❌ Échoué"
        case "pending": return "⏳ En attente"
        default: return status
        }
    }
}

// MARK: - Dashboard Content View (Flexible Format)

struct DashboardContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    // Parse multiple formats
    var parsedData: DashboardData {
        // Format 1: { "widgets": [...] }
        if let dict = data as? [String: Any],
           let widgets = dict["widgets"] as? [[String: Any]] {
            return DashboardData(widgets: widgets.map { DashboardWidgetData(from: $0) })
        }
        
        // Format 2: Direct array of widgets
        if let widgets = data as? [[String: Any]] {
            return DashboardData(widgets: widgets.map { DashboardWidgetData(from: $0) })
        }
        
        // Format 3: Dictionary with key-value pairs (services, metrics, etc.)
        if let dict = data as? [String: Any] {
            var widgets: [DashboardWidgetData] = []
            
            // Check for services array
            if let services = dict["services"] as? [[String: Any]] {
                for service in services {
                    widgets.append(DashboardWidgetData(
                        type: "service",
                        label: service["name"] as? String ?? "Service",
                        value: service["status"] as? String ?? service["state"] as? String ?? "unknown",
                        icon: "server.rack",
                        color: (service["status"] as? String ?? "") == "running" ? "green" : "red"
                    ))
                }
            }
            
            // Check for metrics
            if let metrics = dict["metrics"] as? [String: Any] {
                for (key, value) in metrics {
                    widgets.append(DashboardWidgetData(
                        type: "metric",
                        label: key.capitalized.replacingOccurrences(of: "_", with: " "),
                        value: String(describing: value),
                        icon: "chart.bar.fill",
                        color: "blue"
                    ))
                }
            }
            
            // Check for status fields
            if let status = dict["status"] as? String {
                widgets.append(DashboardWidgetData(
                    type: "status",
                    label: "Statut",
                    value: status,
                    icon: status == "healthy" || status == "running" ? "checkmark.circle.fill" : "exclamationmark.triangle.fill",
                    color: status == "healthy" || status == "running" ? "green" : "orange"
                ))
            }
            
            // Generic key-value extraction for remaining fields
            for (key, value) in dict {
                if !["widgets", "services", "metrics", "status"].contains(key) {
                    if let strValue = value as? String {
                        widgets.append(DashboardWidgetData(
                            type: "info",
                            label: formatLabel(key),
                            value: strValue,
                            icon: "info.circle.fill",
                            color: "blue"
                        ))
                    } else if let numValue = value as? Int {
                        widgets.append(DashboardWidgetData(
                            type: "metric",
                            label: formatLabel(key),
                            value: "\(numValue)",
                            icon: "number.circle.fill",
                            color: "purple"
                        ))
                    } else if let boolValue = value as? Bool {
                        widgets.append(DashboardWidgetData(
                            type: "status",
                            label: formatLabel(key),
                            value: boolValue ? "Oui" : "Non",
                            icon: boolValue ? "checkmark.circle.fill" : "xmark.circle.fill",
                            color: boolValue ? "green" : "red"
                        ))
                    }
                }
            }
            
            if !widgets.isEmpty {
                return DashboardData(widgets: widgets)
            }
        }
        
        // Format 4: String content - create a single info widget
        if let stringData = data as? String {
            return DashboardData(widgets: [
                DashboardWidgetData(type: "text", label: "Information", value: stringData, icon: "doc.text.fill", color: "blue")
            ])
        }
        
        // Fallback - show raw data
        return DashboardData(widgets: [
            DashboardWidgetData(type: "raw", label: "Données", value: String(describing: data), icon: "doc.fill", color: "gray")
        ])
    }
    
    private func formatLabel(_ key: String) -> String {
        key.replacingOccurrences(of: "_", with: " ")
            .replacingOccurrences(of: "-", with: " ")
            .capitalized
    }
    
    var body: some View {
        let dashboard = parsedData
        
        ScrollView {
            if dashboard.widgets.isEmpty {
                EmptyDashboardView()
            } else {
                LazyVGrid(columns: [
                    GridItem(.flexible(), spacing: 12),
                    GridItem(.flexible(), spacing: 12)
                ], spacing: 12) {
                    ForEach(Array(dashboard.widgets.enumerated()), id: \.offset) { index, widget in
                        DashboardWidgetView(widget: widget)
                            .transition(.scale.combined(with: .opacity))
                    }
                }
                .padding(.vertical, 8)
            }
        }
    }
}

// MARK: - Dashboard Data Models

struct DashboardData {
    let widgets: [DashboardWidgetData]
}

struct DashboardWidgetData {
    let type: String
    let label: String
    let value: String
    let icon: String
    let color: String
    var trend: String? = nil
    var subtitle: String? = nil
    
    init(type: String, label: String, value: String, icon: String, color: String, trend: String? = nil, subtitle: String? = nil) {
        self.type = type
        self.label = label
        self.value = value
        self.icon = icon
        self.color = color
        self.trend = trend
        self.subtitle = subtitle
    }
    
    init(from dict: [String: Any]) {
        self.type = dict["type"] as? String ?? "metric"
        self.label = dict["label"] as? String ?? dict["name"] as? String ?? ""
        self.value = dict["value"] as? String ?? String(describing: dict["value"] ?? "")
        self.icon = dict["icon"] as? String ?? "square.fill"
        self.color = dict["color"] as? String ?? "blue"
        self.trend = dict["trend"] as? String
        self.subtitle = dict["subtitle"] as? String ?? dict["description"] as? String
    }
}

// MARK: - Dashboard Widget View

struct DashboardWidgetView: View {
    @EnvironmentObject var theme: ThemeManager
    let widget: DashboardWidgetData
    
    var widgetColor: Color {
        switch widget.color.lowercased() {
        case "green": return .green
        case "red": return .red
        case "orange": return .orange
        case "yellow": return .yellow
        case "blue": return .blue
        case "purple": return .purple
        case "pink": return .pink
        case "cyan": return .cyan
        default: return theme.accentColor
        }
    }
    
    var body: some View {
        VStack(alignment: .leading, spacing: 10) {
            // Header
            HStack(spacing: 8) {
                ZStack {
                    Circle()
                        .fill(widgetColor.opacity(0.2))
                        .frame(width: 32, height: 32)
                    
                    Image(systemName: widget.icon)
                        .font(.system(size: 14))
                        .foregroundColor(widgetColor)
                }
                
                Text(widget.label)
                    .font(.system(size: 12, weight: .medium))
                    .foregroundColor(theme.textSecondary)
                    .lineLimit(1)
                
                Spacer()
                
                // Trend indicator
                if let trend = widget.trend {
                    HStack(spacing: 2) {
                        Image(systemName: trend.hasPrefix("+") ? "arrow.up.right" : "arrow.down.right")
                            .font(.system(size: 10))
                        Text(trend)
                            .font(.system(size: 10, weight: .medium))
                    }
                    .foregroundColor(trend.hasPrefix("+") ? .green : .red)
                }
            }
            
            // Value
            Text(widget.value)
                .font(.system(size: widget.value.count > 15 ? 16 : 22, weight: .bold))
                .foregroundStyle(
                    LinearGradient(
                        colors: [widgetColor, widgetColor.opacity(0.7)],
                        startPoint: .leading,
                        endPoint: .trailing
                    )
                )
                .lineLimit(2)
                .minimumScaleFactor(0.7)
            
            // Subtitle
            if let subtitle = widget.subtitle {
                Text(subtitle)
                    .font(.system(size: 11))
                    .foregroundColor(theme.textMuted)
                    .lineLimit(2)
            }
        }
        .frame(maxWidth: .infinity, alignment: .leading)
        .padding(14)
        .background(
            RoundedRectangle(cornerRadius: 14)
                .fill(theme.surfaceColor)
                .shadow(color: widgetColor.opacity(0.1), radius: 8, y: 4)
        )
        .overlay(
            RoundedRectangle(cornerRadius: 14)
                .stroke(widgetColor.opacity(0.2), lineWidth: 1)
        )
    }
}

// MARK: - Empty Dashboard View

struct EmptyDashboardView: View {
    @EnvironmentObject var theme: ThemeManager
    
    var body: some View {
        VStack(spacing: 16) {
            Image(systemName: "chart.bar.doc.horizontal")
                .font(.system(size: 48))
                .foregroundColor(theme.textMuted)
            
            Text("Aucune donnée à afficher")
                .font(.system(size: 16, weight: .medium))
                .foregroundColor(theme.textSecondary)
            
            Text("Le dashboard sera mis à jour avec les prochaines données")
                .font(.system(size: 13))
                .foregroundColor(theme.textMuted)
                .multilineTextAlignment(.center)
        }
        .padding(40)
    }
}

// MARK: - JSON Content View

struct JSONContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    @State private var isExpanded = true
    
    var jsonString: String {
        if let dict = data as? [String: Any] {
            return prettyPrint(dict)
        } else if let array = data as? [Any] {
            return prettyPrint(array)
        } else {
            return String(describing: data)
        }
    }
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            HStack {
                Text("JSON")
                    .font(.system(size: 12, weight: .semibold))
                    .foregroundColor(theme.accentColor)
                
                Spacer()
                
                Button(action: {
                    UIPasteboard.general.string = jsonString
                }) {
                    Image(systemName: "doc.on.doc")
                        .font(.system(size: 14))
                        .foregroundColor(theme.textSecondary)
                }
            }
            
            ScrollView(.horizontal, showsIndicators: false) {
                Text(jsonString)
                    .font(.system(.caption, design: .monospaced))
                    .foregroundColor(theme.textPrimary)
            }
        }
        .padding()
        .background(Color.black.opacity(0.3))
        .cornerRadius(8)
    }
    
    func prettyPrint(_ value: Any, indent: Int = 0) -> String {
        let indentString = String(repeating: "  ", count: indent)
        
        if let dict = value as? [String: Any] {
            var result = "{\n"
            for (key, val) in dict {
                result += "\(indentString)  \"\(key)\": \(prettyPrint(val, indent: indent + 1)),\n"
            }
            result += "\(indentString)}"
            return result
        } else if let array = value as? [Any] {
            var result = "[\n"
            for item in array {
                result += "\(indentString)  \(prettyPrint(item, indent: indent + 1)),\n"
            }
            result += "\(indentString)]"
            return result
        } else if let string = value as? String {
            return "\"\(string)\""
        } else {
            return String(describing: value)
        }
    }
}

// MARK: - Code Content View

struct CodeContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var codeString: String { String(describing: data) }
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            HStack {
                Text("Code")
                    .font(.system(size: 12, weight: .semibold))
                    .foregroundColor(theme.accentColor)
                
                Spacer()
                
                Button(action: {
                    UIPasteboard.general.string = codeString
                }) {
                    Label("Copier", systemImage: "doc.on.doc")
                        .font(.system(size: 12))
                        .foregroundColor(theme.textSecondary)
                }
            }
            
            ScrollView {
                Text(codeString)
                    .font(.system(.caption, design: .monospaced))
                    .foregroundColor(.green)
            }
        }
        .padding()
        .background(Color.black.opacity(0.5))
        .cornerRadius(8)
    }
}

// MARK: - Markdown Content View (Enhanced)

struct MarkdownContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        Text(LocalizedStringKey(content))
            .font(.system(size: 14))
            .foregroundColor(theme.textPrimary)
    }
}

// MARK: - Text Content View

struct TextContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        Text(content)
            .font(.system(size: 14))
            .foregroundColor(theme.textPrimary)
    }
}

// MARK: - Kanban Content View

struct KanbanContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var body: some View {
        Text("Vue Kanban - Coming soon")
            .foregroundColor(theme.textMuted)
    }
}

// MARK: - PDF Content View

struct PDFContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var body: some View {
        VStack(spacing: 12) {
            Image(systemName: "doc.fill")
                .font(.system(size: 48))
                .foregroundColor(theme.accentColor)
            
            Text("Document PDF")
                .font(.system(size: 14, weight: .medium))
                .foregroundColor(theme.textPrimary)
            
            Button(action: {}) {
                Label("Ouvrir", systemImage: "arrow.up.right.square")
                    .font(.system(size: 14))
            }
            .buttonStyle(.borderedProminent)
            .tint(theme.primaryColor)
        }
        .frame(maxWidth: .infinity)
        .padding()
    }
}

// MARK: - Video Content View

struct VideoContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var body: some View {
        VStack(spacing: 12) {
            Image(systemName: "play.rectangle.fill")
                .font(.system(size: 48))
                .foregroundColor(theme.accentColor)
            
            Text("Contenu Vidéo")
                .font(.system(size: 14, weight: .medium))
                .foregroundColor(theme.textPrimary)
            
            Button(action: {}) {
                Label("Lire", systemImage: "play.fill")
                    .font(.system(size: 14))
            }
            .buttonStyle(.borderedProminent)
            .tint(theme.primaryColor)
        }
        .frame(maxWidth: .infinity)
        .padding()
    }
}

// MARK: - Image Content View

struct ImageContentView: View {
    let data: Any
    
    var body: some View {
        if let urlString = data as? String,
           let url = URL(string: urlString) {
            AsyncImage(url: url) { image in
                image
                    .resizable()
                    .aspectRatio(contentMode: .fit)
            } placeholder: {
                ProgressView()
            }
        } else {
            Image(systemName: "photo")
                .font(.system(size: 48))
                .foregroundColor(.gray)
        }
    }
}

// MARK: - Form Content View

struct FormContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var body: some View {
        Text("Formulaire interactif - Coming soon")
            .foregroundColor(theme.textMuted)
    }
}

// MARK: - Diagram Content View

struct DiagramContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var body: some View {
        Text("Diagramme - Coming soon")
            .foregroundColor(theme.textMuted)
    }
}

// MARK: - Sheet Content View

struct SheetContentView: View {
    @EnvironmentObject var theme: ThemeManager
    let data: Any
    
    var body: some View {
        // Reuse table view for spreadsheet
        TableContentView(data: data)
    }
}

// MARK: - Full Screen Panel View with AYMENOS Branding

struct FullScreenPanelView: View {
    @EnvironmentObject var theme: ThemeManager
    @Environment(\.dismiss) var dismiss
    let attachment: PanelAttachment
    
    var body: some View {
        NavigationStack {
            ZStack {
                theme.backgroundGradient.ignoresSafeArea()
                
                ScrollView {
                    UniversalContentPanel(
                        attachment: attachment,
                        isExpanded: .constant(true),
                        onDismiss: { dismiss() }
                    )
                    .padding()
                }
            }
            .navigationTitle(attachment.title)
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .topBarLeading) {
                    HStack(spacing: 8) {
                        Image("AymenosLogo")
                            .resizable()
                            .aspectRatio(contentMode: .fit)
                            .frame(width: 24, height: 24)
                            .clipShape(Circle())
                        
                        Text("AYMENOS")
                            .font(.system(size: 14, weight: .bold, design: .serif))
                            .foregroundColor(theme.goldAccent)
                    }
                }
                
                ToolbarItem(placement: .topBarTrailing) {
                    Button {
                        dismiss()
                    } label: {
                        Image(systemName: "xmark.circle.fill")
                            .font(.system(size: 24))
                            .foregroundStyle(theme.textMuted, theme.surfaceColor)
                    }
                }
            }
        }
        .preferredColorScheme(.dark)
    }
}

// MARK: - Sophisticated Panel Container for Chat

struct ChatPanelContainer: View {
    @EnvironmentObject var theme: ThemeManager
    let attachments: [PanelAttachment]
    @Binding var isVisible: Bool
    @State private var expandedPanels: Set<String> = []
    @State private var selectedIndex = 0
    @State private var dragOffset: CGFloat = 0
    @State private var showFullScreen = false
    
    var body: some View {
        if !attachments.isEmpty && isVisible {
            VStack(spacing: 0) {
                // AYMENOS Header with drag handle
                panelHeader
                
                // Tab selector if multiple attachments
                if attachments.count > 1 {
                    tabSelector
                }
                
                // Current panel
                if selectedIndex < attachments.count {
                    UniversalContentPanel(
                        attachment: attachments[selectedIndex],
                        isExpanded: Binding(
                            get: { expandedPanels.contains(attachments[selectedIndex].id) },
                            set: { newValue in
                                if newValue {
                                    expandedPanels.insert(attachments[selectedIndex].id)
                                } else {
                                    expandedPanels.remove(attachments[selectedIndex].id)
                                }
                            }
                        ),
                        onDismiss: {
                            withAnimation(.spring()) {
                                isVisible = false
                            }
                        }
                    )
                }
            }
            .background(
                RoundedRectangle(cornerRadius: 20, style: .continuous)
                    .fill(theme.surfaceColor)
                    .overlay(
                        RoundedRectangle(cornerRadius: 20, style: .continuous)
                            .stroke(
                                LinearGradient(
                                    colors: [theme.goldAccent.opacity(0.4), theme.goldAccent.opacity(0.1)],
                                    startPoint: .top,
                                    endPoint: .bottom
                                ),
                                lineWidth: 1
                            )
                    )
                    .shadow(color: theme.goldAccent.opacity(0.2), radius: 15, y: -5)
            )
            .offset(y: dragOffset)
            .gesture(
                DragGesture()
                    .onChanged { value in
                        if value.translation.height > 0 {
                            dragOffset = value.translation.height
                        }
                    }
                    .onEnded { value in
                        if value.translation.height > 100 {
                            withAnimation(.spring()) {
                                isVisible = false
                            }
                        }
                        withAnimation(.spring()) {
                            dragOffset = 0
                        }
                    }
            )
            .transition(.move(edge: .bottom).combined(with: .opacity))
            .fullScreenCover(isPresented: $showFullScreen) {
                if selectedIndex < attachments.count {
                    FullScreenPanelView(attachment: attachments[selectedIndex])
                }
            }
        }
    }
    
    // AYMENOS Branded Header
    private var panelHeader: some View {
        VStack(spacing: 8) {
            // Drag handle
            RoundedRectangle(cornerRadius: 2)
                .fill(theme.goldAccent.opacity(0.5))
                .frame(width: 40, height: 4)
                .padding(.top, 8)
            
            HStack {
                // AYMENOS mini logo
                Image("AymenosLogo")
                    .resizable()
                    .aspectRatio(contentMode: .fit)
                    .frame(width: 28, height: 28)
                    .clipShape(Circle())
                    .overlay(
                        Circle()
                            .stroke(theme.goldAccent.opacity(0.5), lineWidth: 1)
                    )
                
                VStack(alignment: .leading, spacing: 2) {
                    Text("Panel de Collaboration")
                        .font(.system(size: 14, weight: .bold))
                        .foregroundColor(theme.textPrimary)
                    
                    HStack(spacing: 4) {
                        Circle()
                            .fill(theme.successColor)
                            .frame(width: 6, height: 6)
                        
                        Text("AYMENOS • \(attachments.count) résultat\(attachments.count > 1 ? "s" : "")")
                            .font(.system(size: 10))
                            .foregroundColor(theme.textMuted)
                    }
                }
                
                Spacer()
                
                // Actions
                HStack(spacing: 12) {
                    // Fullscreen button
                    Button {
                        showFullScreen = true
                    } label: {
                        Image(systemName: "arrow.up.left.and.arrow.down.right")
                            .font(.system(size: 14))
                            .foregroundColor(theme.goldAccent)
                    }
                    
                    // Close button
                    Button {
                        withAnimation(.spring()) {
                            isVisible = false
                        }
                    } label: {
                        Image(systemName: "xmark.circle.fill")
                            .font(.system(size: 20))
                            .foregroundStyle(theme.textMuted, theme.surfaceElevated)
                    }
                }
            }
            .padding(.horizontal, 16)
            .padding(.bottom, 8)
        }
        .background(
            LinearGradient(
                colors: [theme.surfaceElevated, theme.surfaceColor],
                startPoint: .top,
                endPoint: .bottom
            )
        )
    }
    
    // Tab selector with gold accents
    private var tabSelector: some View {
        ScrollView(.horizontal, showsIndicators: false) {
            HStack(spacing: 8) {
                ForEach(Array(attachments.enumerated()), id: \.element.id) { index, attachment in
                    PanelTabButton(
                        title: attachment.title,
                        type: attachment.type,
                        isSelected: selectedIndex == index,
                        action: { 
                            withAnimation(.spring(response: 0.3)) {
                                selectedIndex = index 
                            }
                        }
                    )
                }
            }
            .padding(.horizontal)
        }
        .padding(.vertical, 8)
        .background(theme.surfaceColor.opacity(0.5))
    }
}

// MARK: - Panel Tab Button

struct PanelTabButton: View {
    @EnvironmentObject var theme: ThemeManager
    let title: String
    let type: PanelContentType
    let isSelected: Bool
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            HStack(spacing: 6) {
                Image(systemName: type.icon)
                    .font(.system(size: 12))
                
                Text(title)
                    .font(.system(size: 12, weight: .medium))
                    .lineLimit(1)
            }
            .foregroundColor(isSelected ? .white : theme.textSecondary)
            .padding(.horizontal, 12)
            .padding(.vertical, 8)
            .background(
                isSelected 
                ? AnyShapeStyle(LinearGradient(
                    colors: [theme.goldAccent, theme.goldDark],
                    startPoint: .leading,
                    endPoint: .trailing
                  ))
                : AnyShapeStyle(theme.surfaceElevated)
            )
            .cornerRadius(8)
            .overlay(
                RoundedRectangle(cornerRadius: 8)
                    .stroke(isSelected ? theme.goldAccent.opacity(0.5) : Color.clear, lineWidth: 1)
            )
            .scaleEffect(isSelected ? 1.02 : 1.0)
            .animation(.spring(response: 0.3), value: isSelected)
        }
    }
}

// Color.init(hex:) is defined in Theme.swift
