//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Unified Collaboration Panel                                  ║
//  ║         Real-Time AI Orchestration + History + Live Mutations                  ║
//  ║         Single Sophisticated Panel for All Collaboration Needs                 ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import Combine

// MARK: - Unified Panel State

class UnifiedPanelManager: ObservableObject {
    static let shared = UnifiedPanelManager()
    
    // Panel visibility
    @Published var isVisible = false
    @Published var isExpanded = false
    @Published var isPinned = false
    
    // Current session
    @Published var currentSession: AISession?
    @Published var sessionHistory: [AISession] = []
    
    // Live state
    @Published var isProcessing = false
    @Published var progress: Double = 0
    @Published var currentStepIndex: Int = 0
    
    // Selected step for detail view
    @Published var selectedStepId: String?
    
    private var cancellables = Set<AnyCancellable>()
    
    private init() {
        // Auto-show when processing starts
        $isProcessing
            .sink { [weak self] processing in
                if processing && !(self?.isPinned ?? false) {
                    withAnimation(.spring(response: 0.4)) {
                        self?.isVisible = true
                    }
                }
            }
            .store(in: &cancellables)
    }
    
    // MARK: - Session Management
    
    func startSession(query: String, requestId: String = UUID().uuidString) {
        let session = AISession(requestId: requestId, query: query)
        currentSession = session
        isProcessing = true
        progress = 0
        currentStepIndex = 0
        selectedStepId = nil
        
        withAnimation(.spring(response: 0.4)) {
            isVisible = true
        }
    }
    
    func addStep(_ step: AIStep) {
        guard var session = currentSession else { return }
        
        // Mark previous in-progress step as completed
        if let lastIndex = session.steps.lastIndex(where: { $0.status == .inProgress }) {
            session.steps[lastIndex].status = .completed
            session.steps[lastIndex].endTime = Date()
        }
        
        session.steps.append(step)
        currentSession = session
        currentStepIndex = session.steps.count - 1
        
        // Update progress
        let expectedSteps = max(session.steps.count, 5)
        progress = min(Double(session.steps.count) / Double(expectedSteps), 0.95)
        
        // Auto-select new step
        selectedStepId = step.id
    }
    
    func updateStep(id: String, status: AIStep.Status, output: String? = nil, error: String? = nil) {
        guard var session = currentSession,
              let index = session.steps.firstIndex(where: { $0.id == id }) else { return }
        
        session.steps[index].status = status
        session.steps[index].output = output ?? session.steps[index].output
        session.steps[index].error = error
        session.steps[index].endTime = status == .completed || status == .failed ? Date() : nil
        currentSession = session
    }
    
    func endSession(output: String? = nil, success: Bool = true) {
        guard var session = currentSession else { return }
        
        // Complete all remaining in-progress steps
        for i in session.steps.indices where session.steps[i].status == .inProgress {
            session.steps[i].status = success ? .completed : .failed
            session.steps[i].endTime = Date()
        }
        
        session.status = success ? .completed : .failed
        session.endTime = Date()
        session.finalOutput = output
        
        currentSession = session
        progress = 1.0
        isProcessing = false
        
        // Add to history
        sessionHistory.insert(session, at: 0)
        if sessionHistory.count > 50 {
            sessionHistory = Array(sessionHistory.prefix(50))
        }
        
        // Hide panel after delay if not pinned
        if !isPinned {
            DispatchQueue.main.asyncAfter(deadline: .now() + 3) { [weak self] in
                guard self?.isProcessing == false else { return }
                withAnimation(.spring(response: 0.4)) {
                    self?.isVisible = false
                }
            }
        }
    }
    
    func toggleVisibility() {
        withAnimation(.spring(response: 0.4)) {
            isVisible.toggle()
        }
    }
    
    func clearHistory() {
        sessionHistory.removeAll()
    }
}

// MARK: - AI Session Model

struct AISession: Identifiable {
    let id: String
    let requestId: String
    let query: String
    let startTime: Date
    var endTime: Date?
    var status: Status = .active
    var steps: [AIStep] = []
    var finalOutput: String?
    
    enum Status: String {
        case active, completed, failed, cancelled
    }
    
    init(requestId: String, query: String) {
        self.id = UUID().uuidString
        self.requestId = requestId
        self.query = query
        self.startTime = Date()
    }
    
    var duration: TimeInterval {
        (endTime ?? Date()).timeIntervalSince(startTime)
    }
    
    var completedStepsCount: Int {
        steps.filter { $0.status == .completed }.count
    }
}

// MARK: - AI Step Model

struct AIStep: Identifiable {
    let id: String
    let type: StepType
    let title: String
    let description: String
    var agent: String?
    var tool: String?
    var parameters: [String: Any]?
    var output: String?
    var error: String?
    let startTime: Date
    var endTime: Date?
    var status: Status = .inProgress
    var subSteps: [AIStep] = []
    
    enum Status: String {
        case pending, inProgress, completed, failed, skipped
    }
    
    enum StepType: String, CaseIterable {
        // Analysis
        case analyzing, routing, planning, decomposing
        // Execution
        case executing, toolExecution, shellCommand, apiCall
        // Data
        case webScraping, webSearch, databaseQuery, documentRead, documentWrite
        // Code
        case codeGeneration, codeExecution
        // Recovery
        case fallback, retry, validation, reformulation
        // Collaboration
        case collaboration, completion, error
        
        var icon: String {
            switch self {
            case .analyzing: return "magnifyingglass.circle.fill"
            case .routing: return "arrow.triangle.branch"
            case .planning: return "map.fill"
            case .decomposing: return "square.split.2x2.fill"
            case .executing: return "bolt.fill"
            case .toolExecution: return "wrench.and.screwdriver.fill"
            case .shellCommand: return "terminal.fill"
            case .apiCall: return "network"
            case .webScraping: return "globe"
            case .webSearch: return "magnifyingglass"
            case .databaseQuery: return "cylinder.fill"
            case .documentRead: return "doc.text.magnifyingglass"
            case .documentWrite: return "doc.badge.plus"
            case .codeGeneration: return "chevron.left.forwardslash.chevron.right"
            case .codeExecution: return "play.fill"
            case .fallback: return "arrow.uturn.backward.circle.fill"
            case .retry: return "arrow.clockwise"
            case .validation: return "checkmark.shield.fill"
            case .reformulation: return "sparkles"
            case .collaboration: return "person.3.fill"
            case .completion: return "checkmark.circle.fill"
            case .error: return "xmark.octagon.fill"
            }
        }
        
        var color: Color {
            switch self {
            case .analyzing: return .blue
            case .routing: return .purple
            case .planning: return .indigo
            case .decomposing: return Color(red: 0.5, green: 0.3, blue: 0.8)
            case .executing: return .orange
            case .toolExecution: return .cyan
            case .shellCommand: return Color(red: 0.2, green: 0.8, blue: 0.4)
            case .apiCall: return .teal
            case .webScraping: return .mint
            case .webSearch: return .yellow
            case .databaseQuery: return Color(red: 0.6, green: 0.4, blue: 0.8)
            case .documentRead: return .blue.opacity(0.8)
            case .documentWrite: return .green.opacity(0.8)
            case .codeGeneration: return Color(red: 0.9, green: 0.5, blue: 0.2)
            case .codeExecution: return .green
            case .fallback: return .orange.opacity(0.8)
            case .retry: return .yellow.opacity(0.8)
            case .validation: return .mint
            case .reformulation: return .yellow
            case .collaboration: return .cyan
            case .completion: return .green
            case .error: return .red
            }
        }
        
        var displayName: String {
            switch self {
            case .analyzing: return "Analyse"
            case .routing: return "Routage"
            case .planning: return "Planification"
            case .decomposing: return "Décomposition"
            case .executing: return "Exécution"
            case .toolExecution: return "Outil"
            case .shellCommand: return "Commande Shell"
            case .apiCall: return "Appel API"
            case .webScraping: return "Scraping Web"
            case .webSearch: return "Recherche Web"
            case .databaseQuery: return "Requête BDD"
            case .documentRead: return "Lecture Doc"
            case .documentWrite: return "Écriture Doc"
            case .codeGeneration: return "Génération Code"
            case .codeExecution: return "Exécution Code"
            case .fallback: return "Fallback"
            case .retry: return "Nouvelle Tentative"
            case .validation: return "Validation"
            case .reformulation: return "Reformulation"
            case .collaboration: return "Collaboration"
            case .completion: return "Terminé"
            case .error: return "Erreur"
            }
        }
    }
    
    init(
        type: StepType,
        title: String,
        description: String,
        agent: String? = nil,
        tool: String? = nil
    ) {
        self.id = UUID().uuidString
        self.type = type
        self.title = title
        self.description = description
        self.agent = agent
        self.tool = tool
        self.startTime = Date()
    }
    
    var duration: TimeInterval? {
        guard let endTime = endTime else { return nil }
        return endTime.timeIntervalSince(startTime)
    }
}

// MARK: - Unified Collaboration Panel View

struct UnifiedCollaborationPanel: View {
    @EnvironmentObject var theme: ThemeManager
    @StateObject private var panelManager = UnifiedPanelManager.shared
    @ObservedObject var wsManager: WebSocketManager
    
    @State private var showHistory = false
    @State private var selectedHistorySession: AISession?
    @State private var dragOffset: CGFloat = 0
    @State private var pulseAnimation = false
    @State private var rotationAngle: Double = 0
    
    var body: some View {
        VStack(spacing: 0) {
            // Drag handle
            dragHandle
            
            // Header
            panelHeader
            
            // Content
            if panelManager.isExpanded {
                expandedContent
            } else {
                compactContent
            }
        }
        .background(panelBackground)
        .clipShape(RoundedRectangle(cornerRadius: 24))
        .shadow(color: theme.primaryDark.opacity(0.4), radius: 20, y: -5)
        .offset(y: dragOffset)
        .gesture(dragGesture)
        .onAppear(perform: setupAnimations)
        .onChange(of: wsManager.isThinking) { _, isThinking in
            handleThinkingChange(isThinking)
        }
        .onChange(of: wsManager.thinkingSteps) { _, steps in
            syncThinkingSteps(steps)
        }
    }
    
    // MARK: - Drag Handle
    
    private var dragHandle: some View {
        VStack(spacing: 4) {
            RoundedRectangle(cornerRadius: 3)
                .fill(theme.textMuted.opacity(0.4))
                .frame(width: 40, height: 5)
        }
        .frame(height: 20)
        .frame(maxWidth: .infinity)
        .contentShape(Rectangle())
    }
    
    // MARK: - Panel Header
    
    private var panelHeader: some View {
        HStack(spacing: 12) {
            // Animated brain icon
            ZStack {
                // Outer glow
                if panelManager.isProcessing {
                    Circle()
                        .stroke(
                            AngularGradient(
                                colors: [theme.accentColor, theme.primaryColor, theme.accentColor],
                                center: .center
                            ),
                            lineWidth: 2
                        )
                        .frame(width: 44, height: 44)
                        .rotationEffect(.degrees(rotationAngle))
                }
                
                // Inner circle
                Circle()
                    .fill(theme.primaryColor.opacity(panelManager.isProcessing ? 0.3 : 0.15))
                    .frame(width: 40, height: 40)
                    .scaleEffect(pulseAnimation && panelManager.isProcessing ? 1.1 : 1.0)
                
                // Brain icon
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 18, weight: .semibold))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [theme.accentColor, theme.primaryLight],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
            }
            
            // Title and status
            VStack(alignment: .leading, spacing: 3) {
                HStack(spacing: 8) {
                    Text("🧠 Orchestrateur AI")
                        .font(.system(size: 15, weight: .bold))
                        .foregroundColor(theme.textPrimary)
                    
                    if panelManager.isProcessing {
                        HStack(spacing: 4) {
                            ForEach(0..<3, id: \.self) { i in
                                Circle()
                                    .fill(theme.accentColor)
                                    .frame(width: 5, height: 5)
                                    .opacity(dotOpacity(for: i))
                            }
                        }
                    }
                }
                
                HStack(spacing: 6) {
                    if let session = panelManager.currentSession {
                        Circle()
                            .fill(panelManager.isProcessing ? Color.green : Color.gray)
                            .frame(width: 6, height: 6)
                        
                        Text("\(session.steps.count) étapes")
                            .font(.system(size: 11))
                            .foregroundColor(theme.textMuted)
                        
                        Text("•")
                            .foregroundColor(theme.textMuted)
                        
                        Text(formatDuration(session.duration))
                            .font(.system(size: 11, weight: .medium))
                            .foregroundColor(theme.accentColor)
                    } else {
                        Text("En attente...")
                            .font(.system(size: 11))
                            .foregroundColor(theme.textMuted)
                    }
                }
            }
            
            Spacer()
            
            // Controls
            HStack(spacing: 8) {
                // History button
                Button(action: { withAnimation { showHistory.toggle() } }) {
                    Image(systemName: showHistory ? "clock.fill" : "clock")
                        .font(.system(size: 16))
                        .foregroundColor(showHistory ? theme.accentColor : theme.textMuted)
                        .padding(8)
                        .background(showHistory ? theme.accentColor.opacity(0.2) : Color.clear)
                        .cornerRadius(8)
                }
                
                // Pin button
                Button(action: { panelManager.isPinned.toggle() }) {
                    Image(systemName: panelManager.isPinned ? "pin.fill" : "pin")
                        .font(.system(size: 14))
                        .foregroundColor(panelManager.isPinned ? theme.accentColor : theme.textMuted)
                        .padding(8)
                        .background(panelManager.isPinned ? theme.accentColor.opacity(0.2) : Color.clear)
                        .cornerRadius(8)
                }
                
                // Expand/collapse
                Button(action: { withAnimation(.spring()) { panelManager.isExpanded.toggle() } }) {
                    Image(systemName: panelManager.isExpanded ? "chevron.down.circle.fill" : "chevron.up.circle.fill")
                        .font(.system(size: 20))
                        .foregroundColor(theme.textSecondary)
                }
            }
        }
        .padding(.horizontal, 16)
        .padding(.vertical, 12)
        .background(theme.surfaceElevated)
    }
    
    // MARK: - Expanded Content
    
    private var expandedContent: some View {
        VStack(spacing: 0) {
            // Progress bar
            if panelManager.isProcessing || panelManager.progress > 0 {
                progressBar
            }
            
            // Query summary
            if let session = panelManager.currentSession {
                querySummary(session.query)
            }
            
            // Content area
            if showHistory {
                historyList
            } else if let session = panelManager.currentSession {
                stepsTimeline(session)
            } else {
                emptyState
            }
        }
    }
    
    // MARK: - Compact Content
    
    private var compactContent: some View {
        VStack(spacing: 8) {
            // Mini progress
            if panelManager.isProcessing {
                GeometryReader { geo in
                    ZStack(alignment: .leading) {
                        Rectangle()
                            .fill(theme.surfaceElevated)
                        Rectangle()
                            .fill(LinearGradient(colors: [theme.primaryColor, theme.accentColor], startPoint: .leading, endPoint: .trailing))
                            .frame(width: geo.size.width * panelManager.progress)
                    }
                }
                .frame(height: 3)
                .cornerRadius(1.5)
                .padding(.horizontal, 16)
            }
            
            // Last 2 steps
            if let session = panelManager.currentSession {
                VStack(spacing: 4) {
                    ForEach(session.steps.suffix(2)) { step in
                        CompactStepView(step: step)
                    }
                }
                .padding(.horizontal, 16)
                .padding(.bottom, 12)
            }
        }
    }
    
    // MARK: - Progress Bar
    
    private var progressBar: some View {
        GeometryReader { geo in
            ZStack(alignment: .leading) {
                Rectangle()
                    .fill(theme.surfaceElevated)
                
                Rectangle()
                    .fill(
                        LinearGradient(
                            colors: [theme.primaryColor, theme.accentColor],
                            startPoint: .leading,
                            endPoint: .trailing
                        )
                    )
                    .frame(width: geo.size.width * panelManager.progress)
                    .animation(.easeInOut(duration: 0.3), value: panelManager.progress)
                
                // Shimmer
                if panelManager.isProcessing {
                    shimmerEffect(width: geo.size.width)
                }
            }
        }
        .frame(height: 4)
    }
    
    private func shimmerEffect(width: CGFloat) -> some View {
        Rectangle()
            .fill(
                LinearGradient(
                    colors: [.clear, .white.opacity(0.4), .clear],
                    startPoint: .leading,
                    endPoint: .trailing
                )
            )
            .frame(width: 60)
            .offset(x: pulseAnimation ? width : -60)
            .animation(.linear(duration: 1.2).repeatForever(autoreverses: false), value: pulseAnimation)
    }
    
    // MARK: - Query Summary
    
    private func querySummary(_ query: String) -> some View {
        HStack(spacing: 8) {
            Image(systemName: "text.bubble.fill")
                .font(.system(size: 11))
                .foregroundColor(theme.accentColor)
            
            Text(query)
                .font(.system(size: 11))
                .foregroundColor(theme.textSecondary)
                .lineLimit(1)
            
            Spacer()
        }
        .padding(.horizontal, 16)
        .padding(.vertical, 8)
        .background(theme.backgroundColor.opacity(0.5))
    }
    
    // MARK: - Steps Timeline
    
    private func stepsTimeline(_ session: AISession) -> some View {
        ScrollViewReader { proxy in
            ScrollView {
                LazyVStack(spacing: 0) {
                    ForEach(Array(session.steps.enumerated()), id: \.element.id) { index, step in
                        StepTimelineRow(
                            step: step,
                            index: index,
                            isSelected: panelManager.selectedStepId == step.id,
                            isLast: index == session.steps.count - 1
                        ) {
                            panelManager.selectedStepId = step.id
                        }
                        .id(step.id)
                    }
                }
                .padding(.vertical, 12)
            }
            .frame(maxHeight: 350)
            .onChange(of: session.steps.count) { _, _ in
                if let lastStep = session.steps.last {
                    withAnimation {
                        proxy.scrollTo(lastStep.id, anchor: .bottom)
                    }
                }
            }
        }
    }
    
    // MARK: - History List
    
    private var historyList: some View {
        ScrollView {
            LazyVStack(spacing: 10) {
                if panelManager.sessionHistory.isEmpty {
                    VStack(spacing: 12) {
                        Image(systemName: "clock.badge.questionmark")
                            .font(.system(size: 36))
                            .foregroundColor(theme.textMuted)
                        Text("Aucun historique")
                            .font(.system(size: 13))
                            .foregroundColor(theme.textMuted)
                    }
                    .padding(40)
                } else {
                    ForEach(panelManager.sessionHistory) { session in
                        HistorySessionRow(session: session)
                            .onTapGesture {
                                selectedHistorySession = session
                            }
                    }
                }
            }
            .padding(16)
        }
        .frame(maxHeight: 350)
        .sheet(item: $selectedHistorySession) { session in
            AISessionDetailSheet(session: session)
        }
    }
    
    // MARK: - Empty State
    
    private var emptyState: some View {
        VStack(spacing: 16) {
            Image(systemName: "brain.head.profile")
                .font(.system(size: 32))
                .foregroundStyle(
                    LinearGradient(
                        colors: [theme.accentColor.opacity(0.5), theme.primaryLight.opacity(0.5)],
                        startPoint: .topLeading,
                        endPoint: .bottomTrailing
                    )
                )
            
            Text("En attente d'une tâche...")
                .font(.system(size: 13))
                .foregroundColor(theme.textMuted)
        }
        .frame(maxWidth: .infinity)
        .padding(32)
    }
    
    // MARK: - Background
    
    private var panelBackground: some View {
        RoundedRectangle(cornerRadius: 24)
            .fill(theme.surfaceColor)
            .overlay(
                RoundedRectangle(cornerRadius: 24)
                    .stroke(
                        LinearGradient(
                            colors: [
                                theme.primaryColor.opacity(panelManager.isProcessing ? 0.5 : 0.3),
                                theme.accentColor.opacity(panelManager.isProcessing ? 0.3 : 0.1)
                            ],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        ),
                        lineWidth: 1.5
                    )
            )
    }
    
    // MARK: - Drag Gesture
    
    private var dragGesture: some Gesture {
        DragGesture()
            .onChanged { value in
                if value.translation.height > 0 {
                    dragOffset = value.translation.height * 0.5
                }
            }
            .onEnded { value in
                if value.translation.height > 100 {
                    withAnimation(.spring()) {
                        panelManager.isVisible = false
                    }
                }
                withAnimation(.spring()) {
                    dragOffset = 0
                }
            }
    }
    
    // MARK: - Helpers
    
    private func setupAnimations() {
        withAnimation(.easeInOut(duration: 1).repeatForever()) {
            pulseAnimation = true
        }
        withAnimation(.linear(duration: 3).repeatForever(autoreverses: false)) {
            rotationAngle = 360
        }
    }
    
    private func handleThinkingChange(_ isThinking: Bool) {
        if isThinking && panelManager.currentSession == nil {
            panelManager.startSession(query: "Traitement en cours...", requestId: wsManager.currentRequestId ?? UUID().uuidString)
        } else if !isThinking && panelManager.isProcessing {
            panelManager.endSession()
        }
    }
    
    private func syncThinkingSteps(_ steps: [ThinkingStep]) {
        guard let lastStep = steps.last,
              panelManager.currentSession?.steps.contains(where: { $0.id == lastStep.id }) != true else {
            return
        }
        
        let aiStep = AIStep(
            type: mapThinkingType(lastStep.type),
            title: lastStep.title,
            description: lastStep.description,
            agent: lastStep.agent
        )
        
        panelManager.addStep(aiStep)
    }
    
    private func mapThinkingType(_ type: ThinkingStepType) -> AIStep.StepType {
        switch type {
        case .analyzing: return .analyzing
        case .routing: return .routing
        case .planning: return .planning
        case .decomposing: return .decomposing
        case .executing: return .executing
        case .collaborating: return .collaboration
        case .reformulating: return .reformulation
        case .validating: return .validation
        case .recovering: return .fallback
        }
    }
    
    private func dotOpacity(for index: Int) -> Double {
        let phase = (Int(Date().timeIntervalSince1970 * 2) + index) % 3
        return phase == 0 ? 1.0 : 0.3
    }
    
    private func formatDuration(_ duration: TimeInterval) -> String {
        if duration < 1 {
            return "\(Int(duration * 1000))ms"
        } else if duration < 60 {
            return String(format: "%.1fs", duration)
        } else {
            return String(format: "%.0fm", duration / 60)
        }
    }
}

// MARK: - Step Timeline Row

struct StepTimelineRow: View {
    @EnvironmentObject var theme: ThemeManager
    let step: AIStep
    let index: Int
    let isSelected: Bool
    let isLast: Bool
    let onTap: () -> Void
    
    @State private var appear = false
    
    var body: some View {
        HStack(alignment: .top, spacing: 12) {
            // Timeline
            VStack(spacing: 0) {
                ZStack {
                    Circle()
                        .fill(step.type.color.opacity(isSelected ? 0.3 : 0.15))
                        .frame(width: 36, height: 36)
                    
                    if step.status == .inProgress {
                        Circle()
                            .stroke(step.type.color, lineWidth: 2)
                            .frame(width: 36, height: 36)
                            .rotationEffect(.degrees(appear ? 360 : 0))
                            .animation(.linear(duration: 1).repeatForever(autoreverses: false), value: appear)
                    }
                    
                    Image(systemName: step.type.icon)
                        .font(.system(size: 14, weight: .semibold))
                        .foregroundColor(step.type.color)
                }
                
                if !isLast {
                    Rectangle()
                        .fill(step.type.color.opacity(0.3))
                        .frame(width: 2, height: 20)
                }
            }
            
            // Content
            VStack(alignment: .leading, spacing: 6) {
                HStack {
                    Text(step.title)
                        .font(.system(size: 13, weight: .semibold))
                        .foregroundColor(theme.textPrimary)
                    
                    Spacer()
                    
                    AIStepStatusBadge(status: step.status)
                }
                
                Text(step.description)
                    .font(.system(size: 11))
                    .foregroundColor(theme.textSecondary)
                    .lineLimit(isSelected ? nil : 2)
                
                // Badges
                HStack(spacing: 8) {
                    if let tool = step.tool {
                        AIStepBadge(icon: "wrench.fill", text: tool, color: .cyan)
                    }
                    if let agent = step.agent {
                        AIStepBadge(icon: "person.fill", text: agent, color: step.type.color)
                    }
                    if let duration = step.duration {
                        Text(String(format: "%.0fms", duration * 1000))
                            .font(.system(size: 9, weight: .medium))
                            .foregroundColor(theme.textMuted)
                    }
                }
                
                // Output preview
                if isSelected, let output = step.output {
                    Text(output)
                        .font(.system(size: 10, design: .monospaced))
                        .foregroundColor(theme.textSecondary)
                        .padding(8)
                        .background(theme.backgroundColor.opacity(0.5))
                        .cornerRadius(6)
                        .lineLimit(5)
                }
            }
            .padding(.vertical, 6)
        }
        .padding(.horizontal, 16)
        .background(isSelected ? theme.surfaceElevated.opacity(0.5) : Color.clear)
        .contentShape(Rectangle())
        .onTapGesture(perform: onTap)
        .opacity(appear ? 1 : 0)
        .offset(x: appear ? 0 : -20)
        .onAppear {
            withAnimation(.spring(response: 0.4).delay(0.05)) {
                appear = true
            }
        }
    }
}

// MARK: - Compact Step View

struct CompactStepView: View {
    @EnvironmentObject var theme: ThemeManager
    let step: AIStep
    
    var body: some View {
        HStack(spacing: 10) {
            Image(systemName: step.type.icon)
                .font(.system(size: 11))
                .foregroundColor(step.type.color)
                .frame(width: 20)
            
            Text(step.title)
                .font(.system(size: 11))
                .foregroundColor(theme.textPrimary)
                .lineLimit(1)
            
            Spacer()
            
            AIStepStatusBadge(status: step.status, compact: true)
        }
        .padding(.vertical, 4)
    }
}

// MARK: - AI Step Status Badge

struct AIStepStatusBadge: View {
    let status: AIStep.Status
    var compact: Bool = false
    
    var body: some View {
        HStack(spacing: compact ? 2 : 4) {
            switch status {
            case .pending:
                Circle()
                    .fill(Color.gray)
                    .frame(width: 6, height: 6)
            case .inProgress:
                ProgressView()
                    .scaleEffect(0.5)
                    .frame(width: 12, height: 12)
            case .completed:
                Image(systemName: "checkmark.circle.fill")
                    .font(.system(size: compact ? 10 : 12))
                    .foregroundColor(.green)
            case .failed:
                Image(systemName: "xmark.circle.fill")
                    .font(.system(size: compact ? 10 : 12))
                    .foregroundColor(.red)
            case .skipped:
                Image(systemName: "arrow.uturn.forward.circle")
                    .font(.system(size: compact ? 10 : 12))
                    .foregroundColor(.gray)
            }
            
            if !compact {
                Text(status.rawValue.capitalized)
                    .font(.system(size: 9))
                    .foregroundColor(statusColor)
            }
        }
    }
    
    private var statusColor: Color {
        switch status {
        case .pending: return .gray
        case .inProgress: return .orange
        case .completed: return .green
        case .failed: return .red
        case .skipped: return .gray
        }
    }
}

// MARK: - AI Step Badge

struct AIStepBadge: View {
    let icon: String
    let text: String
    let color: Color
    
    var body: some View {
        HStack(spacing: 4) {
            Image(systemName: icon)
                .font(.system(size: 9))
            Text(text)
                .font(.system(size: 9, weight: .medium))
        }
        .foregroundColor(color)
        .padding(.horizontal, 8)
        .padding(.vertical, 3)
        .background(color.opacity(0.15))
        .cornerRadius(6)
    }
}

// MARK: - History Session Row

struct HistorySessionRow: View {
    @EnvironmentObject var theme: ThemeManager
    let session: AISession
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            HStack {
                Image(systemName: session.status == .completed ? "checkmark.circle.fill" : "xmark.circle.fill")
                    .foregroundColor(session.status == .completed ? .green : .red)
                
                Text(session.query)
                    .font(.system(size: 12, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                    .lineLimit(1)
                
                Spacer()
                
                Text(session.startTime, style: .relative)
                    .font(.system(size: 10))
                    .foregroundColor(theme.textMuted)
            }
            
            HStack(spacing: 12) {
                Label("\(session.steps.count)", systemImage: "list.number")
                Label(formatDuration(session.duration), systemImage: "clock")
            }
            .font(.system(size: 10))
            .foregroundColor(theme.textMuted)
        }
        .padding(12)
        .background(theme.surfaceElevated)
        .cornerRadius(10)
    }
    
    private func formatDuration(_ duration: TimeInterval) -> String {
        if duration < 60 {
            return String(format: "%.1fs", duration)
        }
        return String(format: "%.0fm", duration / 60)
    }
}

// MARK: - AI Session Detail Sheet

struct AISessionDetailSheet: View {
    @EnvironmentObject var theme: ThemeManager
    @Environment(\.dismiss) var dismiss
    let session: AISession
    
    var body: some View {
        NavigationView {
            ScrollView {
                VStack(alignment: .leading, spacing: 16) {
                    // Query
                    VStack(alignment: .leading, spacing: 8) {
                        Text("Requête")
                            .font(.system(size: 12, weight: .semibold))
                            .foregroundColor(theme.textMuted)
                        Text(session.query)
                            .font(.system(size: 14))
                            .foregroundColor(theme.textPrimary)
                    }
                    .padding(16)
                    .frame(maxWidth: .infinity, alignment: .leading)
                    .background(theme.surfaceElevated)
                    .cornerRadius(12)
                    
                    // Steps
                    VStack(alignment: .leading, spacing: 12) {
                        Text("Étapes (\(session.steps.count))")
                            .font(.system(size: 12, weight: .semibold))
                            .foregroundColor(theme.textMuted)
                        
                        ForEach(Array(session.steps.enumerated()), id: \.element.id) { index, step in
                            AIHistoryStepCard(step: step, index: index + 1)
                        }
                    }
                    .padding(16)
                    .background(theme.surfaceElevated)
                    .cornerRadius(12)
                    
                    // Output
                    if let output = session.finalOutput {
                        VStack(alignment: .leading, spacing: 8) {
                            Text("Résultat")
                                .font(.system(size: 12, weight: .semibold))
                                .foregroundColor(theme.textMuted)
                            Text(output)
                                .font(.system(size: 13))
                                .foregroundColor(theme.textPrimary)
                        }
                        .padding(16)
                        .frame(maxWidth: .infinity, alignment: .leading)
                        .background(theme.surfaceElevated)
                        .cornerRadius(12)
                    }
                }
                .padding(16)
            }
            .background(theme.backgroundColor)
            .navigationTitle("Session")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .navigationBarTrailing) {
                    Button("Fermer") { dismiss() }
                }
            }
        }
    }
}

// MARK: - AI History Step Card

struct AIHistoryStepCard: View {
    @EnvironmentObject var theme: ThemeManager
    let step: AIStep
    let index: Int
    
    @State private var expanded = false
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            HStack {
                Text("\(index)")
                    .font(.system(size: 10, weight: .bold))
                    .foregroundColor(.white)
                    .frame(width: 20, height: 20)
                    .background(step.type.color)
                    .cornerRadius(5)
                
                Image(systemName: step.type.icon)
                    .font(.system(size: 12))
                    .foregroundColor(step.type.color)
                
                Text(step.title)
                    .font(.system(size: 12, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                AIStepStatusBadge(status: step.status, compact: true)
                
                Button(action: { expanded.toggle() }) {
                    Image(systemName: expanded ? "chevron.up" : "chevron.down")
                        .font(.system(size: 11))
                        .foregroundColor(theme.textMuted)
                }
            }
            
            if expanded {
                VStack(alignment: .leading, spacing: 6) {
                    Text(step.description)
                        .font(.system(size: 11))
                        .foregroundColor(theme.textSecondary)
                    
                    if let output = step.output {
                        Text(output)
                            .font(.system(size: 10, design: .monospaced))
                            .foregroundColor(theme.textPrimary)
                            .padding(6)
                            .background(theme.backgroundColor)
                            .cornerRadius(4)
                    }
                }
                .padding(.leading, 28)
            }
        }
        .padding(10)
        .background(theme.backgroundColor.opacity(0.5))
        .cornerRadius(8)
    }
}

// MARK: - Preview

#Preview {
    ZStack {
        Color.black.ignoresSafeArea()
        
        VStack {
            Spacer()
            UnifiedCollaborationPanel(wsManager: WebSocketManager.shared)
                .padding(.horizontal, 8)
        }
    }
    .environmentObject(ThemeManager.shared)
}
