//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Tools View - MCP Tools Browser                              ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

struct ToolsView: View {
    @EnvironmentObject var theme: ThemeManager
    @State private var searchText = ""
    @State private var selectedCategory: String? = nil
    
    let categories = ["docker", "vpn", "identity", "monitoring", "routing", "infrastructure"]
    
    let tools: [ToolItem] = [
        ToolItem(name: "docker_services_list", description: "Liste tous les services Docker Swarm", category: "docker", icon: "cube.box.fill"),
        ToolItem(name: "docker_service_logs", description: "Récupère les logs d'un service", category: "docker", icon: "doc.text.fill"),
        ToolItem(name: "docker_service_restart", description: "Redémarre un service Docker", category: "docker", icon: "arrow.clockwise"),
        ToolItem(name: "docker_service_scale", description: "Scale un service à N replicas", category: "docker", icon: "arrow.up.arrow.down"),
        ToolItem(name: "headscale_nodes_list", description: "Liste les nodes VPN connectés", category: "vpn", icon: "network"),
        ToolItem(name: "headscale_users_list", description: "Liste les utilisateurs VPN", category: "vpn", icon: "person.2.fill"),
        ToolItem(name: "headscale_preauth_key_create", description: "Crée une clé pré-auth VPN", category: "vpn", icon: "key.fill"),
        ToolItem(name: "authentik_users_list", description: "Liste les utilisateurs SSO", category: "identity", icon: "person.badge.key.fill"),
        ToolItem(name: "authentik_applications_list", description: "Liste les applications SSO", category: "identity", icon: "app.badge.fill"),
        ToolItem(name: "prometheus_query", description: "Exécute une requête PromQL", category: "monitoring", icon: "chart.line.uptrend.xyaxis"),
        ToolItem(name: "prometheus_alerts", description: "Affiche les alertes actives", category: "monitoring", icon: "bell.badge.fill"),
        ToolItem(name: "grafana_dashboards_list", description: "Liste les dashboards Grafana", category: "monitoring", icon: "rectangle.3.group.fill"),
        ToolItem(name: "traefik_routers_list", description: "Liste les routers Traefik", category: "routing", icon: "arrow.triangle.branch"),
        ToolItem(name: "infrastructure_health", description: "Vérifie la santé de l'infra", category: "infrastructure", icon: "heart.fill"),
    ]
    
    var filteredTools: [ToolItem] {
        var result = tools
        
        if let category = selectedCategory {
            result = result.filter { $0.category == category }
        }
        
        if !searchText.isEmpty {
            result = result.filter {
                $0.name.localizedCaseInsensitiveContains(searchText) ||
                $0.description.localizedCaseInsensitiveContains(searchText)
            }
        }
        
        return result
    }
    
    var body: some View {
        NavigationStack {
            ZStack {
                theme.backgroundGradient
                    .ignoresSafeArea()
                
                VStack(spacing: 0) {
                    // Search bar
                    HStack(spacing: 12) {
                        Image(systemName: "magnifyingglass")
                            .foregroundColor(theme.textMuted)
                        
                        TextField("Rechercher un outil...", text: $searchText)
                            .foregroundColor(theme.textPrimary)
                    }
                    .padding(12)
                    .background(theme.surfaceColor)
                    .cornerRadius(theme.radiusMedium)
                    .padding(.horizontal)
                    .padding(.top)
                    
                    // Categories
                    ScrollView(.horizontal, showsIndicators: false) {
                        HStack(spacing: 10) {
                            CategoryPill(
                                title: "Tous",
                                isSelected: selectedCategory == nil
                            ) {
                                selectedCategory = nil
                            }
                            
                            ForEach(categories, id: \.self) { category in
                                CategoryPill(
                                    title: category.capitalized,
                                    isSelected: selectedCategory == category
                                ) {
                                    selectedCategory = category
                                }
                            }
                        }
                        .padding(.horizontal)
                        .padding(.vertical, 12)
                    }
                    
                    // Tools list
                    ScrollView {
                        LazyVStack(spacing: 12) {
                            ForEach(filteredTools) { tool in
                                ToolRow(tool: tool)
                            }
                        }
                        .padding()
                    }
                }
            }
            .navigationTitle("Outils MCP")
        }
    }
}

// MARK: - Tool Item

struct ToolItem: Identifiable {
    let id = UUID()
    let name: String
    let description: String
    let category: String
    let icon: String
}

// MARK: - Category Pill

struct CategoryPill: View {
    @EnvironmentObject var theme: ThemeManager
    let title: String
    let isSelected: Bool
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            Text(title)
                .font(.system(size: 14, weight: isSelected ? .semibold : .medium))
                .foregroundColor(isSelected ? .white : theme.textSecondary)
                .padding(.horizontal, 16)
                .padding(.vertical, 8)
                .background(
                    isSelected
                    ? AnyShapeStyle(theme.primaryGradient)
                    : AnyShapeStyle(theme.surfaceColor)
                )
                .cornerRadius(theme.radiusLarge)
        }
    }
}

// MARK: - Tool Row

struct ToolRow: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    let tool: ToolItem
    @State private var showExecuteSheet = false
    
    var body: some View {
        Button(action: {
            showExecuteSheet = true
        }) {
            HStack(spacing: 16) {
                // Icon
                ZStack {
                    RoundedRectangle(cornerRadius: theme.radiusSmall)
                        .fill(categoryColor.opacity(0.2))
                        .frame(width: 44, height: 44)
                    
                    Image(systemName: tool.icon)
                        .font(.system(size: 18))
                        .foregroundColor(categoryColor)
                }
                
                // Info
                VStack(alignment: .leading, spacing: 4) {
                    Text(tool.name)
                        .font(.system(size: 14, weight: .medium, design: .monospaced))
                        .foregroundColor(theme.textPrimary)
                    
                    Text(tool.description)
                        .font(.system(size: 13))
                        .foregroundColor(theme.textSecondary)
                        .lineLimit(2)
                }
                
                Spacer()
                
                Image(systemName: "play.circle.fill")
                    .font(.system(size: 24))
                    .foregroundColor(theme.accentColor)
            }
            .padding(16)
            .background(theme.surfaceColor)
            .cornerRadius(theme.radiusMedium)
        }
        .sheet(isPresented: $showExecuteSheet) {
            ToolExecuteSheet(tool: tool)
        }
    }
    
    var categoryColor: Color {
        switch tool.category {
        case "docker": return theme.infoColor
        case "vpn": return theme.successColor
        case "identity": return theme.warningColor
        case "monitoring": return theme.accentColor
        case "routing": return Color.purple
        default: return theme.textSecondary
        }
    }
}

// MARK: - Tool Execute Sheet

struct ToolExecuteSheet: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    @Environment(\.dismiss) var dismiss
    let tool: ToolItem
    @State private var parameters: [String: String] = [:]
    
    var body: some View {
        NavigationStack {
            ZStack {
                theme.backgroundGradient
                    .ignoresSafeArea()
                
                VStack(spacing: 20) {
                    // Tool info
                    VStack(spacing: 12) {
                        Image(systemName: tool.icon)
                            .font(.system(size: 40))
                            .foregroundColor(theme.accentColor)
                        
                        Text(tool.name)
                            .font(.system(size: 18, weight: .semibold, design: .monospaced))
                            .foregroundColor(theme.textPrimary)
                        
                        Text(tool.description)
                            .font(.system(size: 14))
                            .foregroundColor(theme.textSecondary)
                            .multilineTextAlignment(.center)
                    }
                    .padding()
                    .frame(maxWidth: .infinity)
                    .background(theme.surfaceColor)
                    .cornerRadius(theme.radiusMedium)
                    
                    Spacer()
                    
                    // Execute button
                    Button(action: executeTool) {
                        HStack {
                            Image(systemName: "play.fill")
                            Text("Exécuter via Chat")
                        }
                        .frame(maxWidth: .infinity)
                        .padding(.vertical, 16)
                        .background(theme.primaryGradient)
                        .foregroundColor(.white)
                        .font(.system(size: 16, weight: .semibold))
                        .cornerRadius(theme.radiusMedium)
                    }
                }
                .padding()
            }
            .navigationTitle("Exécuter l'outil")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .cancellationAction) {
                    Button("Annuler") {
                        dismiss()
                    }
                }
            }
        }
    }
    
    private func executeTool() {
        let message = "Exécute l'outil \(tool.name) pour moi"
        Task {
            await chatManager.sendMessage(message)
        }
        dismiss()
    }
}

// MARK: - Preview

#Preview {
    ToolsView()
        .environmentObject(ThemeManager.shared)
        .environmentObject(ChatManager.shared)
}
