//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Thinking Stream View                                        ║
//  ║         Sophisticated Real-Time AI Orchestration Visualization                ║
//  ║         LangGraph Multi-Agent Collaboration Display                           ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

// MARK: - Inline Thinking View (Shows in chat)

struct InlineThinkingView: View {
    @EnvironmentObject var theme: ThemeManager
    @ObservedObject var wsManager: WebSocketManager
    
    @State private var pulseAnimation = false
    @State private var waveOffset: CGFloat = 0
    @State private var rotationAngle: Double = 0
    
    var body: some View {
        VStack(spacing: 16) {
            // Sophisticated Header with animated brain
            HStack(spacing: 14) {
                // Animated brain with glow
                ZStack {
                    // Outer glow ring
                    Circle()
                        .stroke(
                            AngularGradient(
                                colors: [theme.accentColor, theme.primaryColor, theme.accentColor],
                                center: .center
                            ),
                            lineWidth: 3
                        )
                        .frame(width: 54, height: 54)
                        .rotationEffect(.degrees(rotationAngle))
                    
                    // Inner glow
                    Circle()
                        .fill(
                            RadialGradient(
                                colors: [theme.accentColor.opacity(0.4), Color.clear],
                                center: .center,
                                startRadius: 5,
                                endRadius: 30
                            )
                        )
                        .frame(width: 50, height: 50)
                        .scaleEffect(pulseAnimation ? 1.2 : 0.9)
                    
                    // Brain icon
                    Image(systemName: "brain.head.profile")
                        .font(.system(size: 26, weight: .medium))
                        .foregroundStyle(
                            LinearGradient(
                                colors: [theme.accentColor, theme.primaryLight],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            )
                        )
                }
                
                VStack(alignment: .leading, spacing: 6) {
                    HStack(spacing: 8) {
                        Text("🧠 Orchestrateur AI")
                            .font(.system(size: 15, weight: .bold))
                            .foregroundColor(theme.textPrimary)
                        
                        // Animated processing dots
                        HStack(spacing: 4) {
                            ForEach(0..<3) { i in
                                Circle()
                                    .fill(theme.accentColor)
                                    .frame(width: 6, height: 6)
                                    .opacity(dotOpacity(for: i))
                                    .animation(.easeInOut(duration: 0.6).repeatForever().delay(Double(i) * 0.2), value: pulseAnimation)
                            }
                        }
                    }
                    
                    HStack(spacing: 6) {
                        Text("Multi-Agent Collaboration")
                            .font(.system(size: 11, weight: .medium))
                            .foregroundColor(theme.textMuted)
                        
                        Text("•")
                            .foregroundColor(theme.textMuted)
                        
                        Text("\(Int(wsManager.thinkingProgress * 100))%")
                            .font(.system(size: 11, weight: .bold))
                            .foregroundColor(theme.accentColor)
                    }
                }
                
                Spacer()
                
                // Agent count badge
                if !wsManager.thinkingSteps.isEmpty {
                    VStack(spacing: 2) {
                        Text("\(wsManager.thinkingSteps.count)")
                            .font(.system(size: 18, weight: .bold))
                            .foregroundColor(theme.accentColor)
                        Text("étapes")
                            .font(.system(size: 9))
                            .foregroundColor(theme.textMuted)
                    }
                    .padding(.horizontal, 12)
                    .padding(.vertical, 6)
                    .background(theme.surfaceElevated)
                    .cornerRadius(10)
                }
            }
            
            // Sophisticated Progress Bar
            GeometryReader { geo in
                ZStack(alignment: .leading) {
                    // Background track
                    RoundedRectangle(cornerRadius: 4)
                        .fill(theme.surfaceElevated)
                    
                    // Progress fill with gradient
                    RoundedRectangle(cornerRadius: 4)
                        .fill(
                            LinearGradient(
                                colors: [theme.primaryColor, theme.accentColor, theme.primaryLight],
                                startPoint: .leading,
                                endPoint: .trailing
                            )
                        )
                        .frame(width: geo.size.width * wsManager.thinkingProgress)
                    
                    // Shimmer effect
                    if wsManager.thinkingProgress > 0 && wsManager.thinkingProgress < 1 {
                        Rectangle()
                            .fill(
                                LinearGradient(
                                    colors: [.clear, .white.opacity(0.5), .clear],
                                    startPoint: .leading,
                                    endPoint: .trailing
                                )
                            )
                            .frame(width: 60)
                            .offset(x: waveOffset)
                    }
                }
                .onAppear {
                    withAnimation(.linear(duration: 1.2).repeatForever(autoreverses: false)) {
                        waveOffset = geo.size.width
                    }
                }
            }
            .frame(height: 8)
            .cornerRadius(4)
            
            // Agent Steps Timeline
            if !wsManager.thinkingSteps.isEmpty {
                VStack(spacing: 0) {
                    ForEach(Array(wsManager.thinkingSteps.enumerated()), id: \.element.id) { index, step in
                        ThinkingStepCard(step: step, isLast: index == wsManager.thinkingSteps.count - 1)
                    }
                }
            }
        }
        .padding(18)
        .background(
            RoundedRectangle(cornerRadius: 20)
                .fill(theme.surfaceColor)
                .overlay(
                    RoundedRectangle(cornerRadius: 20)
                        .stroke(
                            LinearGradient(
                                colors: [theme.primaryColor.opacity(0.5), theme.accentColor.opacity(0.3)],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            ),
                            lineWidth: 1.5
                        )
                )
        )
        .shadow(color: theme.primaryDark.opacity(0.3), radius: 15, y: 8)
        .onAppear {
            withAnimation(.easeInOut(duration: 1).repeatForever()) {
                pulseAnimation.toggle()
            }
            withAnimation(.linear(duration: 3).repeatForever(autoreverses: false)) {
                rotationAngle = 360
            }
        }
    }
    
    private func dotOpacity(for index: Int) -> Double {
        let phase = (Int(Date().timeIntervalSince1970 * 2) + index) % 3
        return phase == 0 ? 1.0 : 0.3
    }
}

// MARK: - Thinking Step Card

struct ThinkingStepCard: View {
    @EnvironmentObject var theme: ThemeManager
    let step: ThinkingStep
    let isLast: Bool
    
    @State private var appear = false
    
    var body: some View {
        HStack(alignment: .top, spacing: 14) {
            // Timeline connector
            VStack(spacing: 0) {
                // Icon circle
                ZStack {
                    Circle()
                        .fill(step.type.color.opacity(0.2))
                        .frame(width: 38, height: 38)
                    
                    Circle()
                        .stroke(step.type.color.opacity(0.5), lineWidth: 2)
                        .frame(width: 38, height: 38)
                    
                    Image(systemName: step.type.icon)
                        .font(.system(size: 16, weight: .semibold))
                        .foregroundColor(step.type.color)
                }
                
                // Connector line
                if !isLast {
                    Rectangle()
                        .fill(
                            LinearGradient(
                                colors: [step.type.color.opacity(0.5), theme.surfaceElevated],
                                startPoint: .top,
                                endPoint: .bottom
                            )
                        )
                        .frame(width: 2, height: 30)
                }
            }
            
            // Content
            VStack(alignment: .leading, spacing: 6) {
                HStack {
                    Text(step.title)
                        .font(.system(size: 14, weight: .semibold))
                        .foregroundColor(theme.textPrimary)
                    
                    Spacer()
                    
                    // Status indicator
                    Group {
                        switch step.status {
                        case .inProgress:
                            HStack(spacing: 4) {
                                ProgressView()
                                    .scaleEffect(0.6)
                                Text("En cours")
                                    .font(.system(size: 10))
                            }
                            .foregroundColor(theme.accentColor)
                        case .completed:
                            HStack(spacing: 4) {
                                Image(systemName: "checkmark.circle.fill")
                                    .font(.system(size: 12))
                                Text("Terminé")
                                    .font(.system(size: 10))
                            }
                            .foregroundColor(.green)
                        case .failed:
                            HStack(spacing: 4) {
                                Image(systemName: "xmark.circle.fill")
                                    .font(.system(size: 12))
                                Text("Erreur")
                                    .font(.system(size: 10))
                            }
                            .foregroundColor(.red)
                        }
                    }
                }
                
                Text(step.description)
                    .font(.system(size: 12))
                    .foregroundColor(theme.textSecondary)
                    .lineLimit(2)
                
                // Agent badge if available
                if let agent = step.agent {
                    HStack(spacing: 4) {
                        Image(systemName: "person.fill")
                            .font(.system(size: 10))
                        Text(agent)
                            .font(.system(size: 10, weight: .medium))
                    }
                    .foregroundColor(step.type.color)
                    .padding(.horizontal, 8)
                    .padding(.vertical, 3)
                    .background(step.type.color.opacity(0.1))
                    .cornerRadius(6)
                }
            }
            .padding(.vertical, 8)
        }
        .opacity(appear ? 1 : 0)
        .offset(x: appear ? 0 : -30)
        .onAppear {
            withAnimation(.spring(response: 0.5, dampingFraction: 0.7)) {
                appear = true
            }
        }
    }
}

// MARK: - Connection Status Badge

struct ConnectionStatusBadge: View {
    @EnvironmentObject var theme: ThemeManager
    @ObservedObject var wsManager: WebSocketManager
    
    @State private var showLabel = false
    @State private var pulse = false
    
    var body: some View {
        Button(action: {
            withAnimation(.spring()) {
                showLabel.toggle()
            }
            
            if wsManager.connectionState == .disconnected {
                wsManager.connect()
            }
        }) {
            HStack(spacing: 8) {
                // Status indicator
                ZStack {
                    Circle()
                        .fill(wsManager.connectionState.color.opacity(0.3))
                        .frame(width: 20, height: 20)
                        .scaleEffect(pulse ? 1.3 : 1)
                    
                    Circle()
                        .fill(wsManager.connectionState.color)
                        .frame(width: 10, height: 10)
                }
                
                // Label
                if showLabel {
                    VStack(alignment: .leading, spacing: 2) {
                        Text(wsManager.connectionState.statusText)
                            .font(.system(size: 11, weight: .semibold))
                            .foregroundColor(theme.textPrimary)
                        
                        Text(wsManager.connectionState == .connected ? "WebSocket actif" : "Tap pour reconnecter")
                            .font(.system(size: 9))
                            .foregroundColor(theme.textMuted)
                    }
                }
            }
            .padding(.horizontal, showLabel ? 14 : 10)
            .padding(.vertical, 8)
            .background(
                Capsule()
                    .fill(theme.surfaceColor)
                    .shadow(color: wsManager.connectionState.color.opacity(0.3), radius: 8)
            )
            .overlay(
                Capsule()
                    .stroke(wsManager.connectionState.color.opacity(0.4), lineWidth: 1.5)
            )
        }
        .onAppear {
            withAnimation(.easeInOut(duration: 1.2).repeatForever()) {
                pulse.toggle()
            }
        }
    }
}

// MARK: - Floating Thinking Overlay

struct FloatingThinkingOverlay: View {
    @EnvironmentObject var theme: ThemeManager
    @ObservedObject var wsManager: WebSocketManager
    
    @State private var expanded = false
    @State private var position = CGPoint(x: UIScreen.main.bounds.width - 70, y: 180)
    @State private var dragOffset: CGSize = .zero
    
    var body: some View {
        Group {
            if wsManager.isThinking {
                ZStack {
                    if expanded {
                        // Expanded modal
                        VStack(spacing: 0) {
                            // Modal header
                            HStack {
                                HStack(spacing: 8) {
                                    Image(systemName: "brain.head.profile")
                                        .font(.system(size: 18))
                                        .foregroundColor(theme.accentColor)
                                    
                                    Text("Orchestration AI")
                                        .font(.system(size: 16, weight: .bold))
                                        .foregroundColor(theme.textPrimary)
                                }
                                
                                Spacer()
                                
                                Button(action: { withAnimation(.spring()) { expanded = false } }) {
                                    Image(systemName: "minus.circle.fill")
                                        .font(.system(size: 24))
                                        .foregroundColor(theme.textSecondary)
                                }
                            }
                            .padding()
                            .background(theme.surfaceElevated)
                            
                            ScrollView {
                                InlineThinkingView(wsManager: wsManager)
                                    .padding()
                            }
                        }
                        .frame(width: min(UIScreen.main.bounds.width - 40, 380), height: 450)
                        .background(
                            RoundedRectangle(cornerRadius: 24)
                                .fill(theme.backgroundColor)
                        )
                        .overlay(
                            RoundedRectangle(cornerRadius: 24)
                                .stroke(theme.primaryColor.opacity(0.3), lineWidth: 1)
                        )
                        .shadow(color: theme.primaryDark.opacity(0.5), radius: 25, y: 10)
                        .position(x: UIScreen.main.bounds.width / 2, y: UIScreen.main.bounds.height / 2 - 50)
                        .transition(.scale.combined(with: .opacity))
                    } else {
                        // Floating bubble
                        FloatingThinkingBubble(wsManager: wsManager)
                            .position(position)
                            .offset(dragOffset)
                            .gesture(
                                DragGesture()
                                    .onChanged { value in
                                        dragOffset = value.translation
                                    }
                                    .onEnded { value in
                                        position.x += value.translation.width
                                        position.y += value.translation.height
                                        dragOffset = .zero
                                        
                                        // Snap to edges
                                        let screenWidth = UIScreen.main.bounds.width
                                        let screenHeight = UIScreen.main.bounds.height
                                        
                                        withAnimation(.spring(response: 0.3)) {
                                            // Horizontal snap
                                            if position.x < screenWidth / 2 {
                                                position.x = 70
                                            } else {
                                                position.x = screenWidth - 70
                                            }
                                            
                                            // Vertical bounds
                                            position.y = min(max(100, position.y), screenHeight - 150)
                                        }
                                    }
                            )
                            .onTapGesture {
                                withAnimation(.spring(response: 0.4)) {
                                    expanded = true
                                }
                            }
                            .transition(.scale.combined(with: .opacity))
                    }
                }
            }
        }
        .animation(.spring(response: 0.4), value: wsManager.isThinking)
        .animation(.spring(response: 0.4), value: expanded)
    }
}

// MARK: - Floating Thinking Bubble

struct FloatingThinkingBubble: View {
    @EnvironmentObject var theme: ThemeManager
    @ObservedObject var wsManager: WebSocketManager
    
    @State private var rotation: Double = 0
    @State private var pulse = false
    @State private var innerPulse = false
    
    var body: some View {
        ZStack {
            // Background glow
            Circle()
                .fill(
                    RadialGradient(
                        colors: [theme.accentColor.opacity(0.4), Color.clear],
                        center: .center,
                        startRadius: 15,
                        endRadius: 60
                    )
                )
                .frame(width: 100, height: 100)
                .scaleEffect(pulse ? 1.2 : 0.9)
            
            // Outer ring
            Circle()
                .stroke(
                    AngularGradient(
                        colors: [theme.primaryColor, theme.accentColor, theme.primaryLight, theme.primaryColor],
                        center: .center
                    ),
                    lineWidth: 4
                )
                .frame(width: 70, height: 70)
                .rotationEffect(.degrees(rotation))
            
            // Progress ring
            Circle()
                .trim(from: 0, to: wsManager.thinkingProgress)
                .stroke(
                    LinearGradient(
                        colors: [theme.accentColor, theme.primaryLight],
                        startPoint: .leading,
                        endPoint: .trailing
                    ),
                    style: StrokeStyle(lineWidth: 5, lineCap: .round)
                )
                .frame(width: 55, height: 55)
                .rotationEffect(.degrees(-90))
            
            // Inner circle
            Circle()
                .fill(theme.surfaceColor)
                .frame(width: 50, height: 50)
                .shadow(color: theme.primaryDark.opacity(0.5), radius: 8)
            
            // Brain icon
            Image(systemName: "brain.head.profile")
                .font(.system(size: 22, weight: .medium))
                .foregroundStyle(
                    LinearGradient(
                        colors: [theme.accentColor, theme.primaryLight],
                        startPoint: .topLeading,
                        endPoint: .bottomTrailing
                    )
                )
                .scaleEffect(innerPulse ? 1.1 : 1)
            
            // Step count badge
            if !wsManager.thinkingSteps.isEmpty {
                ZStack {
                    Circle()
                        .fill(theme.accentColor)
                        .frame(width: 22, height: 22)
                    
                    Text("\(wsManager.thinkingSteps.count)")
                        .font(.system(size: 11, weight: .bold))
                        .foregroundColor(.white)
                }
                .offset(x: 28, y: -28)
            }
        }
        .frame(width: 100, height: 100)
        .onAppear {
            withAnimation(.easeInOut(duration: 1.5).repeatForever()) {
                pulse.toggle()
            }
            withAnimation(.easeInOut(duration: 0.8).repeatForever()) {
                innerPulse.toggle()
            }
            withAnimation(.linear(duration: 4).repeatForever(autoreverses: false)) {
                rotation = 360
            }
        }
    }
}

// MARK: - Preview

#Preview {
    ZStack {
        Color.black.ignoresSafeArea()
        
        VStack {
            InlineThinkingView(wsManager: WebSocketManager.shared)
                .padding()
            
            Spacer()
            
            HStack {
                ConnectionStatusBadge(wsManager: WebSocketManager.shared)
                Spacer()
            }
            .padding()
        }
    }
    .environmentObject(ThemeManager.shared)
}
