//
//  LanciioConnectButton.swift
//  PersonnelAssistant
//
//  Sophisticated Lanciio VIP MCP Hub integration button
//  - Custom logo with animated effects
//  - "Tools VIP" badge
//  - OAuth2 connection flow
//  - Premium workspace management
//

import SwiftUI

// MARK: - Lanciio Connect Button (Sophisticated)

struct LanciioConnectButton: View {
    @EnvironmentObject var theme: ThemeManager
    @StateObject private var lanciioManager = LanciioManager.shared
    @State private var showConnectionSheet = false
    @State private var isAnimating = false
    @State private var pulseAnimation = false
    
    // Lanciio brand colors
    private let lanciioYellow = Color(red: 0.9, green: 0.95, blue: 0.2)
    private let lanciioPurple = Color(red: 0.5, green: 0.25, blue: 0.5)
    
    var body: some View {
        Button {
            if lanciioManager.isConnected {
                showConnectionSheet = true
            } else {
                Task {
                    await lanciioManager.connect()
                }
            }
        } label: {
            HStack(spacing: 6) {
                // Lanciio Logo with animated glow
                ZStack {
                    // Glow effect when connected
                    if lanciioManager.isConnected {
                        Circle()
                            .fill(
                                RadialGradient(
                                    colors: [lanciioYellow.opacity(0.6), .clear],
                                    center: .center,
                                    startRadius: 0,
                                    endRadius: 20
                                )
                            )
                            .frame(width: 32, height: 32)
                            .scaleEffect(pulseAnimation ? 1.3 : 1.0)
                            .opacity(pulseAnimation ? 0.3 : 0.6)
                    }
                    
                    // Logo
                    Image("LanciioLogo")
                        .resizable()
                        .aspectRatio(contentMode: .fit)
                        .frame(width: 22, height: 22)
                        .rotationEffect(.degrees(isAnimating ? 360 : 0))
                }
                .frame(width: 26, height: 26)
                
                if lanciioManager.isConnecting {
                    // Loading state
                    HStack(spacing: 4) {
                        ProgressView()
                            .scaleEffect(0.6)
                            .tint(lanciioYellow)
                        
                        Text("Connexion...")
                            .font(.system(size: 9, weight: .medium))
                            .foregroundColor(theme.textMuted)
                    }
                } else if lanciioManager.isConnected {
                    // Connected state with VIP badge
                    VStack(alignment: .leading, spacing: 1) {
                        HStack(spacing: 4) {
                            Text("Tools VIP")
                                .font(.system(size: 10, weight: .bold))
                                .foregroundStyle(
                                    LinearGradient(
                                        colors: [lanciioYellow, lanciioPurple],
                                        startPoint: .leading,
                                        endPoint: .trailing
                                    )
                                )
                            
                            // VIP Crown badge
                            Image(systemName: "crown.fill")
                                .font(.system(size: 8))
                                .foregroundColor(lanciioYellow)
                        }
                        
                        HStack(spacing: 3) {
                            Text("\(lanciioManager.syncedToolsCount)")
                                .font(.system(size: 9, weight: .semibold))
                                .foregroundColor(theme.textPrimary)
                            
                            Text("outils")
                                .font(.system(size: 8))
                                .foregroundColor(theme.textMuted)
                            
                            // Connected dot
                            Circle()
                                .fill(Color.green)
                                .frame(width: 5, height: 5)
                                .shadow(color: .green.opacity(0.8), radius: 2)
                        }
                    }
                } else {
                    // Disconnected state
                    VStack(alignment: .leading, spacing: 1) {
                        Text("Lanciio")
                            .font(.system(size: 10, weight: .semibold))
                            .foregroundColor(theme.textSecondary)
                        
                        Text("Tools VIP")
                            .font(.system(size: 8, weight: .medium))
                            .foregroundColor(theme.textMuted)
                    }
                }
            }
            .padding(.horizontal, 10)
            .padding(.vertical, 7)
            .background(
                ZStack {
                    // Base background
                    RoundedRectangle(cornerRadius: 10)
                        .fill(theme.surfaceElevated)
                    
                    // Gradient border when connected
                    if lanciioManager.isConnected {
                        RoundedRectangle(cornerRadius: 10)
                            .stroke(
                                LinearGradient(
                                    colors: [lanciioYellow.opacity(0.6), lanciioPurple.opacity(0.4)],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                ),
                                lineWidth: 1.5
                            )
                    } else {
                        RoundedRectangle(cornerRadius: 10)
                            .stroke(theme.surfaceColor.opacity(0.5), lineWidth: 1)
                    }
                }
            )
            .shadow(
                color: lanciioManager.isConnected ? lanciioYellow.opacity(0.3) : .clear,
                radius: 6,
                y: 2
            )
        }
        .buttonStyle(.plain)
        .onAppear {
            // Start pulse animation when connected
            if lanciioManager.isConnected {
                withAnimation(.easeInOut(duration: 2).repeatForever(autoreverses: true)) {
                    pulseAnimation = true
                }
            }
        }
        .onChange(of: lanciioManager.isConnected) { _, connected in
            if connected {
                // Spin logo once on connect
                withAnimation(.easeInOut(duration: 0.8)) {
                    isAnimating = true
                }
                DispatchQueue.main.asyncAfter(deadline: .now() + 0.8) {
                    isAnimating = false
                }
                // Start pulse
                withAnimation(.easeInOut(duration: 2).repeatForever(autoreverses: true)) {
                    pulseAnimation = true
                }
            } else {
                pulseAnimation = false
            }
        }
        .sheet(isPresented: $showConnectionSheet) {
            LanciioConnectionSheet()
                .environmentObject(theme)
        }
    }
}

// MARK: - Connection Sheet (Sophisticated)

struct LanciioConnectionSheet: View {
    @EnvironmentObject var theme: ThemeManager
    @StateObject private var lanciioManager = LanciioManager.shared
    @Environment(\.dismiss) var dismiss
    @State private var isSyncing = false
    @State private var searchQuery = ""
    @State private var searchResults: [LanciioTool] = []
    @State private var logoRotation: Double = 0
    
    // Lanciio brand colors
    private let lanciioYellow = Color(red: 0.9, green: 0.95, blue: 0.2)
    private let lanciioPurple = Color(red: 0.5, green: 0.25, blue: 0.5)
    
    var body: some View {
        NavigationStack {
            ZStack {
                // Background with subtle gradient
                theme.backgroundGradient.ignoresSafeArea()
                
                // Decorative background elements
                VStack {
                    HStack {
                        Spacer()
                        Circle()
                            .fill(lanciioYellow.opacity(0.05))
                            .frame(width: 300, height: 300)
                            .offset(x: 100, y: -100)
                    }
                    Spacer()
                    HStack {
                        Circle()
                            .fill(lanciioPurple.opacity(0.05))
                            .frame(width: 250, height: 250)
                            .offset(x: -80, y: 50)
                        Spacer()
                    }
                }
                .ignoresSafeArea()
                
                ScrollView {
                    VStack(spacing: 24) {
                        // Header with animated logo
                        headerSection
                        
                        // VIP Stats
                        statsSection
                        
                        // Workspaces
                        if !lanciioManager.workspaces.isEmpty {
                            workspacesSection
                        }
                        
                        // Tools Search
                        toolsSearchSection
                        
                        // Actions
                        actionsSection
                    }
                    .padding()
                }
            }
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .principal) {
                    HStack(spacing: 6) {
                        Image("LanciioLogo")
                            .resizable()
                            .aspectRatio(contentMode: .fit)
                            .frame(width: 24, height: 24)
                        
                        Text("Lanciio")
                            .font(.system(size: 17, weight: .bold))
                            .foregroundStyle(
                                LinearGradient(
                                    colors: [lanciioYellow, lanciioPurple],
                                    startPoint: .leading,
                                    endPoint: .trailing
                                )
                            )
                        
                        // VIP Badge
                        Text("VIP")
                            .font(.system(size: 9, weight: .black))
                            .foregroundColor(.black)
                            .padding(.horizontal, 6)
                            .padding(.vertical, 2)
                            .background(
                                Capsule()
                                    .fill(lanciioYellow)
                            )
                    }
                }
                
                ToolbarItem(placement: .topBarTrailing) {
                    Button {
                        dismiss()
                    } label: {
                        Image(systemName: "xmark.circle.fill")
                            .font(.system(size: 24))
                            .foregroundStyle(theme.textMuted, theme.surfaceElevated)
                    }
                }
            }
        }
        .presentationDetents([.large])
        .presentationDragIndicator(.visible)
    }
    
    // MARK: - Header Section
    
    private var headerSection: some View {
        VStack(spacing: 20) {
            // Animated Logo
            ZStack {
                // Outer glow rings
                ForEach(0..<3) { i in
                    Circle()
                        .stroke(
                            LinearGradient(
                                colors: [
                                    lanciioYellow.opacity(0.3 - Double(i) * 0.1),
                                    lanciioPurple.opacity(0.2 - Double(i) * 0.05)
                                ],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            ),
                            lineWidth: 2
                        )
                        .frame(width: CGFloat(100 + i * 20), height: CGFloat(100 + i * 20))
                        .rotationEffect(.degrees(logoRotation + Double(i * 30)))
                }
                
                // Background circle
                Circle()
                    .fill(
                        LinearGradient(
                            colors: [
                                theme.surfaceElevated,
                                theme.surfaceColor.opacity(0.8)
                            ],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
                    .frame(width: 90, height: 90)
                    .shadow(color: lanciioYellow.opacity(0.3), radius: 15)
                
                // Logo
                Image("LanciioLogo")
                    .resizable()
                    .aspectRatio(contentMode: .fit)
                    .frame(width: 60, height: 60)
            }
            .onAppear {
                withAnimation(.linear(duration: 20).repeatForever(autoreverses: false)) {
                    logoRotation = 360
                }
            }
            
            // Title and status
            VStack(spacing: 8) {
                HStack(spacing: 8) {
                    Text("Lanciio VIP Hub")
                        .font(.system(size: 24, weight: .bold))
                        .foregroundStyle(
                            LinearGradient(
                                colors: [lanciioYellow, lanciioPurple],
                                startPoint: .leading,
                                endPoint: .trailing
                            )
                        )
                    
                    // Crown badge
                    Image(systemName: "crown.fill")
                        .font(.system(size: 16))
                        .foregroundColor(lanciioYellow)
                        .shadow(color: lanciioYellow.opacity(0.5), radius: 4)
                }
                
                Text("Votre hub d'outils MCP personnalisé")
                    .font(.system(size: 13))
                    .foregroundColor(theme.textSecondary)
                
                if lanciioManager.isConnected {
                    HStack(spacing: 6) {
                        Circle()
                            .fill(Color.green)
                            .frame(width: 8, height: 8)
                            .shadow(color: .green, radius: 3)
                        
                        Text("Connecté")
                            .font(.system(size: 12, weight: .medium))
                            .foregroundColor(.green)
                        
                        if let user = lanciioManager.currentUser {
                            Text("• \(user.email)")
                                .font(.system(size: 12))
                                .foregroundColor(theme.textMuted)
                        }
                    }
                    .padding(.top, 4)
                }
                
                if let lastSync = lanciioManager.lastSyncDate {
                    Text("Sync: \(lastSync.formatted(.relative(presentation: .named)))")
                        .font(.system(size: 11))
                        .foregroundColor(theme.textMuted)
                }
            }
        }
        .padding(.top, 10)
    }
    
    // MARK: - Stats Section
    
    private var statsSection: some View {
        HStack(spacing: 12) {
            LanciioStatCard(
                title: "Outils VIP",
                value: "\(lanciioManager.syncedToolsCount)",
                icon: "wrench.and.screwdriver.fill",
                color: lanciioYellow,
                secondaryColor: lanciioPurple
            )
            
            LanciioStatCard(
                title: "Workspaces",
                value: "\(lanciioManager.workspaces.count)",
                icon: "folder.fill",
                color: lanciioPurple,
                secondaryColor: lanciioYellow
            )
            
            LanciioStatCard(
                title: "Serveurs",
                value: "\(lanciioManager.mcpServersCount)",
                icon: "server.rack",
                color: theme.accentColor,
                secondaryColor: lanciioPurple
            )
        }
    }
    
    // MARK: - Workspaces Section
    
    private var workspacesSection: some View {
        VStack(alignment: .leading, spacing: 12) {
            HStack {
                Image(systemName: "folder.badge.gearshape")
                    .foregroundColor(lanciioPurple)
                
                Text("Workspaces")
                    .font(.system(size: 16, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                Text("\(lanciioManager.workspaces.count)")
                    .font(.system(size: 12, weight: .bold))
                    .foregroundColor(.white)
                    .padding(.horizontal, 8)
                    .padding(.vertical, 4)
                    .background(Capsule().fill(lanciioPurple))
            }
            
            ForEach(lanciioManager.workspaces) { workspace in
                WorkspaceCard(workspace: workspace, accentColor: lanciioYellow, secondaryColor: lanciioPurple)
            }
        }
    }
    
    // MARK: - Tools Search Section
    
    private var toolsSearchSection: some View {
        VStack(alignment: .leading, spacing: 12) {
            HStack {
                Image(systemName: "magnifyingglass")
                    .foregroundColor(lanciioYellow)
                
                Text("Rechercher des outils")
                    .font(.system(size: 16, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
            }
            
            HStack {
                Image(systemName: "magnifyingglass")
                    .foregroundColor(theme.textMuted)
                
                TextField("Nom de l'outil...", text: $searchQuery)
                    .textFieldStyle(.plain)
                    .foregroundColor(theme.textPrimary)
                    .onChange(of: searchQuery) { _, newValue in
                        Task {
                            searchResults = await lanciioManager.searchTools(query: newValue)
                        }
                    }
                
                if !searchQuery.isEmpty {
                    Button {
                        searchQuery = ""
                        searchResults = []
                    } label: {
                        Image(systemName: "xmark.circle.fill")
                            .foregroundColor(theme.textMuted)
                    }
                }
            }
            .padding(14)
            .background(
                RoundedRectangle(cornerRadius: 12)
                    .fill(theme.surfaceElevated)
                    .overlay(
                        RoundedRectangle(cornerRadius: 12)
                            .stroke(
                                LinearGradient(
                                    colors: [lanciioYellow.opacity(0.3), lanciioPurple.opacity(0.2)],
                                    startPoint: .leading,
                                    endPoint: .trailing
                                ),
                                lineWidth: 1
                            )
                    )
            )
            
            // Search Results
            if !searchResults.isEmpty {
                ForEach(searchResults) { tool in
                    ToolCard(tool: tool, accentColor: lanciioYellow)
                }
            } else if !searchQuery.isEmpty {
                HStack {
                    Image(systemName: "tray")
                        .foregroundColor(theme.textMuted)
                    Text("Aucun outil trouvé")
                        .font(.system(size: 13))
                        .foregroundColor(theme.textMuted)
                }
                .frame(maxWidth: .infinity)
                .padding()
            }
        }
    }
    
    // MARK: - Actions Section
    
    private var actionsSection: some View {
        VStack(spacing: 12) {
            // Sync Button
            Button {
                isSyncing = true
                Task {
                    await lanciioManager.syncTools()
                    isSyncing = false
                }
            } label: {
                HStack(spacing: 10) {
                    if isSyncing {
                        ProgressView()
                            .tint(.black)
                    } else {
                        Image(systemName: "arrow.triangle.2.circlepath")
                            .font(.system(size: 16, weight: .semibold))
                    }
                    Text(isSyncing ? "Synchronisation..." : "Synchroniser les outils VIP")
                        .font(.system(size: 14, weight: .bold))
                }
                .foregroundColor(.black)
                .frame(maxWidth: .infinity)
                .padding(.vertical, 16)
                .background(
                    LinearGradient(
                        colors: [lanciioYellow, lanciioYellow.opacity(0.8)],
                        startPoint: .leading,
                        endPoint: .trailing
                    )
                )
                .cornerRadius(14)
                .shadow(color: lanciioYellow.opacity(0.4), radius: 8, y: 4)
            }
            .disabled(isSyncing)
            
            // Disconnect Button
            Button {
                Task {
                    await lanciioManager.disconnect()
                    dismiss()
                }
            } label: {
                HStack {
                    Image(systemName: "link.badge.plus")
                        .rotationEffect(.degrees(45))
                    Text("Déconnecter")
                }
                .font(.system(size: 14, weight: .medium))
                .foregroundColor(theme.errorColor)
            }
            .padding(.top, 8)
        }
        .padding(.top, 8)
    }
}

// MARK: - Lanciio Stat Card

struct LanciioStatCard: View {
    @EnvironmentObject var theme: ThemeManager
    let title: String
    let value: String
    let icon: String
    let color: Color
    var secondaryColor: Color = .purple
    
    var body: some View {
        VStack(spacing: 10) {
            ZStack {
                Circle()
                    .fill(
                        LinearGradient(
                            colors: [color.opacity(0.2), secondaryColor.opacity(0.1)],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
                    .frame(width: 44, height: 44)
                
                Image(systemName: icon)
                    .font(.system(size: 18))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [color, secondaryColor],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
            }
            
            Text(value)
                .font(.system(size: 24, weight: .bold))
                .foregroundColor(theme.textPrimary)
            
            Text(title)
                .font(.system(size: 10, weight: .medium))
                .foregroundColor(theme.textMuted)
        }
        .frame(maxWidth: .infinity)
        .padding(.vertical, 18)
        .background(
            RoundedRectangle(cornerRadius: 16)
                .fill(theme.surfaceElevated)
                .overlay(
                    RoundedRectangle(cornerRadius: 16)
                        .stroke(
                            LinearGradient(
                                colors: [color.opacity(0.3), secondaryColor.opacity(0.15)],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            ),
                            lineWidth: 1
                        )
                )
        )
        .shadow(color: color.opacity(0.1), radius: 8, y: 4)
    }
}

// MARK: - Workspace Card

struct WorkspaceCard: View {
    @EnvironmentObject var theme: ThemeManager
    let workspace: LanciioWorkspace
    var accentColor: Color = .yellow
    var secondaryColor: Color = .purple
    @State private var isExpanded = false
    
    var body: some View {
        VStack(alignment: .leading, spacing: 0) {
            // Header
            Button {
                withAnimation(.spring(response: 0.3)) {
                    isExpanded.toggle()
                }
            } label: {
                HStack {
                    ZStack {
                        RoundedRectangle(cornerRadius: 8)
                            .fill(
                                LinearGradient(
                                    colors: [accentColor.opacity(0.2), secondaryColor.opacity(0.1)],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                            .frame(width: 36, height: 36)
                        
                        Image(systemName: "folder.fill")
                            .foregroundStyle(
                                LinearGradient(
                                    colors: [accentColor, secondaryColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                    }
                    
                    VStack(alignment: .leading, spacing: 3) {
                        Text(workspace.name)
                            .font(.system(size: 14, weight: .semibold))
                            .foregroundColor(theme.textPrimary)
                        
                        HStack(spacing: 8) {
                            Label("\(workspace.toolsCount) outils", systemImage: "wrench")
                            Label("\(workspace.servers.count) serveurs", systemImage: "server.rack")
                        }
                        .font(.system(size: 10))
                        .foregroundColor(theme.textMuted)
                    }
                    
                    Spacer()
                    
                    Image(systemName: "chevron.right")
                        .font(.system(size: 12, weight: .semibold))
                        .foregroundColor(theme.textMuted)
                        .rotationEffect(.degrees(isExpanded ? 90 : 0))
                }
                .padding(14)
            }
            .buttonStyle(.plain)
            
            // Expanded Content
            if isExpanded {
                VStack(alignment: .leading, spacing: 10) {
                    if let description = workspace.description, !description.isEmpty {
                        Text(description)
                            .font(.system(size: 12))
                            .foregroundColor(theme.textSecondary)
                    }
                    
                    // Servers
                    VStack(alignment: .leading, spacing: 6) {
                        Text("Serveurs MCP")
                            .font(.system(size: 11, weight: .semibold))
                            .foregroundColor(theme.textMuted)
                        
                        FlowLayout(spacing: 6) {
                            ForEach(workspace.servers, id: \.self) { server in
                                HStack(spacing: 4) {
                                    Circle()
                                        .fill(accentColor)
                                        .frame(width: 4, height: 4)
                                    Text(server)
                                }
                                .font(.system(size: 10, weight: .medium))
                                .foregroundColor(accentColor)
                                .padding(.horizontal, 10)
                                .padding(.vertical, 5)
                                .background(
                                    Capsule()
                                        .fill(accentColor.opacity(0.15))
                                        .overlay(
                                            Capsule()
                                                .stroke(accentColor.opacity(0.3), lineWidth: 0.5)
                                        )
                                )
                            }
                        }
                    }
                }
                .padding(.horizontal, 14)
                .padding(.bottom, 14)
            }
        }
        .background(
            RoundedRectangle(cornerRadius: 14)
                .fill(theme.surfaceElevated)
                .shadow(color: .black.opacity(0.1), radius: 4, y: 2)
        )
    }
}

// MARK: - Tool Card

struct ToolCard: View {
    @EnvironmentObject var theme: ThemeManager
    let tool: LanciioTool
    var accentColor: Color = .yellow
    
    var body: some View {
        HStack(spacing: 12) {
            ZStack {
                RoundedRectangle(cornerRadius: 10)
                    .fill(accentColor.opacity(0.15))
                    .frame(width: 40, height: 40)
                
                Image(systemName: "wrench.adjustable.fill")
                    .font(.system(size: 18))
                    .foregroundColor(accentColor)
            }
            
            VStack(alignment: .leading, spacing: 3) {
                Text(tool.name)
                    .font(.system(size: 13, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
                
                Text(tool.description)
                    .font(.system(size: 11))
                    .foregroundColor(theme.textSecondary)
                    .lineLimit(2)
                
                HStack(spacing: 6) {
                    Label(tool.serverName, systemImage: "server.rack")
                    Text("•")
                    Label(tool.workspaceName, systemImage: "folder")
                }
                .font(.system(size: 9))
                .foregroundColor(theme.textMuted)
            }
            
            Spacer()
            
            // VIP indicator
            Image(systemName: "crown.fill")
                .font(.system(size: 10))
                .foregroundColor(accentColor)
        }
        .padding(12)
        .background(
            RoundedRectangle(cornerRadius: 12)
                .fill(theme.surfaceElevated)
                .overlay(
                    RoundedRectangle(cornerRadius: 12)
                        .stroke(accentColor.opacity(0.2), lineWidth: 1)
                )
        )
    }
}

// MARK: - Flow Layout

struct FlowLayout: Layout {
    var spacing: CGFloat = 8
    
    func sizeThatFits(proposal: ProposedViewSize, subviews: Subviews, cache: inout ()) -> CGSize {
        let result = FlowResult(
            in: proposal.width ?? 0,
            spacing: spacing,
            subviews: subviews
        )
        return result.size
    }
    
    func placeSubviews(in bounds: CGRect, proposal: ProposedViewSize, subviews: Subviews, cache: inout ()) {
        let result = FlowResult(
            in: bounds.width,
            spacing: spacing,
            subviews: subviews
        )
        for (index, subview) in subviews.enumerated() {
            subview.place(at: CGPoint(x: bounds.minX + result.positions[index].x,
                                      y: bounds.minY + result.positions[index].y),
                         proposal: .unspecified)
        }
    }
    
    struct FlowResult {
        var size: CGSize = .zero
        var positions: [CGPoint] = []
        
        init(in maxWidth: CGFloat, spacing: CGFloat, subviews: Subviews) {
            var x: CGFloat = 0
            var y: CGFloat = 0
            var rowHeight: CGFloat = 0
            
            for subview in subviews {
                let size = subview.sizeThatFits(.unspecified)
                
                if x + size.width > maxWidth && x > 0 {
                    x = 0
                    y += rowHeight + spacing
                    rowHeight = 0
                }
                
                positions.append(CGPoint(x: x, y: y))
                rowHeight = max(rowHeight, size.height)
                x += size.width + spacing
            }
            
            self.size = CGSize(width: maxWidth, height: y + rowHeight)
        }
    }
}

// MARK: - Preview

#Preview {
    VStack(spacing: 20) {
        LanciioConnectButton()
        
        LanciioConnectButton()
    }
    .environmentObject(ThemeManager())
    .padding()
    .background(Color.black)
}
