//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Collaboration History Panel                                  ║
//  ║         Real-Time Task Visualization with Full History                        ║
//  ║         WebSocket/tRPC Mutations & Step Navigation                            ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import Combine

// MARK: - Collaboration Step Types

enum CollaborationStepType: String, Codable, CaseIterable {
    case analyzing = "analyzing"
    case routing = "routing"
    case planning = "planning"
    case executing = "executing"
    case toolExecution = "tool_execution"
    case shellCommand = "shell_command"
    case apiCall = "api_call"
    case webScraping = "web_scraping"
    case documentRead = "document_read"
    case documentWrite = "document_write"
    case webSearch = "web_search"
    case databaseQuery = "database_query"
    case fileOperation = "file_operation"
    case imageProcessing = "image_processing"
    case codeGeneration = "code_generation"
    case codeExecution = "code_execution"
    case fallback = "fallback"
    case retry = "retry"
    case validation = "validation"
    case reformulation = "reformulation"
    case collaboration = "collaboration"
    case completion = "completion"
    case error = "error"
    
    var icon: String {
        switch self {
        case .analyzing: return "magnifyingglass.circle.fill"
        case .routing: return "arrow.triangle.branch"
        case .planning: return "map.fill"
        case .executing: return "bolt.fill"
        case .toolExecution: return "wrench.and.screwdriver.fill"
        case .shellCommand: return "terminal.fill"
        case .apiCall: return "network"
        case .webScraping: return "globe"
        case .documentRead: return "doc.text.magnifyingglass"
        case .documentWrite: return "doc.badge.plus"
        case .webSearch: return "magnifyingglass"
        case .databaseQuery: return "cylinder.fill"
        case .fileOperation: return "folder.fill"
        case .imageProcessing: return "photo.fill"
        case .codeGeneration: return "chevron.left.forwardslash.chevron.right"
        case .codeExecution: return "play.fill"
        case .fallback: return "arrow.uturn.backward.circle.fill"
        case .retry: return "arrow.clockwise"
        case .validation: return "checkmark.shield.fill"
        case .reformulation: return "sparkles"
        case .collaboration: return "person.3.fill"
        case .completion: return "checkmark.circle.fill"
        case .error: return "xmark.octagon.fill"
        }
    }
    
    var color: Color {
        switch self {
        case .analyzing: return .blue
        case .routing: return .purple
        case .planning: return .indigo
        case .executing: return .orange
        case .toolExecution: return .cyan
        case .shellCommand: return Color(red: 0.2, green: 0.8, blue: 0.4)
        case .apiCall: return .teal
        case .webScraping: return .mint
        case .documentRead: return .blue.opacity(0.8)
        case .documentWrite: return .green.opacity(0.8)
        case .webSearch: return .yellow
        case .databaseQuery: return Color(red: 0.6, green: 0.4, blue: 0.8)
        case .fileOperation: return .brown
        case .imageProcessing: return .pink
        case .codeGeneration: return Color(red: 0.9, green: 0.5, blue: 0.2)
        case .codeExecution: return .green
        case .fallback: return .orange.opacity(0.8)
        case .retry: return .yellow.opacity(0.8)
        case .validation: return .mint
        case .reformulation: return .yellow
        case .collaboration: return .cyan
        case .completion: return .green
        case .error: return .red
        }
    }
    
    var displayName: String {
        switch self {
        case .analyzing: return "Analyse"
        case .routing: return "Routage"
        case .planning: return "Planification"
        case .executing: return "Exécution"
        case .toolExecution: return "Outil"
        case .shellCommand: return "Commande Shell"
        case .apiCall: return "Appel API"
        case .webScraping: return "Scraping Web"
        case .documentRead: return "Lecture Document"
        case .documentWrite: return "Écriture Document"
        case .webSearch: return "Recherche Web"
        case .databaseQuery: return "Requête BDD"
        case .fileOperation: return "Opération Fichier"
        case .imageProcessing: return "Traitement Image"
        case .codeGeneration: return "Génération Code"
        case .codeExecution: return "Exécution Code"
        case .fallback: return "Fallback"
        case .retry: return "Nouvelle Tentative"
        case .validation: return "Validation"
        case .reformulation: return "Reformulation"
        case .collaboration: return "Collaboration"
        case .completion: return "Terminé"
        case .error: return "Erreur"
        }
    }
}

// MARK: - Collaboration Step Model

struct CollaborationStep: Identifiable, Equatable {
    let id: String
    let type: CollaborationStepType
    let title: String
    let description: String
    let agent: String?
    let tool: String?
    let parameters: [String: Any]?
    let output: String?
    let outputType: String?
    let duration: TimeInterval?
    let timestamp: Date
    var status: StepStatus
    var subSteps: [CollaborationStep]
    
    enum StepStatus: String, Codable {
        case pending = "pending"
        case inProgress = "in_progress"
        case completed = "completed"
        case failed = "failed"
        case skipped = "skipped"
    }
    
    static func == (lhs: CollaborationStep, rhs: CollaborationStep) -> Bool {
        lhs.id == rhs.id && lhs.status == rhs.status
    }
    
    init(
        id: String = UUID().uuidString,
        type: CollaborationStepType,
        title: String,
        description: String,
        agent: String? = nil,
        tool: String? = nil,
        parameters: [String: Any]? = nil,
        output: String? = nil,
        outputType: String? = nil,
        duration: TimeInterval? = nil,
        timestamp: Date = Date(),
        status: StepStatus = .pending,
        subSteps: [CollaborationStep] = []
    ) {
        self.id = id
        self.type = type
        self.title = title
        self.description = description
        self.agent = agent
        self.tool = tool
        self.parameters = parameters
        self.output = output
        self.outputType = outputType
        self.duration = duration
        self.timestamp = timestamp
        self.status = status
        self.subSteps = subSteps
    }
}

// MARK: - Collaboration Session

class CollaborationSession: ObservableObject, Identifiable {
    let id: String
    let requestId: String
    let query: String
    let startTime: Date
    
    @Published var steps: [CollaborationStep] = []
    @Published var currentStepIndex: Int = 0
    @Published var status: SessionStatus = .active
    @Published var progress: Double = 0
    @Published var endTime: Date?
    @Published var finalOutput: String?
    
    enum SessionStatus: String {
        case active = "active"
        case completed = "completed"
        case failed = "failed"
        case cancelled = "cancelled"
    }
    
    init(id: String = UUID().uuidString, requestId: String, query: String) {
        self.id = id
        self.requestId = requestId
        self.query = query
        self.startTime = Date()
    }
    
    var totalDuration: TimeInterval {
        (endTime ?? Date()).timeIntervalSince(startTime)
    }
    
    var completedSteps: Int {
        steps.filter { $0.status == .completed }.count
    }
}

// MARK: - Collaboration History Manager

class CollaborationHistoryManager: ObservableObject {
    static let shared = CollaborationHistoryManager()
    
    @Published var sessions: [CollaborationSession] = []
    @Published var currentSession: CollaborationSession?
    @Published var selectedStepId: String?
    @Published var isExpanded = false
    
    private var cancellables = Set<AnyCancellable>()
    
    private init() {
        loadHistory()
    }
    
    // MARK: - Session Management
    
    func startSession(requestId: String, query: String) -> CollaborationSession {
        let session = CollaborationSession(requestId: requestId, query: query)
        currentSession = session
        return session
    }
    
    func endSession(withOutput output: String? = nil, status: CollaborationSession.SessionStatus = .completed) {
        currentSession?.status = status
        currentSession?.endTime = Date()
        currentSession?.finalOutput = output
        currentSession?.progress = 1.0
        
        if let session = currentSession {
            sessions.insert(session, at: 0)
            // Keep only last 50 sessions
            if sessions.count > 50 {
                sessions = Array(sessions.prefix(50))
            }
            saveHistory()
        }
        
        // Keep session visible briefly before clearing
        DispatchQueue.main.asyncAfter(deadline: .now() + 2) { [weak self] in
            self?.currentSession = nil
        }
    }
    
    // MARK: - Step Management
    
    func addStep(_ step: CollaborationStep) {
        guard let session = currentSession else { return }
        
        // Mark previous step as completed if it was in progress
        if let lastIndex = session.steps.lastIndex(where: { $0.status == .inProgress }) {
            session.steps[lastIndex].status = .completed
        }
        
        session.steps.append(step)
        session.currentStepIndex = session.steps.count - 1
        
        // Update progress
        let totalExpectedSteps = max(session.steps.count, 5)
        session.progress = min(Double(session.steps.count) / Double(totalExpectedSteps), 0.95)
        
        // Auto-select new step
        selectedStepId = step.id
    }
    
    func updateStep(id: String, status: CollaborationStep.StepStatus, output: String? = nil, duration: TimeInterval? = nil) {
        guard let session = currentSession,
              let index = session.steps.firstIndex(where: { $0.id == id }) else { return }
        
        var step = session.steps[index]
        step.status = status
        if let output = output {
            session.steps[index] = CollaborationStep(
                id: step.id,
                type: step.type,
                title: step.title,
                description: step.description,
                agent: step.agent,
                tool: step.tool,
                parameters: step.parameters,
                output: output,
                outputType: step.outputType,
                duration: duration,
                timestamp: step.timestamp,
                status: status,
                subSteps: step.subSteps
            )
        } else {
            session.steps[index].status = status
        }
    }
    
    func addSubStep(toStepId parentId: String, subStep: CollaborationStep) {
        guard let session = currentSession,
              let index = session.steps.firstIndex(where: { $0.id == parentId }) else { return }
        
        session.steps[index].subSteps.append(subStep)
    }
    
    // MARK: - Persistence
    
    private func saveHistory() {
        // Save to UserDefaults (simplified - could use CoreData for production)
        // For now, we just keep in memory
    }
    
    private func loadHistory() {
        // Load from storage
    }
    
    func clearHistory() {
        sessions.removeAll()
    }
}

// MARK: - Collaboration History Panel View

struct CollaborationHistoryPanel: View {
    @EnvironmentObject var theme: ThemeManager
    @StateObject private var historyManager = CollaborationHistoryManager.shared
    @State private var showHistory = false
    @State private var selectedHistorySession: CollaborationSession?
    
    var body: some View {
        VStack(spacing: 0) {
            // Header
            panelHeader
            
            // Current Session or History
            if let session = historyManager.currentSession {
                currentSessionView(session)
            } else if showHistory {
                historyListView
            } else {
                emptyStateView
            }
        }
        .background(
            RoundedRectangle(cornerRadius: 20)
                .fill(theme.surfaceColor)
                .overlay(
                    RoundedRectangle(cornerRadius: 20)
                        .stroke(
                            LinearGradient(
                                colors: [theme.primaryColor.opacity(0.4), theme.accentColor.opacity(0.2)],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            ),
                            lineWidth: 1.5
                        )
                )
        )
        .shadow(color: theme.primaryDark.opacity(0.3), radius: 15, y: 8)
    }
    
    // MARK: - Panel Header
    
    private var panelHeader: some View {
        HStack(spacing: 12) {
            // Icon with animation
            ZStack {
                Circle()
                    .fill(theme.primaryColor.opacity(0.2))
                    .frame(width: 44, height: 44)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 20, weight: .semibold))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [theme.accentColor, theme.primaryLight],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
            }
            
            VStack(alignment: .leading, spacing: 2) {
                Text("🧠 Collaboration AI")
                    .font(.system(size: 16, weight: .bold))
                    .foregroundColor(theme.textPrimary)
                
                if let session = historyManager.currentSession {
                    HStack(spacing: 6) {
                        Circle()
                            .fill(Color.green)
                            .frame(width: 8, height: 8)
                        
                        Text("\(session.steps.count) étapes")
                            .font(.system(size: 11))
                            .foregroundColor(theme.textMuted)
                        
                        Text("•")
                            .foregroundColor(theme.textMuted)
                        
                        Text(formatDuration(session.totalDuration))
                            .font(.system(size: 11, weight: .medium))
                            .foregroundColor(theme.accentColor)
                    }
                } else {
                    Text("Historique des tâches")
                        .font(.system(size: 11))
                        .foregroundColor(theme.textMuted)
                }
            }
            
            Spacer()
            
            // History toggle
            Button(action: { withAnimation(.spring()) { showHistory.toggle() } }) {
                Image(systemName: showHistory ? "clock.fill" : "clock")
                    .font(.system(size: 18))
                    .foregroundColor(showHistory ? theme.accentColor : theme.textSecondary)
                    .padding(8)
                    .background(showHistory ? theme.accentColor.opacity(0.2) : Color.clear)
                    .cornerRadius(10)
            }
            
            // Expand button
            Button(action: { withAnimation(.spring()) { historyManager.isExpanded.toggle() } }) {
                Image(systemName: historyManager.isExpanded ? "chevron.down.circle.fill" : "chevron.up.circle.fill")
                    .font(.system(size: 22))
                    .foregroundColor(theme.textSecondary)
            }
        }
        .padding(16)
        .background(theme.surfaceElevated)
    }
    
    // MARK: - Current Session View
    
    @ViewBuilder
    private func currentSessionView(_ session: CollaborationSession) -> some View {
        VStack(spacing: 0) {
            // Progress bar
            GeometryReader { geo in
                ZStack(alignment: .leading) {
                    Rectangle()
                        .fill(theme.surfaceElevated)
                    
                    Rectangle()
                        .fill(
                            LinearGradient(
                                colors: [theme.primaryColor, theme.accentColor],
                                startPoint: .leading,
                                endPoint: .trailing
                            )
                        )
                        .frame(width: geo.size.width * session.progress)
                        .animation(.easeInOut(duration: 0.3), value: session.progress)
                }
            }
            .frame(height: 4)
            
            // Query summary
            HStack {
                Image(systemName: "text.bubble.fill")
                    .font(.system(size: 12))
                    .foregroundColor(theme.accentColor)
                
                Text(session.query)
                    .font(.system(size: 12))
                    .foregroundColor(theme.textSecondary)
                    .lineLimit(1)
                
                Spacer()
            }
            .padding(.horizontal, 16)
            .padding(.vertical, 8)
            .background(theme.backgroundColor.opacity(0.5))
            
            // Steps timeline
            if historyManager.isExpanded {
                ScrollViewReader { proxy in
                    ScrollView {
                        LazyVStack(spacing: 0) {
                            ForEach(Array(session.steps.enumerated()), id: \.element.id) { index, step in
                                CollaborationStepRow(
                                    step: step,
                                    isSelected: historyManager.selectedStepId == step.id,
                                    isLast: index == session.steps.count - 1,
                                    onTap: { historyManager.selectedStepId = step.id }
                                )
                                .id(step.id)
                            }
                        }
                        .padding(.vertical, 8)
                    }
                    .frame(maxHeight: 400)
                    .onChange(of: session.steps.count) { _ in
                        if let lastStep = session.steps.last {
                            withAnimation(.easeInOut) {
                                proxy.scrollTo(lastStep.id, anchor: .bottom)
                            }
                        }
                    }
                }
                
                // Step detail view
                if let selectedId = historyManager.selectedStepId,
                   let step = session.steps.first(where: { $0.id == selectedId }) {
                    StepDetailView(step: step)
                }
            } else {
                // Compact view - show last 3 steps
                VStack(spacing: 0) {
                    ForEach(session.steps.suffix(3)) { step in
                        CompactStepRow(step: step)
                    }
                }
                .padding(.vertical, 8)
            }
        }
    }
    
    // MARK: - History List View
    
    private var historyListView: some View {
        ScrollView {
            LazyVStack(spacing: 12) {
                if historyManager.sessions.isEmpty {
                    VStack(spacing: 12) {
                        Image(systemName: "clock.badge.questionmark")
                            .font(.system(size: 40))
                            .foregroundColor(theme.textMuted)
                        
                        Text("Aucun historique")
                            .font(.system(size: 14))
                            .foregroundColor(theme.textMuted)
                    }
                    .frame(maxWidth: .infinity)
                    .padding(40)
                } else {
                    ForEach(historyManager.sessions) { session in
                        HistorySessionCard(session: session)
                            .onTapGesture {
                                withAnimation(.spring()) {
                                    selectedHistorySession = session
                                }
                            }
                    }
                }
            }
            .padding(16)
        }
        .frame(maxHeight: historyManager.isExpanded ? 500 : 200)
        .sheet(item: $selectedHistorySession) { session in
            HistoryDetailSheet(session: session)
        }
    }
    
    // MARK: - Empty State
    
    private var emptyStateView: some View {
        VStack(spacing: 16) {
            ZStack {
                Circle()
                    .fill(theme.primaryColor.opacity(0.1))
                    .frame(width: 80, height: 80)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 36))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [theme.accentColor.opacity(0.5), theme.primaryLight.opacity(0.5)],
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
            }
            
            Text("En attente d'une tâche...")
                .font(.system(size: 14))
                .foregroundColor(theme.textMuted)
            
            Text("L'orchestrateur AI affichera ici les étapes en temps réel")
                .font(.system(size: 12))
                .foregroundColor(theme.textMuted.opacity(0.7))
                .multilineTextAlignment(.center)
        }
        .frame(maxWidth: .infinity)
        .padding(32)
    }
    
    // MARK: - Helpers
    
    private func formatDuration(_ duration: TimeInterval) -> String {
        if duration < 1 {
            return "\(Int(duration * 1000))ms"
        } else if duration < 60 {
            return String(format: "%.1fs", duration)
        } else {
            let minutes = Int(duration) / 60
            let seconds = Int(duration) % 60
            return "\(minutes)m \(seconds)s"
        }
    }
}

// MARK: - Collaboration Step Row

struct CollaborationStepRow: View {
    @EnvironmentObject var theme: ThemeManager
    let step: CollaborationStep
    let isSelected: Bool
    let isLast: Bool
    let onTap: () -> Void
    
    @State private var appear = false
    
    var body: some View {
        HStack(alignment: .top, spacing: 12) {
            // Timeline connector
            VStack(spacing: 0) {
                // Step icon
                ZStack {
                    Circle()
                        .fill(step.type.color.opacity(isSelected ? 0.3 : 0.15))
                        .frame(width: 40, height: 40)
                    
                    if step.status == .inProgress {
                        Circle()
                            .stroke(step.type.color, lineWidth: 2)
                            .frame(width: 40, height: 40)
                            .rotationEffect(.degrees(appear ? 360 : 0))
                            .animation(.linear(duration: 1).repeatForever(autoreverses: false), value: appear)
                    }
                    
                    Image(systemName: step.type.icon)
                        .font(.system(size: 16, weight: .semibold))
                        .foregroundColor(step.type.color)
                }
                
                // Connector line
                if !isLast {
                    Rectangle()
                        .fill(
                            LinearGradient(
                                colors: [step.type.color.opacity(0.4), theme.surfaceElevated],
                                startPoint: .top,
                                endPoint: .bottom
                            )
                        )
                        .frame(width: 2, height: 24)
                }
            }
            
            // Content
            VStack(alignment: .leading, spacing: 6) {
                HStack {
                    Text(step.title)
                        .font(.system(size: 13, weight: .semibold))
                        .foregroundColor(theme.textPrimary)
                    
                    Spacer()
                    
                    // Status badge
                    StepStatusBadge(status: step.status)
                }
                
                Text(step.description)
                    .font(.system(size: 11))
                    .foregroundColor(theme.textSecondary)
                    .lineLimit(isSelected ? nil : 2)
                
                // Tool/Agent badges
                HStack(spacing: 8) {
                    if let tool = step.tool {
                        HStack(spacing: 4) {
                            Image(systemName: "wrench.fill")
                                .font(.system(size: 9))
                            Text(tool)
                                .font(.system(size: 9, weight: .medium))
                        }
                        .foregroundColor(.cyan)
                        .padding(.horizontal, 8)
                        .padding(.vertical, 3)
                        .background(Color.cyan.opacity(0.15))
                        .cornerRadius(6)
                    }
                    
                    if let agent = step.agent {
                        HStack(spacing: 4) {
                            Image(systemName: "person.fill")
                                .font(.system(size: 9))
                            Text(agent)
                                .font(.system(size: 9, weight: .medium))
                        }
                        .foregroundColor(step.type.color)
                        .padding(.horizontal, 8)
                        .padding(.vertical, 3)
                        .background(step.type.color.opacity(0.15))
                        .cornerRadius(6)
                    }
                    
                    if let duration = step.duration {
                        Text(String(format: "%.0fms", duration * 1000))
                            .font(.system(size: 9, weight: .medium))
                            .foregroundColor(theme.textMuted)
                    }
                }
                
                // Sub-steps indicator
                if !step.subSteps.isEmpty {
                    HStack(spacing: 4) {
                        Image(systemName: "list.bullet.indent")
                            .font(.system(size: 10))
                        Text("\(step.subSteps.count) sous-étapes")
                            .font(.system(size: 10))
                    }
                    .foregroundColor(theme.textMuted)
                }
            }
            .padding(.vertical, 8)
        }
        .padding(.horizontal, 16)
        .background(isSelected ? theme.surfaceElevated.opacity(0.5) : Color.clear)
        .contentShape(Rectangle())
        .onTapGesture(perform: onTap)
        .opacity(appear ? 1 : 0)
        .offset(x: appear ? 0 : -20)
        .onAppear {
            withAnimation(.spring(response: 0.4, dampingFraction: 0.7).delay(0.1)) {
                appear = true
            }
        }
    }
}

// MARK: - Compact Step Row

struct CompactStepRow: View {
    @EnvironmentObject var theme: ThemeManager
    let step: CollaborationStep
    
    var body: some View {
        HStack(spacing: 10) {
            Image(systemName: step.type.icon)
                .font(.system(size: 12))
                .foregroundColor(step.type.color)
                .frame(width: 24)
            
            Text(step.title)
                .font(.system(size: 12))
                .foregroundColor(theme.textPrimary)
                .lineLimit(1)
            
            Spacer()
            
            StepStatusBadge(status: step.status, compact: true)
        }
        .padding(.horizontal, 16)
        .padding(.vertical, 6)
    }
}

// MARK: - Step Status Badge

struct StepStatusBadge: View {
    @EnvironmentObject var theme: ThemeManager
    let status: CollaborationStep.StepStatus
    var compact: Bool = false
    
    var body: some View {
        HStack(spacing: compact ? 2 : 4) {
            switch status {
            case .pending:
                Circle()
                    .fill(theme.textMuted)
                    .frame(width: 6, height: 6)
                if !compact {
                    Text("En attente")
                        .font(.system(size: 9))
                        .foregroundColor(theme.textMuted)
                }
                
            case .inProgress:
                ProgressView()
                    .scaleEffect(0.5)
                    .frame(width: 12, height: 12)
                if !compact {
                    Text("En cours")
                        .font(.system(size: 9))
                        .foregroundColor(theme.accentColor)
                }
                
            case .completed:
                Image(systemName: "checkmark.circle.fill")
                    .font(.system(size: compact ? 10 : 12))
                    .foregroundColor(.green)
                if !compact {
                    Text("Terminé")
                        .font(.system(size: 9))
                        .foregroundColor(.green)
                }
                
            case .failed:
                Image(systemName: "xmark.circle.fill")
                    .font(.system(size: compact ? 10 : 12))
                    .foregroundColor(.red)
                if !compact {
                    Text("Erreur")
                        .font(.system(size: 9))
                        .foregroundColor(.red)
                }
                
            case .skipped:
                Image(systemName: "arrow.uturn.forward.circle")
                    .font(.system(size: compact ? 10 : 12))
                    .foregroundColor(theme.textMuted)
                if !compact {
                    Text("Ignoré")
                        .font(.system(size: 9))
                        .foregroundColor(theme.textMuted)
                }
            }
        }
    }
}

// MARK: - Step Detail View

struct StepDetailView: View {
    @EnvironmentObject var theme: ThemeManager
    let step: CollaborationStep
    
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            // Header
            HStack {
                Image(systemName: step.type.icon)
                    .font(.system(size: 16))
                    .foregroundColor(step.type.color)
                
                Text(step.type.displayName)
                    .font(.system(size: 14, weight: .bold))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                Text(step.timestamp, style: .time)
                    .font(.system(size: 11))
                    .foregroundColor(theme.textMuted)
            }
            
            Divider()
                .background(theme.surfaceElevated)
            
            // Parameters if available
            if let params = step.parameters, !params.isEmpty {
                VStack(alignment: .leading, spacing: 6) {
                    Text("Paramètres")
                        .font(.system(size: 11, weight: .semibold))
                        .foregroundColor(theme.textMuted)
                    
                    ForEach(Array(params.keys.sorted()), id: \.self) { key in
                        HStack(alignment: .top, spacing: 8) {
                            Text(key)
                                .font(.system(size: 11, weight: .medium))
                                .foregroundColor(theme.accentColor)
                            
                            Text(String(describing: params[key] ?? "nil"))
                                .font(.system(size: 11, design: .monospaced))
                                .foregroundColor(theme.textSecondary)
                                .lineLimit(3)
                        }
                    }
                }
                .padding(10)
                .background(theme.backgroundColor.opacity(0.5))
                .cornerRadius(8)
            }
            
            // Output if available
            if let output = step.output {
                VStack(alignment: .leading, spacing: 6) {
                    HStack {
                        Text("Sortie")
                            .font(.system(size: 11, weight: .semibold))
                            .foregroundColor(theme.textMuted)
                        
                        if let outputType = step.outputType {
                            Text("(\(outputType))")
                                .font(.system(size: 10))
                                .foregroundColor(theme.textMuted.opacity(0.7))
                        }
                    }
                    
                    ScrollView {
                        Text(output)
                            .font(.system(size: 11, design: .monospaced))
                            .foregroundColor(theme.textPrimary)
                            .textSelection(.enabled)
                    }
                    .frame(maxHeight: 150)
                }
                .padding(10)
                .background(theme.backgroundColor.opacity(0.5))
                .cornerRadius(8)
            }
            
            // Sub-steps
            if !step.subSteps.isEmpty {
                VStack(alignment: .leading, spacing: 8) {
                    Text("Sous-étapes")
                        .font(.system(size: 11, weight: .semibold))
                        .foregroundColor(theme.textMuted)
                    
                    ForEach(step.subSteps) { subStep in
                        HStack(spacing: 8) {
                            Image(systemName: subStep.type.icon)
                                .font(.system(size: 10))
                                .foregroundColor(subStep.type.color)
                            
                            Text(subStep.title)
                                .font(.system(size: 11))
                                .foregroundColor(theme.textPrimary)
                            
                            Spacer()
                            
                            StepStatusBadge(status: subStep.status, compact: true)
                        }
                        .padding(.vertical, 4)
                    }
                }
            }
        }
        .padding(16)
        .background(theme.surfaceElevated)
        .cornerRadius(12)
        .padding(.horizontal, 16)
        .padding(.bottom, 16)
    }
}

// MARK: - History Session Card

struct HistorySessionCard: View {
    @EnvironmentObject var theme: ThemeManager
    let session: CollaborationSession
    
    var body: some View {
        VStack(alignment: .leading, spacing: 10) {
            HStack {
                // Status icon
                Image(systemName: session.status == .completed ? "checkmark.circle.fill" : "xmark.circle.fill")
                    .foregroundColor(session.status == .completed ? .green : .red)
                
                Text(session.query)
                    .font(.system(size: 13, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                    .lineLimit(1)
                
                Spacer()
                
                Text(session.startTime, style: .relative)
                    .font(.system(size: 10))
                    .foregroundColor(theme.textMuted)
            }
            
            HStack(spacing: 16) {
                HStack(spacing: 4) {
                    Image(systemName: "list.number")
                        .font(.system(size: 10))
                    Text("\(session.steps.count) étapes")
                        .font(.system(size: 10))
                }
                .foregroundColor(theme.textMuted)
                
                HStack(spacing: 4) {
                    Image(systemName: "clock")
                        .font(.system(size: 10))
                    Text(formatDuration(session.totalDuration))
                        .font(.system(size: 10))
                }
                .foregroundColor(theme.textMuted)
            }
        }
        .padding(14)
        .background(theme.surfaceElevated)
        .cornerRadius(12)
    }
    
    private func formatDuration(_ duration: TimeInterval) -> String {
        if duration < 60 {
            return String(format: "%.1fs", duration)
        } else {
            return String(format: "%.0fm", duration / 60)
        }
    }
}

// MARK: - History Detail Sheet

struct HistoryDetailSheet: View {
    @EnvironmentObject var theme: ThemeManager
    @Environment(\.dismiss) var dismiss
    let session: CollaborationSession
    
    var body: some View {
        NavigationView {
            ScrollView {
                VStack(alignment: .leading, spacing: 16) {
                    // Query
                    VStack(alignment: .leading, spacing: 8) {
                        Text("Requête")
                            .font(.system(size: 12, weight: .semibold))
                            .foregroundColor(theme.textMuted)
                        
                        Text(session.query)
                            .font(.system(size: 14))
                            .foregroundColor(theme.textPrimary)
                    }
                    .padding(16)
                    .frame(maxWidth: .infinity, alignment: .leading)
                    .background(theme.surfaceElevated)
                    .cornerRadius(12)
                    
                    // Steps
                    VStack(alignment: .leading, spacing: 12) {
                        Text("Étapes (\(session.steps.count))")
                            .font(.system(size: 12, weight: .semibold))
                            .foregroundColor(theme.textMuted)
                        
                        ForEach(Array(session.steps.enumerated()), id: \.element.id) { index, step in
                            HistoryStepCard(step: step, index: index + 1)
                        }
                    }
                    .padding(16)
                    .frame(maxWidth: .infinity, alignment: .leading)
                    .background(theme.surfaceElevated)
                    .cornerRadius(12)
                    
                    // Final output
                    if let output = session.finalOutput {
                        VStack(alignment: .leading, spacing: 8) {
                            Text("Résultat Final")
                                .font(.system(size: 12, weight: .semibold))
                                .foregroundColor(theme.textMuted)
                            
                            Text(output)
                                .font(.system(size: 13))
                                .foregroundColor(theme.textPrimary)
                        }
                        .padding(16)
                        .frame(maxWidth: .infinity, alignment: .leading)
                        .background(theme.surfaceElevated)
                        .cornerRadius(12)
                    }
                }
                .padding(16)
            }
            .background(theme.backgroundColor)
            .navigationTitle("Détails de la session")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .navigationBarTrailing) {
                    Button("Fermer") { dismiss() }
                }
            }
        }
    }
}

// MARK: - History Step Card

struct HistoryStepCard: View {
    @EnvironmentObject var theme: ThemeManager
    let step: CollaborationStep
    let index: Int
    
    @State private var expanded = false
    
    var body: some View {
        VStack(alignment: .leading, spacing: 8) {
            HStack {
                Text("\(index)")
                    .font(.system(size: 11, weight: .bold))
                    .foregroundColor(.white)
                    .frame(width: 22, height: 22)
                    .background(step.type.color)
                    .cornerRadius(6)
                
                Image(systemName: step.type.icon)
                    .font(.system(size: 14))
                    .foregroundColor(step.type.color)
                
                Text(step.title)
                    .font(.system(size: 13, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                StepStatusBadge(status: step.status, compact: true)
                
                Button(action: { withAnimation { expanded.toggle() } }) {
                    Image(systemName: expanded ? "chevron.up" : "chevron.down")
                        .font(.system(size: 12))
                        .foregroundColor(theme.textMuted)
                }
            }
            
            if expanded {
                VStack(alignment: .leading, spacing: 8) {
                    Text(step.description)
                        .font(.system(size: 12))
                        .foregroundColor(theme.textSecondary)
                    
                    if let output = step.output {
                        Text(output)
                            .font(.system(size: 11, design: .monospaced))
                            .foregroundColor(theme.textPrimary)
                            .padding(8)
                            .background(theme.backgroundColor)
                            .cornerRadius(6)
                    }
                }
                .padding(.leading, 30)
            }
        }
        .padding(12)
        .background(theme.backgroundColor.opacity(0.5))
        .cornerRadius(10)
    }
}

// MARK: - Preview

#Preview {
    ZStack {
        Color.black.ignoresSafeArea()
        
        CollaborationHistoryPanel()
            .padding()
    }
    .environmentObject(ThemeManager.shared)
}
