//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Action Confirmation Views                                   ║
//  ║         UI for confirming sensitive Odoo operations                           ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

// MARK: - Action Confirmation Sheet

struct ActionConfirmationSheet: View {
    @EnvironmentObject var theme: ThemeManager
    @Environment(\.dismiss) var dismiss
    
    let action: QueuedOdooAction
    let onConfirm: () -> Void
    let onCancel: () -> Void
    
    @State private var isConfirming = false
    
    var body: some View {
        NavigationStack {
            ZStack {
                theme.backgroundGradient.ignoresSafeArea()
                
                ScrollView {
                    VStack(spacing: 24) {
                        // Warning Icon
                        warningHeader
                        
                        // Action Details
                        actionDetailsCard
                        
                        // Classification Info
                        classificationCard
                        
                        // Warning Message
                        if let warning = warningMessage {
                            warningMessageView(warning)
                        }
                        
                        // Action Buttons
                        actionButtons
                    }
                    .padding()
                }
            }
            .navigationTitle("Confirmation requise")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .topBarTrailing) {
                    Button {
                        dismiss()
                        onCancel()
                    } label: {
                        Image(systemName: "xmark.circle.fill")
                            .foregroundStyle(theme.textMuted)
                    }
                }
            }
        }
        .presentationDetents([.medium, .large])
        .presentationDragIndicator(.visible)
    }
    
    private var warningMessage: String? {
        switch action.sensitivityLevel {
        case "critical":
            return "⚠️ **ACTION CRITIQUE** sur `\(action.model)` - Cette action peut avoir des conséquences importantes."
        case "high":
            return "⚡ **Action sensible** sur `\(action.model)` - Cette action modifie des données importantes."
        default:
            return nil
        }
    }
    
    // MARK: - Warning Header
    
    private var warningHeader: some View {
        VStack(spacing: 16) {
            ZStack {
                // Animated rings
                ForEach(0..<3, id: \.self) { i in
                    Circle()
                        .stroke(action.sensitivityColor.opacity(0.3 - Double(i) * 0.1), lineWidth: 2)
                        .frame(width: CGFloat(80 + i * 20), height: CGFloat(80 + i * 20))
                }
                
                // Icon
                ZStack {
                    Circle()
                        .fill(action.sensitivityColor.opacity(0.2))
                        .frame(width: 70, height: 70)
                    
                    Image(systemName: sensitivityIcon)
                        .font(.system(size: 30, weight: .bold))
                        .foregroundColor(action.sensitivityColor)
                }
            }
            
            VStack(spacing: 4) {
                Text(sensitivityTitle)
                    .font(.system(size: 22, weight: .bold))
                    .foregroundColor(action.sensitivityColor)
                
                Text("Cette action nécessite votre confirmation")
                    .font(.system(size: 14))
                    .foregroundColor(theme.textSecondary)
            }
        }
        .padding(.top, 10)
    }
    
    private var sensitivityIcon: String {
        switch action.sensitivityLevel {
        case "critical": return "exclamationmark.triangle.fill"
        case "high": return "exclamationmark.circle.fill"
        case "moderate": return "info.circle.fill"
        default: return "checkmark.circle.fill"
        }
    }
    
    private var sensitivityTitle: String {
        switch action.sensitivityLevel {
        case "critical": return "Action Critique"
        case "high": return "Action Sensible"
        case "moderate": return "Action Modérée"
        default: return "Action"
        }
    }
    
    // MARK: - Action Details Card
    
    private var actionDetailsCard: some View {
        VStack(alignment: .leading, spacing: 12) {
            HStack {
                Image(systemName: "doc.text")
                    .foregroundColor(theme.accentColor)
                Text("Détails de l'action")
                    .font(.system(size: 16, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
            }
            
            VStack(spacing: 10) {
                detailRow(label: "Modèle", value: action.model, icon: "cube")
                detailRow(label: "Action", value: action.method, icon: "play.circle")
                detailRow(label: "Statut", value: action.status, icon: "flag")
                detailRow(label: "Créé le", value: formatDate(action.createdAt), icon: "calendar")
            }
        }
        .padding(16)
        .background(
            RoundedRectangle(cornerRadius: 16)
                .fill(theme.surfaceElevated)
        )
    }
    
    private func detailRow(label: String, value: String, icon: String) -> some View {
        HStack {
            Image(systemName: icon)
                .font(.system(size: 14))
                .foregroundColor(theme.textMuted)
                .frame(width: 24)
            
            Text(label)
                .font(.system(size: 14))
                .foregroundColor(theme.textSecondary)
            
            Spacer()
            
            Text(value)
                .font(.system(size: 14, weight: .medium))
                .foregroundColor(theme.textPrimary)
        }
    }
    
    // MARK: - Classification Card
    
    private var classificationCard: some View {
        VStack(alignment: .leading, spacing: 12) {
            HStack {
                Image(systemName: "shield")
                    .foregroundColor(action.sensitivityColor)
                Text("Classification")
                    .font(.system(size: 16, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
            }
            
            HStack(spacing: 16) {
                classificationBadge(
                    label: "Niveau",
                    value: action.sensitivityLevel.capitalized,
                    color: action.sensitivityColor
                )
                
                if action.requiresConfirmation {
                    classificationBadge(
                        label: "Confirmation",
                        value: "Requise",
                        color: .orange
                    )
                }
            }
        }
        .padding(16)
        .background(
            RoundedRectangle(cornerRadius: 16)
                .fill(theme.surfaceElevated)
        )
    }
    
    private func classificationBadge(label: String, value: String, color: Color) -> some View {
        VStack(spacing: 4) {
            Text(value)
                .font(.system(size: 14, weight: .bold))
                .foregroundColor(color)
            
            Text(label)
                .font(.system(size: 10))
                .foregroundColor(theme.textMuted)
        }
        .padding(.horizontal, 12)
        .padding(.vertical, 8)
        .background(
            RoundedRectangle(cornerRadius: 10)
                .fill(color.opacity(0.15))
        )
    }
    
    // MARK: - Warning Message
    
    private func warningMessageView(_ message: String) -> some View {
        HStack(alignment: .top, spacing: 12) {
            Image(systemName: "exclamationmark.triangle.fill")
                .foregroundColor(action.sensitivityColor)
            
            Text(message)
                .font(.system(size: 14))
                .foregroundColor(theme.textPrimary)
                .fixedSize(horizontal: false, vertical: true)
        }
        .padding(16)
        .background(
            RoundedRectangle(cornerRadius: 12)
                .fill(action.sensitivityColor.opacity(0.1))
                .overlay(
                    RoundedRectangle(cornerRadius: 12)
                        .stroke(action.sensitivityColor.opacity(0.3), lineWidth: 1)
                )
        )
    }
    
    // MARK: - Action Buttons
    
    private var actionButtons: some View {
        VStack(spacing: 12) {
            // Confirm Button
            Button {
                isConfirming = true
                Task {
                    onConfirm()
                    dismiss()
                }
            } label: {
                HStack(spacing: 10) {
                    if isConfirming {
                        ProgressView()
                            .tint(.white)
                    } else {
                        Image(systemName: "checkmark.shield")
                    }
                    Text(isConfirming ? "Confirmation..." : "Confirmer et exécuter")
                        .font(.system(size: 16, weight: .bold))
                }
                .foregroundColor(.white)
                .frame(maxWidth: .infinity)
                .padding(.vertical, 16)
                .background(
                    LinearGradient(
                        colors: [action.sensitivityColor, action.sensitivityColor.opacity(0.8)],
                        startPoint: .leading,
                        endPoint: .trailing
                    )
                )
                .cornerRadius(14)
            }
            .disabled(isConfirming)
            
            // Cancel Button
            Button {
                dismiss()
                onCancel()
            } label: {
                Text("Annuler")
                    .font(.system(size: 16, weight: .medium))
                    .foregroundColor(theme.textSecondary)
            }
            .disabled(isConfirming)
        }
        .padding(.top, 8)
    }
    
    private func formatDate(_ date: Date) -> String {
        let formatter = DateFormatter()
        formatter.dateStyle = .short
        formatter.timeStyle = .short
        return formatter.string(from: date)
    }
}

// MARK: - Pending Actions Panel

struct PendingActionsPanel: View {
    @EnvironmentObject var theme: ThemeManager
    @StateObject private var taskManager = BackgroundTaskManager.shared
    @State private var selectedAction: QueuedOdooAction?
    
    var body: some View {
        VStack(alignment: .leading, spacing: 16) {
            // Header
            HStack {
                Image(systemName: "clock.badge.exclamationmark")
                    .foregroundColor(theme.goldAccent)
                
                Text("Actions en attente")
                    .font(.system(size: 18, weight: .bold))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                if !taskManager.pendingActions.isEmpty {
                    Text("\(taskManager.pendingActions.count)")
                        .font(.system(size: 12, weight: .bold))
                        .foregroundColor(.white)
                        .padding(.horizontal, 8)
                        .padding(.vertical, 4)
                        .background(Capsule().fill(theme.accentColor))
                }
            }
            
            if taskManager.pendingActions.isEmpty {
                emptyState
            } else {
                ForEach(taskManager.pendingActions) { action in
                    PendingActionCard(action: action) {
                        selectedAction = action
                    } onCancel: {
                        taskManager.cancelAction(action.id)
                    }
                }
            }
        }
        .padding(16)
        .background(
            RoundedRectangle(cornerRadius: 20)
                .fill(theme.surfaceColor)
        )
        .sheet(item: $selectedAction) { action in
            ActionConfirmationSheet(action: action) {
                Task {
                    await taskManager.confirmAction(action.id)
                }
            } onCancel: {
                taskManager.cancelAction(action.id)
            }
        }
    }
    
    private var emptyState: some View {
        HStack {
            Spacer()
            VStack(spacing: 8) {
                Image(systemName: "checkmark.circle")
                    .font(.system(size: 30))
                    .foregroundColor(theme.successColor)
                
                Text("Aucune action en attente")
                    .font(.system(size: 14))
                    .foregroundColor(theme.textMuted)
            }
            .padding(.vertical, 20)
            Spacer()
        }
    }
}

// MARK: - Pending Action Card

struct PendingActionCard: View {
    @EnvironmentObject var theme: ThemeManager
    let action: QueuedOdooAction
    let onConfirm: () -> Void
    let onCancel: () -> Void
    
    var body: some View {
        VStack(alignment: .leading, spacing: 10) {
            // Header
            HStack {
                Circle()
                    .fill(action.sensitivityColor)
                    .frame(width: 10, height: 10)
                
                Text(action.model)
                    .font(.system(size: 14, weight: .semibold))
                    .foregroundColor(theme.textPrimary)
                
                Text("•")
                    .foregroundColor(theme.textMuted)
                
                Text(action.method)
                    .font(.system(size: 14))
                    .foregroundColor(theme.textSecondary)
                
                Spacer()
                
                Text(timeAgo(action.createdAt))
                    .font(.system(size: 11))
                    .foregroundColor(theme.textMuted)
            }
            
            // Status
            HStack {
                Label(
                    action.requiresConfirmation ? "Confirmation requise" : "Prêt",
                    systemImage: action.requiresConfirmation ? "exclamationmark.circle" : "checkmark.circle"
                )
                .font(.system(size: 12))
                .foregroundColor(action.requiresConfirmation ? .orange : .green)
                
                Spacer()
                
                // Actions
                HStack(spacing: 8) {
                    Button {
                        onCancel()
                    } label: {
                        Text("Annuler")
                            .font(.system(size: 12, weight: .medium))
                            .foregroundColor(theme.textMuted)
                            .padding(.horizontal, 12)
                            .padding(.vertical, 6)
                            .background(
                                Capsule().fill(theme.surfaceElevated)
                            )
                    }
                    
                    if action.requiresConfirmation {
                        Button {
                            onConfirm()
                        } label: {
                            Text("Confirmer")
                                .font(.system(size: 12, weight: .bold))
                                .foregroundColor(.white)
                                .padding(.horizontal, 12)
                                .padding(.vertical, 6)
                                .background(
                                    Capsule().fill(action.sensitivityColor)
                                )
                        }
                    }
                }
            }
        }
        .padding(12)
        .background(
            RoundedRectangle(cornerRadius: 12)
                .fill(theme.surfaceElevated)
                .overlay(
                    RoundedRectangle(cornerRadius: 12)
                        .stroke(action.sensitivityColor.opacity(0.3), lineWidth: 1)
                )
        )
    }
    
    private func timeAgo(_ date: Date) -> String {
        let formatter = RelativeDateTimeFormatter()
        formatter.unitsStyle = .abbreviated
        return formatter.localizedString(for: date, relativeTo: Date())
    }
}

// MARK: - Action History View

struct ActionHistoryView: View {
    @EnvironmentObject var theme: ThemeManager
    @StateObject private var taskManager = BackgroundTaskManager.shared
    
    var body: some View {
        List {
            ForEach(taskManager.actionHistory) { action in
                ActionHistoryRow(action: action)
            }
        }
        .listStyle(.insetGrouped)
        .navigationTitle("Historique des actions")
    }
}

struct ActionHistoryRow: View {
    @EnvironmentObject var theme: ThemeManager
    let action: QueuedOdooAction
    
    var body: some View {
        HStack(spacing: 12) {
            // Status Icon
            ZStack {
                Circle()
                    .fill(statusColor.opacity(0.2))
                    .frame(width: 40, height: 40)
                
                Image(systemName: statusIcon)
                    .foregroundColor(statusColor)
            }
            
            VStack(alignment: .leading, spacing: 4) {
                HStack {
                    Text(action.model)
                        .font(.system(size: 14, weight: .semibold))
                    
                    Text("→")
                        .foregroundColor(theme.textMuted)
                    
                    Text(action.method)
                        .font(.system(size: 14))
                        .foregroundColor(theme.textSecondary)
                }
                
                if let date = action.executedAt {
                    Text(formatDate(date))
                        .font(.system(size: 11))
                        .foregroundColor(theme.textMuted)
                }
            }
            
            Spacer()
            
            // Status Badge
            Text(statusText)
                .font(.system(size: 10, weight: .bold))
                .foregroundColor(.white)
                .padding(.horizontal, 8)
                .padding(.vertical, 4)
                .background(Capsule().fill(statusColor))
        }
        .padding(.vertical, 4)
    }
    
    private var statusIcon: String {
        switch action.status {
        case "completed": return "checkmark"
        case "failed": return "xmark"
        case "cancelled": return "minus"
        default: return "clock"
        }
    }
    
    private var statusColor: Color {
        switch action.status {
        case "completed": return .green
        case "failed": return .red
        case "cancelled": return .gray
        default: return .orange
        }
    }
    
    private var statusText: String {
        switch action.status {
        case "completed": return "OK"
        case "failed": return "Échec"
        case "cancelled": return "Annulé"
        default: return action.status
        }
    }
    
    private func formatDate(_ date: Date) -> String {
        let formatter = DateFormatter()
        formatter.dateStyle = .short
        formatter.timeStyle = .short
        return formatter.string(from: date)
    }
}
