//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Enhanced Chat View                                          ║
//  ║         Main Conversation Interface with Universal Content Panel              ║
//  ║         Supports Rich Responses: Tables, Graphs, Dashboards, etc.             ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import UniformTypeIdentifiers

struct ChatView: View {
    @EnvironmentObject var chatManager: ChatManager
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var appState: AppState
    @StateObject private var voiceManager = VoiceManager.shared
    @ObservedObject private var wsManager = WebSocketManager.shared
    
    @State private var inputText = ""
    @State private var showVoiceOverlay = false
    @State private var showProviderPicker = false
    @State private var showAttachmentPicker = false
    @State private var showClearConfirmation = false
    @State private var showToast = false
    @State private var toastMessage = ""
    @State private var toastType: BloodToast.ToastType = .info
    @State private var panelHeight: CGFloat = 300
    @State private var useWebSocket = true
    @FocusState private var isInputFocused: Bool
    
    // Adaptive display settings
    @State private var isUserFocused = true
    @State private var lastInteraction = Date()
    
    // Unified Collaboration panel
    @StateObject private var unifiedPanel = UnifiedPanelManager.shared
    
    var body: some View {
        NavigationStack {
            ZStack {
                // Background with tap to dismiss keyboard
                theme.backgroundGradient
                    .ignoresSafeArea()
                    .onTapGesture {
                        dismissKeyboard()
                    }
                
                VStack(spacing: 0) {
                    // Messages list
                    ScrollViewReader { proxy in
                        ScrollView {
                            LazyVStack(spacing: 16) {
                                // Welcome message if empty
                                if chatManager.messages.isEmpty {
                                    WelcomeView(
                                        onVoiceAction: { showVoiceOverlay = true },
                                        onInfraAction: {
                                            Task {
                                                await chatManager.sendMessage("Montre-moi l'état complet de l'infrastructure Docker avec tous les services et leur statut")
                                            }
                                        },
                                        onAssistAction: {
                                            Task {
                                                await chatManager.sendMessage("Bonjour ! Que puis-je faire pour toi aujourd'hui ? Voici ce que je peux t'aider à faire: gérer l'infrastructure Docker, créer des tickets iTop, consulter Odoo CRM, et bien plus.")
                                            }
                                        }
                                    )
                                    .transition(.opacity.combined(with: .scale))
                                }
                                
                                ForEach(Array(chatManager.messages.enumerated()), id: \.element.id) { index, message in
                                    VStack(spacing: 8) {
                                        // Message bubble
                                        MessageBubble(message: message)
                                            .messageAppear(delay: Double(index) * 0.05)
                                            .id(message.id)
                                            .contextMenu {
                                                Button {
                                                    UIPasteboard.general.string = message.content
                                                    showNotification("Copié!", type: .success)
                                                } label: {
                                                    Label("Copier", systemImage: "doc.on.doc")
                                                }
                                                
                                                if message.role == .assistant {
                                                    Button {
                                                        voiceManager.speak(message.content)
                                                    } label: {
                                                        Label("Lire à voix haute", systemImage: "speaker.wave.3")
                                                    }
                                                    
                                                    if let attachments = message.attachments, !attachments.isEmpty {
                                                        Button {
                                                            showPanelForMessage(message)
                                                        } label: {
                                                            Label("Voir les détails", systemImage: "rectangle.expand.vertical")
                                                        }
                                                    }
                                                }
                                            }
                                        
                                        // Suggested actions (inline)
                                        if message.role == .assistant,
                                           let actions = message.suggestedActions,
                                           !actions.isEmpty,
                                           index == chatManager.messages.count - 1 {
                                            SuggestedActionsView(actions: actions) { action in
                                                Task {
                                                    await chatManager.executeSuggestedAction(action)
                                                }
                                            }
                                        }
                                        
                                        // Quality/Complexity badge for last message
                                        if message.role == .assistant,
                                           index == chatManager.messages.count - 1 {
                                            MessageMetaBadge(message: message)
                                        }
                                    }
                                }
                                
                                // Streaming content
                                if !chatManager.streamingContent.isEmpty {
                                    StreamingBubble(content: chatManager.streamingContent)
                                }
                                
                                // Typing indicator (HTTP mode)
                                if chatManager.isTyping && chatManager.streamingContent.isEmpty && !wsManager.isThinking {
                                    HStack {
                                        TypingIndicator()
                                        Spacer()
                                    }
                                    .padding(.horizontal)
                                }
                                
                                // WebSocket Thinking indicator (inline mini)
                                if wsManager.isThinking {
                                    HStack(alignment: .top, spacing: 12) {
                                        ZStack {
                                            Circle()
                                                .fill(theme.primaryGradient)
                                                .frame(width: 36, height: 36)
                                            Image(systemName: "brain.head.profile")
                                                .font(.system(size: 16))
                                                .foregroundColor(.white)
                                        }
                                        
                                        VStack(alignment: .leading, spacing: 8) {
                                            HStack(spacing: 8) {
                                                Text("🧠 Réflexion...")
                                                    .font(.system(size: 14, weight: .medium))
                                                    .foregroundColor(theme.textPrimary)
                                                
                                                ProgressView()
                                                    .scaleEffect(0.7)
                                            }
                                            
                                            if let lastStep = wsManager.thinkingSteps.last {
                                                HStack(spacing: 6) {
                                                    Image(systemName: lastStep.type.icon)
                                                        .font(.system(size: 11))
                                                        .foregroundColor(lastStep.type.color)
                                                    
                                                    Text(lastStep.description)
                                                        .font(.system(size: 12))
                                                        .foregroundColor(theme.textSecondary)
                                                        .lineLimit(1)
                                                }
                                            }
                                            
                                            // Progress bar
                                            GeometryReader { geo in
                                                ZStack(alignment: .leading) {
                                                    Capsule()
                                                        .fill(theme.surfaceElevated)
                                                    Capsule()
                                                        .fill(LinearGradient(colors: [theme.primaryColor, theme.accentColor], startPoint: .leading, endPoint: .trailing))
                                                        .frame(width: geo.size.width * wsManager.thinkingProgress)
                                                }
                                            }
                                            .frame(height: 4)
                                            .cornerRadius(2)
                                        }
                                        .padding(12)
                                        .background(theme.surfaceColor)
                                        .cornerRadius(16)
                                        
                                        Spacer(minLength: 60)
                                    }
                                    .padding(.horizontal)
                                }
                            }
                            .padding()
                            .padding(.bottom, chatManager.showContentPanel ? panelHeight + 20 : 0)
                        }
                        .scrollDismissesKeyboard(.interactively)
                        .onTapGesture {
                            dismissKeyboard()
                            trackUserFocus()
                        }
                        .gesture(
                            DragGesture()
                                .onChanged { _ in
                                    dismissKeyboard()
                                    trackUserFocus()
                                }
                        )
                        .onChange(of: chatManager.messages.count) { _, _ in
                            withAnimation(.spring(response: 0.3)) {
                                if let lastId = chatManager.messages.last?.id {
                                    proxy.scrollTo(lastId, anchor: .bottom)
                                }
                            }
                        }
                        .onChange(of: chatManager.streamingContent) { _, _ in
                            withAnimation {
                                if let lastId = chatManager.messages.last?.id {
                                    proxy.scrollTo(lastId, anchor: .bottom)
                                }
                            }
                        }
                    }
                    
                    // Universal Content Panel (auto-appearing)
                    if chatManager.showContentPanel && !chatManager.currentPanelAttachments.isEmpty {
                        ChatPanelContainer(
                            attachments: chatManager.currentPanelAttachments,
                            isVisible: $chatManager.showContentPanel
                        )
                        .frame(height: panelHeight)
                        .transition(.move(edge: .bottom).combined(with: .opacity))
                        .gesture(
                            DragGesture()
                                .onChanged { value in
                                    let newHeight = panelHeight - value.translation.height
                                    panelHeight = min(max(150, newHeight), 500)
                                }
                        )
                    }
                    
                    // Input area
                    EnhancedChatInputBar(
                        text: $inputText,
                        isInputFocused: $isInputFocused,
                        onSend: sendMessage,
                        onVoice: { showVoiceOverlay = true },
                        onAttachment: { showAttachmentPicker = true }
                    )
                }
                
                // Voice Overlay (Full Screen)
                if showVoiceOverlay {
                    VoiceOverlay(
                        voiceManager: voiceManager,
                        isPresented: $showVoiceOverlay
                    ) { transcribedText in
                        Task {
                            await chatManager.sendMessage(transcribedText, isVoice: true)
                        }
                    }
                    .transition(.opacity.combined(with: .scale(scale: 0.95)))
                    .zIndex(50)
                }
                
                // Speaking indicator
                if voiceManager.isSpeaking {
                    SpeakingIndicator {
                        voiceManager.stopSpeaking()
                    }
                    .transition(.move(edge: .bottom).combined(with: .opacity))
                    .zIndex(40)
                }
                
                // Floating Thinking Overlay (WebSocket real-time)
                FloatingThinkingOverlay(wsManager: wsManager)
                    .zIndex(45)
                
                // Unified Collaboration Panel (bottom sheet style)
                if unifiedPanel.isVisible {
                    VStack {
                        Spacer()
                        
                        UnifiedCollaborationPanel(wsManager: wsManager)
                            .padding(.horizontal, 8)
                            .padding(.bottom, 100)
                    }
                    .transition(.move(edge: .bottom).combined(with: .opacity))
                    .zIndex(46)
                }
                
                // Connection status badge + Unified Panel toggle
                VStack {
                    HStack {
                        ConnectionStatusBadge(wsManager: wsManager)
                            .padding(.leading, 16)
                            .padding(.top, 8)
                        
                        Spacer()
                        
                        // Unified panel toggle
                        Button(action: {
                            unifiedPanel.toggleVisibility()
                        }) {
                            HStack(spacing: 6) {
                                Image(systemName: "brain.head.profile")
                                    .font(.system(size: 14))
                                
                                if unifiedPanel.isProcessing {
                                    ProgressView()
                                        .scaleEffect(0.5)
                                        .frame(width: 14, height: 14)
                                }
                                
                                if let session = unifiedPanel.currentSession, !session.steps.isEmpty {
                                    Text("\(session.steps.count)")
                                        .font(.system(size: 11, weight: .bold))
                                }
                            }
                            .foregroundColor(unifiedPanel.isVisible || unifiedPanel.isProcessing ? theme.accentColor : theme.textMuted)
                            .padding(.horizontal, 12)
                            .padding(.vertical, 8)
                            .background(
                                Capsule()
                                    .fill(unifiedPanel.isVisible ? theme.accentColor.opacity(0.2) : theme.surfaceColor)
                            )
                            .overlay(
                                Capsule()
                                    .stroke(unifiedPanel.isProcessing ? theme.accentColor.opacity(0.5) : Color.clear, lineWidth: 1)
                            )
                        }
                        .padding(.trailing, 16)
                        .padding(.top, 8)
                    }
                    Spacer()
                }
                .zIndex(35)
            }
            .navigationTitle("")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .principal) {
                    // AYMENOS Logo + Title
                    HStack(spacing: 10) {
                        // Mini logo
                        Image("AymenosLogo")
                            .resizable()
                            .aspectRatio(contentMode: .fit)
                            .frame(width: 30, height: 30)
                            .clipShape(Circle())
                            .overlay(
                                Circle()
                                    .stroke(theme.goldAccent.opacity(0.5), lineWidth: 1)
                            )
                            .shadow(color: theme.goldAccent.opacity(0.3), radius: 3)
                        
                        VStack(alignment: .leading, spacing: 1) {
                            Text("AYMENOS")
                                .font(.system(size: 15, weight: .bold, design: .serif))
                                .foregroundStyle(
                                    LinearGradient(
                                        colors: [theme.goldAccent, theme.goldLight],
                                        startPoint: .leading,
                                        endPoint: .trailing
                                    )
                                )
                            
                            Text("Assistant Personnel")
                                .font(.system(size: 9, weight: .medium))
                                .foregroundColor(theme.textMuted)
                        }
                        
                        // Version badge
                        Text("v2.6")
                            .font(.system(size: 8, weight: .bold))
                            .foregroundColor(theme.goldAccent)
                            .padding(.horizontal, 5)
                            .padding(.vertical, 2)
                            .background(
                                Capsule()
                                    .stroke(theme.goldAccent.opacity(0.5), lineWidth: 1)
                            )
                    }
                }
                
                ToolbarItem(placement: .topBarLeading) {
                    Button(action: {
                        if !chatManager.messages.isEmpty {
                            showClearConfirmation = true
                        }
                    }) {
                        Image(systemName: "plus.message.fill")
                            .foregroundColor(theme.goldAccent)
                    }
                }
                
                ToolbarItem(placement: .topBarTrailing) {
                    HStack(spacing: 10) {
                        // Lanciio VIP Connect Button
                        LanciioConnectButton()
                        
                        // Panel toggle
                        if !chatManager.currentPanelAttachments.isEmpty {
                            Button(action: {
                                withAnimation(.spring()) {
                                    chatManager.showContentPanel.toggle()
                                }
                            }) {
                                Image(systemName: chatManager.showContentPanel ? "rectangle.compress.vertical" : "rectangle.expand.vertical")
                                    .foregroundColor(theme.accentColor)
                            }
                        }
                        
                        // Provider menu
                        Menu {
                            ForEach(AIProvider.allCases, id: \.self) { provider in
                                Button(action: {
                                    withAnimation(.spring()) {
                                        chatManager.selectedProvider = provider
                                    }
                                    showNotification("Provider: \(provider.displayName)", type: .info)
                                }) {
                                    Label(provider.displayName, systemImage: provider.icon)
                                    if chatManager.selectedProvider == provider {
                                        Image(systemName: "checkmark")
                                    }
                                }
                            }
                        } label: {
                            HStack(spacing: 4) {
                                Image(systemName: chatManager.selectedProvider.icon)
                                Image(systemName: "chevron.down")
                                    .font(.system(size: 10))
                            }
                            .foregroundColor(theme.accentColor)
                        }
                    }
                }
            }
            .animation(.spring(response: 0.4), value: showVoiceOverlay)
            .animation(.spring(response: 0.4), value: chatManager.showContentPanel)
        }
        .sheet(isPresented: $showAttachmentPicker) {
            AttachmentPickerView()
        }
        .bloodAlert(
            isPresented: $showClearConfirmation,
            title: "Nouvelle conversation",
            message: "Voulez-vous effacer la conversation actuelle et en démarrer une nouvelle ?",
            icon: "trash.fill",
            iconColor: theme.warningColor,
            primaryAction: ModalAction(title: "Effacer", isDestructive: true) {
                withAnimation(.spring()) {
                    chatManager.clearConversation()
                }
            },
            secondaryAction: ModalAction(title: "Annuler") {}
        )
        .bloodToast(isPresented: $showToast, message: toastMessage, type: toastType)
        .onAppear {
            setupVoiceCallbacks()
            setupWebSocket()
            
            // Show version notification on first appear
            DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
                showNotification("✨ AYMENOS v2.6 • Panel de Collaboration", type: .success)
            }
        }
        .onChange(of: appState.triggerVoiceInput) { _, newValue in
            if newValue {
                showVoiceOverlay = true
            }
        }
        .onChange(of: wsManager.streamedContent) { _, newContent in
            // Handle streamed content from WebSocket
            if !newContent.isEmpty && wsManager.isStreaming {
                chatManager.streamingContent = newContent
            }
        }
        .onChange(of: wsManager.isStreaming) { _, isStreaming in
            if !isStreaming && !wsManager.streamedContent.isEmpty {
                // Stream ended, add message
                let assistantMessage = ChatMessage(
                    content: wsManager.streamedContent,
                    role: .assistant,
                    contentType: .markdown
                )
                chatManager.messages.append(assistantMessage)
                chatManager.streamingContent = ""
                wsManager.streamedContent = ""
            }
        }
        .onChange(of: wsManager.showPanel) { _, showPanel in
            if showPanel {
                chatManager.currentPanelAttachments = wsManager.panelAttachments
                chatManager.showContentPanel = true
            }
        }
    }
    
    private func setupWebSocket() {
        // Connect to WebSocket on appear
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
            if wsManager.connectionState == .disconnected {
                wsManager.connect()
            }
        }
    }
    
    // Simulate thinking for demo purposes (remove in production)
    private func simulateThinking() {
        wsManager.isThinking = true
        wsManager.thinkingProgress = 0
        
        // Simulate steps
        let steps: [(ThinkingStepType, String, String)] = [
            (.analyzing, "🔍 Analyse", "Analyse de votre demande..."),
            (.routing, "🔀 Routage", "Délégation vers l'agent approprié"),
            (.executing, "⚡ Exécution", "Traitement en cours..."),
            (.reformulating, "✨ Reformulation", "Amélioration de la réponse")
        ]
        
        for (index, step) in steps.enumerated() {
            DispatchQueue.main.asyncAfter(deadline: .now() + Double(index) * 0.8) {
                let thinkingStep = ThinkingStep(
                    id: UUID().uuidString,
                    type: step.0,
                    title: step.1,
                    description: step.2,
                    agent: nil,
                    progress: Double(index + 1) / Double(steps.count),
                    timestamp: Date(),
                    status: .inProgress
                )
                wsManager.thinkingSteps.append(thinkingStep)
                wsManager.thinkingProgress = thinkingStep.progress
            }
        }
        
        // End thinking
        DispatchQueue.main.asyncAfter(deadline: .now() + Double(steps.count) * 0.8 + 0.5) {
            for i in wsManager.thinkingSteps.indices {
                wsManager.thinkingSteps[i].status = .completed
            }
            wsManager.thinkingProgress = 1.0
            
            DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
                wsManager.isThinking = false
                wsManager.thinkingSteps.removeAll()
            }
        }
    }
    
    // MARK: - Keyboard Management
    
    private func dismissKeyboard() {
        isInputFocused = false
        UIApplication.shared.sendAction(#selector(UIResponder.resignFirstResponder), to: nil, from: nil, for: nil)
    }
    
    private func trackUserFocus() {
        lastInteraction = Date()
        isUserFocused = true
        
        // Auto-detect if user is not focused after 3 seconds of no interaction
        DispatchQueue.main.asyncAfter(deadline: .now() + 3) { [self] in
            if Date().timeIntervalSince(lastInteraction) >= 3 {
                isUserFocused = false
            }
        }
    }
    
    private func sendMessage() {
        guard !inputText.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty else { return }
        
        let message = inputText
        inputText = ""
        dismissKeyboard() // Auto-dismiss keyboard after sending
        
        // Use WebSocket for real-time communication if connected
        if useWebSocket && wsManager.connectionState == .connected {
            wsManager.sendChat(content: message)
            
            // Add user message immediately
            let userMessage = ChatMessage(
                content: message,
                role: .user,
                contentType: .text,
                isVoice: false
            )
            chatManager.messages.append(userMessage)
        } else {
            // Fallback to HTTP
            Task {
                await chatManager.sendMessage(message, isVoice: false)
            }
        }
    }
    
    private func showNotification(_ message: String, type: BloodToast.ToastType) {
        toastMessage = message
        toastType = type
        showToast = true
    }
    
    private func setupVoiceCallbacks() {
        voiceManager.onAutoStop = { text in
            Task {
                await chatManager.sendMessage(text, isVoice: true)
            }
            showVoiceOverlay = false
        }
    }
    
    private func showPanelForMessage(_ message: ChatMessage) {
        guard let attachments = message.attachments else { return }
        
        chatManager.currentPanelAttachments = attachments.map { att in
            PanelAttachment(
                id: att.id,
                type: PanelContentType(rawValue: att.type) ?? .text,
                title: att.title,
                data: att.data.dictionary ?? [:],
                mimeType: att.mimeType,
                interactive: att.interactive
            )
        }
        
        withAnimation(.spring()) {
            chatManager.showContentPanel = true
        }
    }
}

// MARK: - Suggested Actions View

struct SuggestedActionsView: View {
    @EnvironmentObject var theme: ThemeManager
    let actions: [SuggestedAction]
    let onAction: (SuggestedAction) -> Void
    
    var body: some View {
        ScrollView(.horizontal, showsIndicators: false) {
            HStack(spacing: 8) {
                ForEach(actions) { action in
                    Button(action: { onAction(action) }) {
                        HStack(spacing: 6) {
                            Image(systemName: "arrow.right.circle")
                                .font(.system(size: 12))
                            
                            Text(action.action)
                                .font(.system(size: 12, weight: .medium))
                                .lineLimit(1)
                        }
                        .foregroundColor(theme.accentColor)
                        .padding(.horizontal, 12)
                        .padding(.vertical, 8)
                        .background(theme.primaryColor.opacity(0.15))
                        .cornerRadius(16)
                        .overlay(
                            RoundedRectangle(cornerRadius: 16)
                                .stroke(theme.primaryColor.opacity(0.3), lineWidth: 1)
                        )
                    }
                }
            }
            .padding(.horizontal)
        }
    }
}

// MARK: - Message Meta Badge

struct MessageMetaBadge: View {
    @EnvironmentObject var theme: ThemeManager
    let message: ChatMessage
    
    var body: some View {
        HStack(spacing: 8) {
            // Complexity badge
            if let complexity = message.complexity {
                ComplexityBadge(complexity: complexity)
            }
            
            // Quality score
            if let quality = message.qualityScore, quality > 0 {
                QualityBadge(score: quality)
            }
            
            // Agents used
            if let agents = message.agentsUsed, !agents.isEmpty {
                AgentsBadge(agents: agents)
            }
        }
        .padding(.leading, 48)
    }
}

struct ComplexityBadge: View {
    @EnvironmentObject var theme: ThemeManager
    let complexity: String
    
    var color: Color {
        switch complexity.lowercased() {
        case "simple": return .green
        case "moderate": return .orange
        case "complex": return .red
        case "strategic": return .purple
        default: return .gray
        }
    }
    
    var body: some View {
        HStack(spacing: 4) {
            Circle()
                .fill(color)
                .frame(width: 6, height: 6)
            
            Text(complexity.capitalized)
                .font(.system(size: 10))
                .foregroundColor(theme.textMuted)
        }
        .padding(.horizontal, 8)
        .padding(.vertical, 4)
        .background(color.opacity(0.1))
        .cornerRadius(8)
    }
}

struct QualityBadge: View {
    @EnvironmentObject var theme: ThemeManager
    let score: Double
    
    var body: some View {
        HStack(spacing: 4) {
            Image(systemName: score >= 0.8 ? "star.fill" : score >= 0.5 ? "star.leadinghalf.filled" : "star")
                .font(.system(size: 10))
                .foregroundColor(.yellow)
            
            Text("\(Int(score * 100))%")
                .font(.system(size: 10))
                .foregroundColor(theme.textMuted)
        }
        .padding(.horizontal, 8)
        .padding(.vertical, 4)
        .background(Color.yellow.opacity(0.1))
        .cornerRadius(8)
    }
}

struct AgentsBadge: View {
    @EnvironmentObject var theme: ThemeManager
    let agents: [String]
    
    var body: some View {
        HStack(spacing: 4) {
            Image(systemName: "person.3.fill")
                .font(.system(size: 10))
                .foregroundColor(theme.accentColor)
            
            Text("\(agents.count) agents")
                .font(.system(size: 10))
                .foregroundColor(theme.textMuted)
        }
        .padding(.horizontal, 8)
        .padding(.vertical, 4)
        .background(theme.primaryColor.opacity(0.1))
        .cornerRadius(8)
    }
}

// MARK: - Welcome View

struct WelcomeView: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    @ObservedObject private var wsManager = WebSocketManager.shared
    @State private var showContent = false
    @State private var pulse = false
    
    // Action callbacks
    var onVoiceAction: () -> Void = {}
    var onInfraAction: () -> Void = {}
    var onAssistAction: () -> Void = {}
    
    var body: some View {
        VStack(spacing: 24) {
            // Version Badge
            HStack {
                Spacer()
                HStack(spacing: 6) {
                    Image(systemName: "sparkles")
                        .font(.system(size: 10))
                    Text("v2.6 • AYMENOS")
                        .font(.system(size: 10, weight: .semibold))
                }
                .foregroundColor(theme.goldAccent)
                .padding(.horizontal, 10)
                .padding(.vertical, 4)
                .background(
                    Capsule()
                        .fill(theme.goldAccent.opacity(0.15))
                        .overlay(
                            Capsule()
                                .stroke(theme.goldAccent.opacity(0.3), lineWidth: 1)
                        )
                )
                Spacer()
            }
            .padding(.bottom, -10)
            
            // AYMENOS Logo
            ZStack {
                // Outer glow ring
                Circle()
                    .stroke(
                        LinearGradient(
                            colors: [theme.goldAccent.opacity(0.4), theme.goldAccent.opacity(0.1)],
                            startPoint: .top,
                            endPoint: .bottom
                        ),
                        lineWidth: 3
                    )
                    .frame(width: 180, height: 180)
                    .scaleEffect(pulse ? 1.1 : 1.0)
                
                // Pulse effect
                Circle()
                    .fill(
                        RadialGradient(
                            colors: [theme.goldAccent.opacity(0.2), Color.clear],
                            center: .center,
                            startRadius: 40,
                            endRadius: 100
                        )
                    )
                    .frame(width: 170, height: 170)
                    .scaleEffect(pulse ? 1.15 : 0.95)
                
                // AYMENOS Logo Image
                Image("AymenosLogo")
                    .resizable()
                    .aspectRatio(contentMode: .fit)
                    .frame(width: 150, height: 150)
                    .shadow(color: theme.goldAccent.opacity(0.5), radius: 20)
            }
            .scaleEffect(showContent ? 1 : 0.5)
            .opacity(showContent ? 1 : 0)
            
            VStack(spacing: 12) {
                Text("AYMENOS")
                    .font(.system(size: 32, weight: .bold, design: .serif))
                    .foregroundStyle(
                        LinearGradient(
                            colors: [theme.goldAccent, Color(hex: "D4AF37")],
                            startPoint: .leading,
                            endPoint: .trailing
                        )
                    )
                    .shadow(color: theme.goldAccent.opacity(0.3), radius: 5)
                
                Text("L'assistant personnel de Aymen")
                    .font(.system(size: 16, weight: .medium))
                    .foregroundColor(theme.textSecondary)
                
                // Connection status
                HStack(spacing: 8) {
                    Circle()
                        .fill(wsManager.connectionState.color)
                        .frame(width: 8, height: 8)
                    
                    Text(wsManager.connectionState == .connected ? "✅ Connecté en temps réel" : "⏳ Mode standard")
                        .font(.system(size: 12))
                        .foregroundColor(theme.textSecondary)
                }
                .padding(.top, 4)
            }
            .offset(y: showContent ? 0 : 20)
            .opacity(showContent ? 1 : 0)
            
            // New features badges
            HStack(spacing: 10) {
                FeatureBadge(icon: "waveform", text: "Thinking Stream")
                FeatureBadge(icon: "rectangle.stack", text: "Multi-Panel")
                FeatureBadge(icon: "bolt.fill", text: "Real-time")
            }
            .padding(.top, 8)
            .offset(y: showContent ? 0 : 20)
            .opacity(showContent ? 1 : 0)
            
            // Quick actions with real functionality
            VStack(spacing: 12) {
                WelcomeActionButton(
                    icon: "mic.fill",
                    title: "Commencer une conversation vocale",
                    color: theme.accentColor,
                    action: onVoiceAction
                )
                
                WelcomeActionButton(
                    icon: "server.rack",
                    title: "Vérifier l'état de l'infrastructure",
                    color: theme.successColor,
                    action: onInfraAction
                )
                
                WelcomeActionButton(
                    icon: "wand.and.stars",
                    title: "Demander une assistance",
                    color: theme.warningColor,
                    action: onAssistAction
                )
            }
            .padding(.top, 20)
            .offset(y: showContent ? 0 : 30)
            .opacity(showContent ? 1 : 0)
        }
        .padding(.vertical, 40)
        .onAppear {
            withAnimation(.spring(response: 0.6, dampingFraction: 0.7).delay(0.2)) {
                showContent = true
            }
            withAnimation(.easeInOut(duration: 1.5).repeatForever()) {
                pulse.toggle()
            }
        }
    }
}

struct FeatureBadge: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let text: String
    
    var body: some View {
        HStack(spacing: 4) {
            Image(systemName: icon)
                .font(.system(size: 10))
            Text(text)
                .font(.system(size: 10, weight: .medium))
        }
        .foregroundColor(theme.textSecondary)
        .padding(.horizontal, 8)
        .padding(.vertical, 4)
        .background(theme.surfaceColor)
        .cornerRadius(8)
    }
}

struct WelcomeActionButton: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let title: String
    let color: Color
    var action: () -> Void = {}
    
    @State private var isPressed = false
    
    var body: some View {
        Button(action: {
            // Haptic feedback
            let impactFeedback = UIImpactFeedbackGenerator(style: .medium)
            impactFeedback.impactOccurred()
            action()
        }) {
            HStack(spacing: 12) {
                ZStack {
                    Circle()
                        .fill(color.opacity(0.15))
                        .frame(width: 40, height: 40)
                    
                    Image(systemName: icon)
                        .font(.system(size: 18, weight: .semibold))
                        .foregroundColor(color)
                }
                
                Text(title)
                    .font(.system(size: 14, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                    .lineLimit(1)
                
                Spacer()
                
                Image(systemName: "chevron.right")
                    .font(.system(size: 12, weight: .semibold))
                    .foregroundColor(color.opacity(0.7))
            }
            .padding(.horizontal, 16)
            .padding(.vertical, 14)
            .background(
                RoundedRectangle(cornerRadius: 14)
                    .fill(theme.surfaceColor)
                    .shadow(color: color.opacity(0.1), radius: 8, y: 4)
            )
            .overlay(
                RoundedRectangle(cornerRadius: 14)
                    .stroke(color.opacity(0.2), lineWidth: 1)
            )
        }
        .buttonStyle(ScaleButtonStyle())
    }
}

// MARK: - Scale Button Style
struct ScaleButtonStyle: ButtonStyle {
    func makeBody(configuration: Configuration) -> some View {
        configuration.label
            .scaleEffect(configuration.isPressed ? 0.97 : 1)
            .animation(.easeInOut(duration: 0.15), value: configuration.isPressed)
    }
}

// MARK: - Enhanced Chat Input Bar

struct EnhancedChatInputBar: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    @Binding var text: String
    var isInputFocused: FocusState<Bool>.Binding
    let onSend: () -> Void
    let onVoice: () -> Void
    let onAttachment: () -> Void
    
    @State private var isExpanded = false
    
    var body: some View {
        VStack(spacing: 0) {
            // Top divider with gradient
            Rectangle()
                .fill(
                    LinearGradient(
                        colors: [Color.clear, theme.primaryColor.opacity(0.3), Color.clear],
                        startPoint: .leading,
                        endPoint: .trailing
                    )
                )
                .frame(height: 1)
            
            HStack(spacing: 12) {
                // Attachment button
                Button(action: onAttachment) {
                    Image(systemName: "plus.circle.fill")
                        .font(.system(size: 28))
                        .foregroundStyle(
                            LinearGradient(
                                colors: [theme.textMuted, theme.textSecondary],
                                startPoint: .top,
                                endPoint: .bottom
                            )
                        )
                }
                
                // Text input with glow when focused
                HStack(spacing: 8) {
                    TextField("Message...", text: $text, axis: .vertical)
                        .focused(isInputFocused)
                        .lineLimit(1...5)
                        .font(.system(size: 16))
                        .foregroundColor(theme.textPrimary)
                    
                    // Voice button
                    Button(action: onVoice) {
                        ZStack {
                            Circle()
                                .fill(theme.primaryColor.opacity(0.2))
                                .frame(width: 36, height: 36)
                            
                            Image(systemName: "mic.fill")
                                .font(.system(size: 16))
                                .foregroundColor(theme.accentColor)
                        }
                    }
                }
                .padding(.horizontal, 16)
                .padding(.vertical, 10)
                .background(theme.surfaceColor)
                .cornerRadius(24)
                .overlay(
                    RoundedRectangle(cornerRadius: 24)
                        .stroke(
                            isInputFocused.wrappedValue
                            ? theme.accentColor.opacity(0.5)
                            : theme.surfaceElevated,
                            lineWidth: 1
                        )
                )
                .shadow(
                    color: isInputFocused.wrappedValue ? theme.accentColor.opacity(0.2) : .clear,
                    radius: 10
                )
                .animation(.easeOut(duration: 0.2), value: isInputFocused.wrappedValue)
                
                // Send button
                Button(action: onSend) {
                    ZStack {
                        Circle()
                            .fill(
                                canSend
                                ? LinearGradient(
                                    colors: [theme.primaryLight, theme.primaryColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                                : LinearGradient(
                                    colors: [theme.surfaceElevated, theme.surfaceColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                            .frame(width: 44, height: 44)
                        
                        Image(systemName: "arrow.up")
                            .font(.system(size: 18, weight: .bold))
                            .foregroundColor(canSend ? .white : theme.textMuted)
                    }
                    .shadow(color: canSend ? theme.accentColor.opacity(0.4) : .clear, radius: 10)
                    .scaleEffect(canSend ? 1.0 : 0.95)
                    .animation(.spring(response: 0.3), value: canSend)
                }
                .disabled(!canSend || chatManager.isLoading)
            }
            .padding(.horizontal, 16)
            .padding(.vertical, 12)
            .background(theme.backgroundColor)
        }
    }
    
    private var canSend: Bool {
        !text.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty && !chatManager.isLoading
    }
}

// MARK: - Speaking Indicator

struct SpeakingIndicator: View {
    @EnvironmentObject var theme: ThemeManager
    let onStop: () -> Void
    
    @State private var wavePhase: CGFloat = 0
    
    var body: some View {
        VStack {
            Spacer()
            
            HStack(spacing: 16) {
                // Waveform animation
                HStack(spacing: 3) {
                    ForEach(0..<7, id: \.self) { i in
                        Capsule()
                            .fill(theme.accentColor)
                            .frame(width: 3, height: 8 + sin(wavePhase + Double(i) * 0.5) * 8)
                    }
                }
                
                Text("Lecture en cours...")
                    .font(.system(size: 14, weight: .medium))
                    .foregroundColor(theme.textPrimary)
                
                Spacer()
                
                // Stop button
                Button(action: onStop) {
                    Image(systemName: "stop.fill")
                        .font(.system(size: 14))
                        .foregroundColor(.white)
                        .padding(10)
                        .background(theme.primaryColor)
                        .clipShape(Circle())
                }
            }
            .padding(.horizontal, 20)
            .padding(.vertical, 14)
            .background(
                RoundedRectangle(cornerRadius: 20)
                    .fill(theme.surfaceColor)
                    .shadow(color: theme.primaryDark.opacity(0.3), radius: 15)
            )
            .padding(.horizontal, 16)
            .padding(.bottom, 100)
        }
        .onAppear {
            withAnimation(.linear(duration: 0.5).repeatForever(autoreverses: false)) {
                wavePhase = .pi * 2
            }
        }
    }
}

// MARK: - Chat Input Bar

struct ChatInputBar: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    @Binding var text: String
    @Binding var isRecording: Bool
    @Binding var showProviderPicker: Bool
    @Binding var showAttachmentPicker: Bool
    var isInputFocused: FocusState<Bool>.Binding
    let onSend: () -> Void
    
    var body: some View {
        VStack(spacing: 0) {
            Divider()
                .background(theme.surfaceElevated)
            
            HStack(spacing: 12) {
                // Attachment button
                Button(action: {
                    showAttachmentPicker = true
                }) {
                    Image(systemName: "paperclip")
                        .font(.system(size: 20))
                        .foregroundColor(theme.textSecondary)
                }
                
                // Text input
                HStack(spacing: 8) {
                    TextField("Message...", text: $text, axis: .vertical)
                        .focused(isInputFocused)
                        .lineLimit(1...5)
                        .font(theme.fontBody)
                        .foregroundColor(theme.textPrimary)
                    
                    // Voice button
                    Button(action: {
                        isRecording.toggle()
                    }) {
                        Image(systemName: isRecording ? "waveform.circle.fill" : "mic.fill")
                            .font(.system(size: 20))
                            .foregroundColor(isRecording ? theme.accentGlow : theme.textSecondary)
                            .animation(.easeInOut, value: isRecording)
                    }
                }
                .padding(.horizontal, 16)
                .padding(.vertical, 10)
                .background(theme.surfaceColor)
                .cornerRadius(theme.radiusLarge)
                
                // Send button
                Button(action: onSend) {
                    ZStack {
                        Circle()
                            .fill(
                                LinearGradient(
                                    colors: canSend ? [theme.primaryLight, theme.primaryColor] : [theme.surfaceElevated, theme.surfaceColor],
                                    startPoint: .topLeading,
                                    endPoint: .bottomTrailing
                                )
                            )
                            .frame(width: 40, height: 40)
                        
                        Image(systemName: "arrow.up")
                            .font(.system(size: 16, weight: .bold))
                            .foregroundColor(canSend ? .white : theme.textMuted)
                    }
                    .shadow(color: canSend ? theme.accentColor.opacity(0.4) : .clear, radius: 8)
                }
                .disabled(!canSend || chatManager.isLoading)
            }
            .padding(.horizontal, 16)
            .padding(.vertical, 12)
            .background(theme.backgroundColor)
        }
    }
    
    private var canSend: Bool {
        !text.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty && !chatManager.isLoading
    }
}

// MARK: - Message Bubble (Full-Width for Assistant)

struct MessageBubble: View {
    @EnvironmentObject var theme: ThemeManager
    let message: ChatMessage
    
    var isUser: Bool {
        message.role == .user
    }
    
    var body: some View {
        if isUser {
            // User message - right aligned, compact
            userMessageView
        } else {
            // Assistant message - full width, sophisticated
            assistantMessageView
        }
    }
    
    // MARK: - User Message (Compact, Right-aligned)
    
    private var userMessageView: some View {
        HStack(alignment: .top, spacing: 12) {
            Spacer(minLength: 60)
            
            VStack(alignment: .trailing, spacing: 6) {
                Text(message.content)
                    .font(.system(size: 15))
                    .foregroundColor(.white)
                    .padding(.horizontal, 16)
                    .padding(.vertical, 12)
                    .background(theme.primaryGradient)
                    .cornerRadius(20, corners: [.topLeft, .topRight, .bottomLeft])
                    .shadow(color: theme.primaryDark.opacity(0.3), radius: 5, x: 0, y: 2)
                
                // Timestamp
                Text(formatTime(message.createdAt))
                    .font(.system(size: 10))
                    .foregroundColor(theme.textMuted)
            }
        }
        .padding(.horizontal, 12)
    }
    
    // MARK: - Assistant Message (Full-Width, Rich)
    
    private var assistantMessageView: some View {
        VStack(alignment: .leading, spacing: 0) {
            // Header with avatar and info
            HStack(alignment: .center, spacing: 10) {
                // AYMENOS Avatar
                ZStack {
                    Circle()
                        .fill(
                            RadialGradient(
                                colors: [theme.goldAccent.opacity(0.3), theme.surfaceElevated],
                                center: .center,
                                startRadius: 0,
                                endRadius: 18
                            )
                        )
                        .frame(width: 36, height: 36)
                    
                    Image("AymenosLogo")
                        .resizable()
                        .aspectRatio(contentMode: .fit)
                        .frame(width: 28, height: 28)
                        .clipShape(Circle())
                }
                .shadow(color: theme.goldAccent.opacity(0.4), radius: 4)
                
                VStack(alignment: .leading, spacing: 2) {
                    HStack(spacing: 6) {
                        Text("AYMENOS")
                            .font(.system(size: 13, weight: .bold))
                            .foregroundStyle(
                                LinearGradient(
                                    colors: [theme.goldAccent, theme.goldLight],
                                    startPoint: .leading,
                                    endPoint: .trailing
                                )
                            )
                        
                        if let provider = message.provider {
                            HStack(spacing: 3) {
                                Image(systemName: provider.icon)
                                    .font(.system(size: 9))
                                Text(provider.rawValue)
                                    .font(.system(size: 9, weight: .medium))
                            }
                            .foregroundColor(theme.textMuted)
                            .padding(.horizontal, 6)
                            .padding(.vertical, 2)
                            .background(Capsule().fill(theme.surfaceElevated))
                        }
                    }
                    
                    HStack(spacing: 6) {
                        Text(formatTime(message.createdAt))
                            .font(.system(size: 10))
                        
                        if let time = message.processingTimeMs {
                            Text("•")
                            Text("\(time)ms")
                                .font(.system(size: 10))
                        }
                    }
                    .foregroundColor(theme.textMuted)
                }
                
                Spacer()
            }
            .padding(.horizontal, 16)
            .padding(.top, 14)
            .padding(.bottom, 10)
            
            // Divider
            Rectangle()
                .fill(
                    LinearGradient(
                        colors: [.clear, theme.goldAccent.opacity(0.3), .clear],
                        startPoint: .leading,
                        endPoint: .trailing
                    )
                )
                .frame(height: 1)
                .padding(.horizontal, 16)
            
            // Content - Full width
            VStack(alignment: .leading, spacing: 0) {
                MessageContent(message: message)
                    .frame(maxWidth: .infinity, alignment: .leading)
            }
            .padding(.horizontal, 16)
            .padding(.vertical, 14)
        }
        .background(
            RoundedRectangle(cornerRadius: 16)
                .fill(theme.surfaceColor.opacity(0.6))
                .overlay(
                    RoundedRectangle(cornerRadius: 16)
                        .stroke(
                            LinearGradient(
                                colors: [theme.goldAccent.opacity(0.2), theme.surfaceElevated.opacity(0.3)],
                                startPoint: .topLeading,
                                endPoint: .bottomTrailing
                            ),
                            lineWidth: 1
                        )
                )
        )
        .shadow(color: .black.opacity(0.1), radius: 8, x: 0, y: 4)
        .padding(.horizontal, 8)
    }
    
    private func formatTime(_ date: Date) -> String {
        let formatter = DateFormatter()
        formatter.timeStyle = .short
        return formatter.string(from: date)
    }
}

// MARK: - Message Content

struct MessageContent: View {
    @EnvironmentObject var theme: ThemeManager
    let message: ChatMessage
    
    var body: some View {
        switch message.contentType {
        case .markdown:
            AdaptiveMarkdownView(content: message.content, isAssistant: message.role == .assistant)
        case .json:
            JSONView(content: message.content)
        case .code:
            CodeView(content: message.content)
        default:
            Text(message.content)
                .font(theme.fontBody)
                .foregroundColor(message.role == .user ? .white : theme.textPrimary)
        }
    }
}

// MARK: - Adaptive Markdown View (Human-like reading speed)

struct AdaptiveMarkdownView: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    let isAssistant: Bool
    
    @State private var displayedContent = ""
    @State private var isAnimating = false
    @State private var animationComplete = false
    @AppStorage("adaptiveTextSpeed") private var adaptiveSpeedEnabled = true
    
    // Reading speed: characters per second
    private let humanReadingSpeed: Double = 40 // ~240 words/minute
    private let fastSpeed: Double = 200
    
    var body: some View {
        Group {
            if animationComplete || !isAssistant || !adaptiveSpeedEnabled {
                MarkdownView(content: content)
            } else {
                MarkdownView(content: displayedContent)
            }
        }
        .onAppear {
            if isAssistant && adaptiveSpeedEnabled && !animationComplete {
                startTypewriterAnimation()
            } else {
                displayedContent = content
                animationComplete = true
            }
        }
        .onTapGesture(count: 2) {
            // Double tap to skip animation
            displayedContent = content
            animationComplete = true
            isAnimating = false
        }
    }
    
    private func startTypewriterAnimation() {
        guard !content.isEmpty && !isAnimating else { return }
        
        isAnimating = true
        displayedContent = ""
        
        // Split by lines for better UX
        let lines = content.components(separatedBy: "\n")
        var currentIndex = 0
        var currentLine = 0
        
        // Animate character by character with adaptive speed
        func animateNextCharacter() {
            guard isAnimating && currentLine < lines.count else {
                animationComplete = true
                displayedContent = content
                return
            }
            
            let line = lines[currentLine]
            
            if currentIndex < line.count {
                let index = line.index(line.startIndex, offsetBy: currentIndex)
                displayedContent += String(line[index])
                currentIndex += 1
                
                // Calculate delay based on character type
                var delay: Double = 1.0 / humanReadingSpeed
                
                let char = line[index]
                if char == "." || char == "!" || char == "?" {
                    delay *= 3 // Longer pause at sentence end
                } else if char == "," || char == ":" || char == ";" {
                    delay *= 1.5 // Medium pause at commas
                } else if char == "\n" {
                    delay *= 2 // Pause at new lines
                }
                
                // Faster for headers and code blocks
                if line.hasPrefix("#") || line.hasPrefix("```") || line.hasPrefix("|") {
                    delay = 1.0 / fastSpeed
                }
                
                DispatchQueue.main.asyncAfter(deadline: .now() + delay) {
                    animateNextCharacter()
                }
            } else {
                // Move to next line
                displayedContent += "\n"
                currentLine += 1
                currentIndex = 0
                
                DispatchQueue.main.asyncAfter(deadline: .now() + 0.05) {
                    animateNextCharacter()
                }
            }
        }
        
        animateNextCharacter()
    }
}

// MARK: - Streaming Bubble

struct StreamingBubble: View {
    @EnvironmentObject var theme: ThemeManager
    let content: String
    
    var body: some View {
        HStack(alignment: .top, spacing: 12) {
            // Avatar with glow
            ZStack {
                GlowPulse(color: theme.accentColor, size: 36)
                
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 16))
                    .foregroundColor(.white)
            }
            
            VStack(alignment: .leading, spacing: 6) {
                MarkdownView(content: content)
                    .padding(.horizontal, 16)
                    .padding(.vertical, 12)
                    .background(theme.messageGradient)
                    .cornerRadius(theme.radiusLarge, corners: [.topLeft, .topRight, .bottomRight])
                    .overlay(
                        RoundedRectangle(cornerRadius: theme.radiusLarge)
                            .stroke(theme.accentColor.opacity(0.3), lineWidth: 1)
                    )
                
                HStack(spacing: 4) {
                    ProgressView()
                        .scaleEffect(0.7)
                    Text("Streaming...")
                        .font(.system(size: 10))
                }
                .foregroundColor(theme.textMuted)
            }
            
            Spacer(minLength: 60)
        }
    }
}

// MARK: - Attachment Picker

// MARK: - Sophisticated File Picker

struct AttachmentPickerView: View {
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    @Environment(\.dismiss) var dismiss
    
    @State private var selectedSource: FileSource = .local
    @State private var showDocumentPicker = false
    @State private var showPhotoPicker = false
    @State private var showCameraPicker = false
    @State private var isLoadingCloud = false
    @State private var cloudFiles: [CloudFile] = []
    @State private var searchText = ""
    
    enum FileSource: String, CaseIterable {
        case local = "Appareil"
        case googleDrive = "Google Drive"
        case oneDrive = "OneDrive"
        case iCloud = "iCloud"
        
        var icon: String {
            switch self {
            case .local: return "iphone"
            case .googleDrive: return "globe"
            case .oneDrive: return "cloud.fill"
            case .iCloud: return "icloud.fill"
            }
        }
        
        var color: Color {
            switch self {
            case .local: return .blue
            case .googleDrive: return .green
            case .oneDrive: return .blue
            case .iCloud: return .cyan
            }
        }
    }
    
    var body: some View {
        NavigationStack {
            VStack(spacing: 0) {
                // Source selector
                ScrollView(.horizontal, showsIndicators: false) {
                    HStack(spacing: 12) {
                        ForEach(FileSource.allCases, id: \.self) { source in
                            SourceButton(
                                source: source,
                                isSelected: selectedSource == source
                            ) {
                                withAnimation(.spring(response: 0.3)) {
                                    selectedSource = source
                                    if source != .local {
                                        loadCloudFiles(from: source)
                                    }
                                }
                            }
                        }
                    }
                    .padding(.horizontal)
                    .padding(.vertical, 12)
                }
                .background(theme.surfaceColor)
                
                Divider()
                    .background(theme.primaryColor.opacity(0.3))
                
                // Content based on source
                if selectedSource == .local {
                    LocalFilesView(
                        showDocumentPicker: $showDocumentPicker,
                        showPhotoPicker: $showPhotoPicker,
                        showCameraPicker: $showCameraPicker,
                        onFileSelected: handleFileSelected
                    )
                } else {
                    CloudFilesView(
                        source: selectedSource,
                        isLoading: isLoadingCloud,
                        files: cloudFiles,
                        searchText: $searchText,
                        onFileSelected: handleCloudFileSelected
                    )
                }
            }
            .background(theme.backgroundColor)
            .navigationTitle("Joindre un fichier")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .cancellationAction) {
                    Button("Annuler") {
                        dismiss()
                    }
                    .foregroundColor(theme.accentColor)
                }
            }
        }
        .sheet(isPresented: $showDocumentPicker) {
            DocumentPickerView { urls in
                for url in urls {
                    addAttachmentFromURL(url)
                }
                dismiss()
            }
        }
    }
    
    private func loadCloudFiles(from source: FileSource) {
        isLoadingCloud = true
        cloudFiles = []
        
        // Simulate loading cloud files
        DispatchQueue.main.asyncAfter(deadline: .now() + 1) {
            cloudFiles = generateMockCloudFiles(for: source)
            isLoadingCloud = false
        }
    }
    
    private func generateMockCloudFiles(for source: FileSource) -> [CloudFile] {
        // Mock data - in production, this would call the actual cloud APIs
        return [
            CloudFile(name: "Rapport Q4 2024.pdf", type: .pdf, size: "2.4 MB", modified: Date()),
            CloudFile(name: "Budget 2025.xlsx", type: .spreadsheet, size: "1.1 MB", modified: Date().addingTimeInterval(-86400)),
            CloudFile(name: "Présentation.pptx", type: .presentation, size: "5.2 MB", modified: Date().addingTimeInterval(-172800)),
            CloudFile(name: "Notes de réunion.docx", type: .document, size: "156 KB", modified: Date().addingTimeInterval(-259200)),
        ]
    }
    
    private func handleFileSelected(_ attachment: Attachment) {
        chatManager.pendingAttachments.append(attachment)
        dismiss()
    }
    
    private func handleCloudFileSelected(_ file: CloudFile) {
        // Download and add attachment
        let attachment = Attachment(
            type: file.type,
            name: file.name,
            data: Data(), // Would be actual data from cloud
            mimeType: file.type.mimeType
        )
        chatManager.pendingAttachments.append(attachment)
        dismiss()
    }
    
    private func addAttachmentFromURL(_ url: URL) {
        guard url.startAccessingSecurityScopedResource() else { return }
        defer { url.stopAccessingSecurityScopedResource() }
        
        if let data = try? Data(contentsOf: url) {
            let type = AttachmentType.fromExtension(url.pathExtension)
            let attachment = Attachment(
                type: type,
                name: url.lastPathComponent,
                data: data,
                mimeType: type.mimeType
            )
            chatManager.pendingAttachments.append(attachment)
        }
    }
}

// MARK: - Source Button

struct SourceButton: View {
    @EnvironmentObject var theme: ThemeManager
    let source: AttachmentPickerView.FileSource
    let isSelected: Bool
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            VStack(spacing: 8) {
                ZStack {
                    Circle()
                        .fill(isSelected ? source.color : theme.surfaceElevated)
                        .frame(width: 50, height: 50)
                    
                    Image(systemName: source.icon)
                        .font(.system(size: 22))
                        .foregroundColor(isSelected ? .white : source.color)
                }
                
                Text(source.rawValue)
                    .font(.system(size: 11, weight: isSelected ? .semibold : .medium))
                    .foregroundColor(isSelected ? theme.textPrimary : theme.textSecondary)
            }
        }
        .buttonStyle(ScaleButtonStyle())
    }
}

// MARK: - Local Files View

struct LocalFilesView: View {
    @EnvironmentObject var theme: ThemeManager
    @Binding var showDocumentPicker: Bool
    @Binding var showPhotoPicker: Bool
    @Binding var showCameraPicker: Bool
    let onFileSelected: (Attachment) -> Void
    
    var body: some View {
        ScrollView {
            VStack(spacing: 16) {
                // Quick actions
                LazyVGrid(columns: [
                    GridItem(.flexible()),
                    GridItem(.flexible()),
                    GridItem(.flexible())
                ], spacing: 16) {
                    QuickActionCard(
                        icon: "doc.fill",
                        title: "Documents",
                        subtitle: "PDF, Word, Excel...",
                        color: .blue
                    ) {
                        showDocumentPicker = true
                    }
                    
                    QuickActionCard(
                        icon: "photo.fill",
                        title: "Photos",
                        subtitle: "Galerie",
                        color: .green
                    ) {
                        showPhotoPicker = true
                    }
                    
                    QuickActionCard(
                        icon: "camera.fill",
                        title: "Caméra",
                        subtitle: "Prendre une photo",
                        color: .orange
                    ) {
                        showCameraPicker = true
                    }
                }
                .padding(.horizontal)
                
                // File type sections
                VStack(alignment: .leading, spacing: 12) {
                    Text("Types de fichiers supportés")
                        .font(.system(size: 14, weight: .semibold))
                        .foregroundColor(theme.textSecondary)
                        .padding(.horizontal)
                    
                    ForEach(FileTypeGroup.allCases, id: \.self) { group in
                        FileTypeRow(group: group) {
                            showDocumentPicker = true
                        }
                    }
                }
                .padding(.top, 8)
            }
            .padding(.vertical)
        }
    }
}

// MARK: - Quick Action Card

struct QuickActionCard: View {
    @EnvironmentObject var theme: ThemeManager
    let icon: String
    let title: String
    let subtitle: String
    let color: Color
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            VStack(spacing: 10) {
                ZStack {
                    RoundedRectangle(cornerRadius: 14)
                        .fill(color.opacity(0.15))
                        .frame(width: 56, height: 56)
                    
                    Image(systemName: icon)
                        .font(.system(size: 24))
                        .foregroundColor(color)
                }
                
                VStack(spacing: 2) {
                    Text(title)
                        .font(.system(size: 13, weight: .semibold))
                        .foregroundColor(theme.textPrimary)
                    
                    Text(subtitle)
                        .font(.system(size: 10))
                        .foregroundColor(theme.textMuted)
                }
            }
            .frame(maxWidth: .infinity)
            .padding(.vertical, 16)
            .background(theme.surfaceColor)
            .cornerRadius(16)
            .overlay(
                RoundedRectangle(cornerRadius: 16)
                    .stroke(theme.surfaceElevated, lineWidth: 1)
            )
        }
        .buttonStyle(ScaleButtonStyle())
    }
}

// MARK: - File Type Group

enum FileTypeGroup: String, CaseIterable {
    case documents = "Documents"
    case spreadsheets = "Tableurs"
    case presentations = "Présentations"
    case images = "Images"
    case code = "Code"
    
    var icon: String {
        switch self {
        case .documents: return "doc.text.fill"
        case .spreadsheets: return "tablecells.fill"
        case .presentations: return "play.rectangle.fill"
        case .images: return "photo.fill"
        case .code: return "chevron.left.forwardslash.chevron.right"
        }
    }
    
    var color: Color {
        switch self {
        case .documents: return .blue
        case .spreadsheets: return .green
        case .presentations: return .orange
        case .images: return .purple
        case .code: return .pink
        }
    }
    
    var extensions: String {
        switch self {
        case .documents: return "PDF, DOC, DOCX, TXT"
        case .spreadsheets: return "XLS, XLSX, CSV"
        case .presentations: return "PPT, PPTX, KEY"
        case .images: return "PNG, JPG, GIF, WEBP"
        case .code: return "JSON, JS, PY, SWIFT"
        }
    }
}

struct FileTypeRow: View {
    @EnvironmentObject var theme: ThemeManager
    let group: FileTypeGroup
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            HStack(spacing: 14) {
                ZStack {
                    RoundedRectangle(cornerRadius: 10)
                        .fill(group.color.opacity(0.15))
                        .frame(width: 44, height: 44)
                    
                    Image(systemName: group.icon)
                        .font(.system(size: 18))
                        .foregroundColor(group.color)
                }
                
                VStack(alignment: .leading, spacing: 2) {
                    Text(group.rawValue)
                        .font(.system(size: 15, weight: .medium))
                        .foregroundColor(theme.textPrimary)
                    
                    Text(group.extensions)
                        .font(.system(size: 12))
                        .foregroundColor(theme.textMuted)
                }
                
                Spacer()
                
                Image(systemName: "chevron.right")
                    .font(.system(size: 12))
                    .foregroundColor(theme.textMuted)
            }
            .padding(.horizontal)
            .padding(.vertical, 12)
            .background(theme.surfaceColor)
        }
    }
}

// MARK: - Cloud Files View

struct CloudFilesView: View {
    @EnvironmentObject var theme: ThemeManager
    let source: AttachmentPickerView.FileSource
    let isLoading: Bool
    let files: [CloudFile]
    @Binding var searchText: String
    let onFileSelected: (CloudFile) -> Void
    
    var filteredFiles: [CloudFile] {
        if searchText.isEmpty { return files }
        return files.filter { $0.name.localizedCaseInsensitiveContains(searchText) }
    }
    
    var body: some View {
        VStack(spacing: 0) {
            // Search bar
            HStack(spacing: 10) {
                Image(systemName: "magnifyingglass")
                    .foregroundColor(theme.textMuted)
                
                TextField("Rechercher dans \(source.rawValue)...", text: $searchText)
                    .font(.system(size: 15))
                    .foregroundColor(theme.textPrimary)
                
                if !searchText.isEmpty {
                    Button(action: { searchText = "" }) {
                        Image(systemName: "xmark.circle.fill")
                            .foregroundColor(theme.textMuted)
                    }
                }
            }
            .padding(.horizontal, 14)
            .padding(.vertical, 10)
            .background(theme.surfaceColor)
            .cornerRadius(12)
            .padding()
            
            if isLoading {
                Spacer()
                VStack(spacing: 16) {
                    ProgressView()
                        .scaleEffect(1.2)
                    Text("Connexion à \(source.rawValue)...")
                        .font(.system(size: 14))
                        .foregroundColor(theme.textSecondary)
                }
                Spacer()
            } else if filteredFiles.isEmpty {
                Spacer()
                VStack(spacing: 12) {
                    Image(systemName: "doc.questionmark")
                        .font(.system(size: 48))
                        .foregroundColor(theme.textMuted)
                    
                    Text("Aucun fichier trouvé")
                        .font(.system(size: 16, weight: .medium))
                        .foregroundColor(theme.textSecondary)
                }
                Spacer()
            } else {
                ScrollView {
                    LazyVStack(spacing: 0) {
                        ForEach(filteredFiles) { file in
                            CloudFileRow(file: file) {
                                onFileSelected(file)
                            }
                            Divider()
                                .padding(.leading, 70)
                        }
                    }
                }
            }
        }
    }
}

// MARK: - Cloud File Model

struct CloudFile: Identifiable {
    let id = UUID()
    let name: String
    let type: AttachmentType
    let size: String
    let modified: Date
    
    var icon: String {
        switch type {
        case .pdf: return "doc.richtext.fill"
        case .document: return "doc.text.fill"
        case .spreadsheet: return "tablecells.fill"
        case .presentation: return "play.rectangle.fill"
        case .image: return "photo.fill"
        case .json, .code: return "chevron.left.forwardslash.chevron.right"
        }
    }
    
    var color: Color {
        switch type {
        case .pdf: return .red
        case .document: return .blue
        case .spreadsheet: return .green
        case .presentation: return .orange
        case .image: return .purple
        case .json, .code: return .pink
        }
    }
}

struct CloudFileRow: View {
    @EnvironmentObject var theme: ThemeManager
    let file: CloudFile
    let action: () -> Void
    
    var body: some View {
        Button(action: action) {
            HStack(spacing: 14) {
                ZStack {
                    RoundedRectangle(cornerRadius: 8)
                        .fill(file.color.opacity(0.15))
                        .frame(width: 44, height: 44)
                    
                    Image(systemName: file.icon)
                        .font(.system(size: 18))
                        .foregroundColor(file.color)
                }
                
                VStack(alignment: .leading, spacing: 3) {
                    Text(file.name)
                        .font(.system(size: 15, weight: .medium))
                        .foregroundColor(theme.textPrimary)
                        .lineLimit(1)
                    
                    HStack(spacing: 8) {
                        Text(file.size)
                        Text("•")
                        Text(file.modified, style: .relative)
                    }
                    .font(.system(size: 12))
                    .foregroundColor(theme.textMuted)
                }
                
                Spacer()
                
                Image(systemName: "plus.circle.fill")
                    .font(.system(size: 22))
                    .foregroundColor(theme.accentColor)
            }
            .padding(.horizontal)
            .padding(.vertical, 12)
        }
    }
}

// MARK: - Document Picker

struct DocumentPickerView: UIViewControllerRepresentable {
    let onPick: ([URL]) -> Void
    
    func makeUIViewController(context: Context) -> UIDocumentPickerViewController {
        let picker = UIDocumentPickerViewController(forOpeningContentTypes: [
            .pdf, .plainText, .json, .spreadsheet, .presentation,
            .image, .png, .jpeg, .gif, .webP
        ], asCopy: true)
        picker.allowsMultipleSelection = true
        picker.delegate = context.coordinator
        return picker
    }
    
    func updateUIViewController(_ uiViewController: UIDocumentPickerViewController, context: Context) {}
    
    func makeCoordinator() -> Coordinator {
        Coordinator(onPick: onPick)
    }
    
    class Coordinator: NSObject, UIDocumentPickerDelegate {
        let onPick: ([URL]) -> Void
        
        init(onPick: @escaping ([URL]) -> Void) {
            self.onPick = onPick
        }
        
        func documentPicker(_ controller: UIDocumentPickerViewController, didPickDocumentsAt urls: [URL]) {
            onPick(urls)
        }
    }
}

// MARK: - Attachment Type Extensions

extension AttachmentType {
    static func fromExtension(_ ext: String) -> AttachmentType {
        switch ext.lowercased() {
        case "pdf": return .pdf
        case "doc", "docx", "txt", "rtf": return .document
        case "xls", "xlsx", "csv": return .spreadsheet
        case "ppt", "pptx", "key": return .presentation
        case "png", "jpg", "jpeg", "gif", "webp", "heic": return .image
        case "json": return .json
        case "swift", "py", "js", "ts", "java", "go", "rs": return .code
        default: return .document
        }
    }
    
    var mimeType: String {
        switch self {
        case .pdf: return "application/pdf"
        case .document: return "application/msword"
        case .spreadsheet: return "application/vnd.ms-excel"
        case .presentation: return "application/vnd.ms-powerpoint"
        case .image: return "image/png"
        case .json: return "application/json"
        case .code: return "text/plain"
        }
    }
}

// MARK: - Corner Radius Extension

extension View {
    func cornerRadius(_ radius: CGFloat, corners: UIRectCorner) -> some View {
        clipShape(RoundedCorner(radius: radius, corners: corners))
    }
}

struct RoundedCorner: Shape {
    var radius: CGFloat = .infinity
    var corners: UIRectCorner = .allCorners
    
    func path(in rect: CGRect) -> Path {
        let path = UIBezierPath(
            roundedRect: rect,
            byRoundingCorners: corners,
            cornerRadii: CGSize(width: radius, height: radius)
        )
        return Path(path.cgPath)
    }
}

// ConnectionStatusBadge is defined in ThinkingStreamView.swift

// MARK: - Preview

#Preview {
    ChatView()
        .environmentObject(ChatManager.shared)
        .environmentObject(ThemeManager.shared)
        .environmentObject(AppState())
}
