//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Theme Manager - Dark Blood Theme                            ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

// MARK: - Color Extension (Single Definition)

extension Color {
    init(hex: String) {
        let hex = hex.trimmingCharacters(in: CharacterSet.alphanumerics.inverted)
        var int: UInt64 = 0
        Scanner(string: hex).scanHexInt64(&int)
        let a, r, g, b: UInt64
        switch hex.count {
        case 3: // RGB (12-bit)
            (a, r, g, b) = (255, (int >> 8) * 17, (int >> 4 & 0xF) * 17, (int & 0xF) * 17)
        case 6: // RGB (24-bit)
            (a, r, g, b) = (255, int >> 16, int >> 8 & 0xFF, int & 0xFF)
        case 8: // ARGB (32-bit)
            (a, r, g, b) = (int >> 24, int >> 16 & 0xFF, int >> 8 & 0xFF, int & 0xFF)
        default:
            (a, r, g, b) = (255, 0, 0, 0)
        }
        self.init(
            .sRGB,
            red: Double(r) / 255,
            green: Double(g) / 255,
            blue: Double(b) / 255,
            opacity: Double(a) / 255
        )
    }
}

// MARK: - Theme Colors (Static)

struct ThemeColors {
    // Primary - Deep Blood Red
    static let primary = Color(hex: "8B0000")
    static let primaryLight = Color(hex: "B22222")
    static let primaryDark = Color(hex: "5C0000")
    
    // Accent - Blood Red Glow
    static let accent = Color(hex: "DC143C")
    static let accentGlow = Color(hex: "FF2D2D")
    
    // Background - Deep Black with Red Tint
    static let background = Color(hex: "0D0D0D")
    static let backgroundSecondary = Color(hex: "1A0A0A")
    static let backgroundTertiary = Color(hex: "2D1515")
    
    // Surface - Card backgrounds
    static let surface = Color(hex: "1F1010")
    static let surfaceElevated = Color(hex: "2A1A1A")
    
    // Text
    static let textPrimary = Color(hex: "F5F5F5")
    static let textSecondary = Color(hex: "B0B0B0")
    static let textMuted = Color(hex: "6B6B6B")
    
    // Status Colors
    static let success = Color(hex: "228B22")
    static let warning = Color(hex: "FF8C00")
    static let error = Color(hex: "FF4444")
    static let info = Color(hex: "4169E1")
    
    // Gold Accent for AYMENOS branding
    static let goldAccent = Color(hex: "D4AF37")
    static let goldLight = Color(hex: "FFD700")
    static let goldDark = Color(hex: "B8860B")
}

// MARK: - Theme Manager

class ThemeManager: ObservableObject {
    static let shared = ThemeManager()
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Dark Blood Color Palette
    // ═══════════════════════════════════════════════════════════════════════════
    
    // Primary - Deep Blood Red
    let primaryColor = ThemeColors.primary
    let primaryLight = ThemeColors.primaryLight
    let primaryDark = ThemeColors.primaryDark
    
    // Accent - Blood Red Glow
    let accentColor = ThemeColors.accent
    let accentGlow = ThemeColors.accentGlow
    
    // Background - Deep Black with Red Tint
    let backgroundColor = ThemeColors.background
    let backgroundSecondary = ThemeColors.backgroundSecondary
    let backgroundTertiary = ThemeColors.backgroundTertiary
    
    // Surface - Card backgrounds
    let surfaceColor = ThemeColors.surface
    let surfaceElevated = ThemeColors.surfaceElevated
    
    // Text
    let textPrimary = ThemeColors.textPrimary
    let textSecondary = ThemeColors.textSecondary
    let textMuted = ThemeColors.textMuted
    
    // Status Colors
    let successColor = ThemeColors.success
    let warningColor = ThemeColors.warning
    let errorColor = ThemeColors.error
    let infoColor = ThemeColors.info
    
    // AYMENOS Gold Branding
    let goldAccent = ThemeColors.goldAccent
    let goldLight = ThemeColors.goldLight
    let goldDark = ThemeColors.goldDark
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Gradients
    // ═══════════════════════════════════════════════════════════════════════════
    
    var backgroundGradient: LinearGradient {
        LinearGradient(
            colors: [
                backgroundColor,
                backgroundSecondary,
                Color(hex: "150808")
            ],
            startPoint: .top,
            endPoint: .bottom
        )
    }
    
    var primaryGradient: LinearGradient {
        LinearGradient(
            colors: [primaryLight, primaryColor, primaryDark],
            startPoint: .topLeading,
            endPoint: .bottomTrailing
        )
    }
    
    var glowGradient: RadialGradient {
        RadialGradient(
            colors: [
                accentGlow.opacity(0.4),
                accentColor.opacity(0.2),
                Color.clear
            ],
            center: .center,
            startRadius: 0,
            endRadius: 150
        )
    }
    
    var messageGradient: LinearGradient {
        LinearGradient(
            colors: [
                surfaceElevated,
                surfaceColor
            ],
            startPoint: .top,
            endPoint: .bottom
        )
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Typography
    // ═══════════════════════════════════════════════════════════════════════════
    
    let fontTitle = Font.system(size: 28, weight: .bold, design: .rounded)
    let fontHeadline = Font.system(size: 20, weight: .semibold, design: .rounded)
    let fontBody = Font.system(size: 16, weight: .regular, design: .default)
    let fontCaption = Font.system(size: 12, weight: .medium, design: .default)
    let fontCode = Font.system(size: 14, weight: .regular, design: .monospaced)
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Corner Radius
    // ═══════════════════════════════════════════════════════════════════════════
    
    let radiusSmall: CGFloat = 8
    let radiusMedium: CGFloat = 12
    let radiusLarge: CGFloat = 20
    let radiusXL: CGFloat = 28
}

// MARK: - View Modifiers

struct BloodGlowModifier: ViewModifier {
    let intensity: Double
    
    func body(content: Content) -> some View {
        content
            .shadow(color: ThemeColors.accent.opacity(intensity * 0.6), radius: 15, x: 0, y: 0)
            .shadow(color: ThemeColors.primary.opacity(intensity * 0.4), radius: 25, x: 0, y: 5)
    }
}

struct CardStyle: ViewModifier {
    func body(content: Content) -> some View {
        content
            .background(ThemeColors.surface)
            .cornerRadius(12)
            .shadow(color: ThemeColors.primaryDark.opacity(0.3), radius: 8, x: 0, y: 4)
    }
}

struct PrimaryButtonStyle: ButtonStyle {
    func makeBody(configuration: Configuration) -> some View {
        configuration.label
            .padding(.horizontal, 24)
            .padding(.vertical, 14)
            .background(
                LinearGradient(
                    colors: [
                        ThemeColors.primaryLight,
                        ThemeColors.primary
                    ],
                    startPoint: .top,
                    endPoint: .bottom
                )
            )
            .foregroundColor(.white)
            .font(.system(size: 16, weight: .semibold))
            .cornerRadius(12)
            .scaleEffect(configuration.isPressed ? 0.95 : 1.0)
            .shadow(
                color: ThemeColors.accent.opacity(configuration.isPressed ? 0.8 : 0.4),
                radius: configuration.isPressed ? 20 : 10,
                x: 0,
                y: 4
            )
            .animation(.easeInOut(duration: 0.2), value: configuration.isPressed)
    }
}

// MARK: - View Extensions

extension View {
    func bloodGlow(intensity: Double = 1.0) -> some View {
        modifier(BloodGlowModifier(intensity: intensity))
    }
    
    func cardStyle() -> some View {
        modifier(CardStyle())
    }
}
