//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    WebSocket Manager                                           ║
//  ║         Real-Time Communication with Thinking Stream                          ║
//  ║         Instant Connectivity & Live Updates                                   ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import Combine

// MARK: - Event Types

enum WSEventType: String, Codable {
    case connected
    case disconnected
    case heartbeat
    case thinkingStart = "thinking_start"
    case thinkingStep = "thinking_step"
    case thinkingEnd = "thinking_end"
    case agentActivated = "agent_activated"
    case agentWorking = "agent_working"
    case agentCompleted = "agent_completed"
    case agentError = "agent_error"
    case toolInvoked = "tool_invoked"
    case toolResult = "tool_result"
    case streamStart = "stream_start"
    case streamChunk = "stream_chunk"
    case streamEnd = "stream_end"
    case panelData = "panel_data"
    case mutationStart = "mutation_start"
    case mutationSuccess = "mutation_success"
    case mutationError = "mutation_error"
    case subscriptionData = "subscription_data"
    case error
}

enum ThinkingStepType: String, Codable {
    case analyzing
    case routing
    case decomposing
    case planning
    case executing
    case collaborating
    case reformulating
    case validating
    case recovering
    
    var icon: String {
        switch self {
        case .analyzing: return "magnifyingglass"
        case .routing: return "arrow.triangle.branch"
        case .decomposing: return "square.split.2x2"
        case .planning: return "map"
        case .executing: return "bolt.fill"
        case .collaborating: return "person.3.fill"
        case .reformulating: return "sparkles"
        case .validating: return "checkmark.shield"
        case .recovering: return "arrow.counterclockwise"
        }
    }
    
    var color: Color {
        switch self {
        case .analyzing: return .blue
        case .routing: return .purple
        case .decomposing: return .orange
        case .planning: return .green
        case .executing: return .red
        case .collaborating: return .cyan
        case .reformulating: return .yellow
        case .validating: return .mint
        case .recovering: return .pink
        }
    }
}

// MARK: - Thinking Step Model

struct ThinkingStep: Identifiable, Equatable {
    let id: String
    let type: ThinkingStepType
    let title: String
    let description: String
    let agent: String?
    let progress: Double
    let timestamp: Date
    var status: StepStatus
    
    enum StepStatus: String {
        case inProgress = "in_progress"
        case completed = "completed"
        case failed = "failed"
    }
    
    static func == (lhs: ThinkingStep, rhs: ThinkingStep) -> Bool {
        lhs.id == rhs.id
    }
}

// MARK: - Connection State

enum ConnectionState: String {
    case disconnected
    case connecting
    case connected
    case reconnecting
    
    var statusText: String {
        switch self {
        case .disconnected: return "Déconnecté"
        case .connecting: return "Connexion..."
        case .connected: return "Connecté"
        case .reconnecting: return "Reconnexion..."
        }
    }
    
    var color: Color {
        switch self {
        case .disconnected: return .red
        case .connecting: return .orange
        case .connected: return .green
        case .reconnecting: return .yellow
        }
    }
}

// MARK: - WebSocket Manager

class WebSocketManager: ObservableObject {
    static let shared = WebSocketManager()
    
    // Connection state
    @Published var connectionState: ConnectionState = .disconnected
    @Published var connectionId: String?
    
    // Thinking state
    @Published var isThinking = false
    @Published var currentRequestId: String?
    @Published var thinkingSteps: [ThinkingStep] = []
    @Published var thinkingProgress: Double = 0
    
    // Streaming state
    @Published var isStreaming = false
    @Published var streamedContent = ""
    
    // Panel state
    @Published var showPanel = false
    @Published var panelType: String?
    @Published var panelAttachments: [PanelAttachment] = []
    
    private var webSocketTask: URLSessionWebSocketTask?
    private var session: URLSession!
    private var pingTimer: Timer?
    private var reconnectAttempts = 0
    private let maxReconnectAttempts = 5
    private let baseURL = "wss://assistant.dmagh.me/api/v1"
    
    private init() {
        session = URLSession(configuration: .default)
    }
    
    // MARK: - Connection
    
    func connect(userId: String = "dmagh") {
        guard connectionState == .disconnected else { return }
        
        DispatchQueue.main.async {
            self.connectionState = .connecting
        }
        
        guard let url = URL(string: "\(baseURL)/ws?user_id=\(userId)") else {
            DispatchQueue.main.async {
                self.connectionState = .disconnected
            }
            return
        }
        
        var request = URLRequest(url: url)
        request.timeoutInterval = 60
        
        if let token = AuthManager.shared.accessToken {
            request.setValue("Bearer \(token)", forHTTPHeaderField: "Authorization")
        }
        
        webSocketTask = session.webSocketTask(with: request)
        webSocketTask?.resume()
        
        receiveMessage()
        startPingTimer()
    }
    
    func disconnect() {
        stopPingTimer()
        webSocketTask?.cancel(with: .normalClosure, reason: nil)
        webSocketTask = nil
        DispatchQueue.main.async {
            self.connectionState = .disconnected
            self.connectionId = nil
        }
        reconnectAttempts = 0
    }
    
    private func reconnect() {
        guard reconnectAttempts < maxReconnectAttempts else {
            DispatchQueue.main.async {
                self.connectionState = .disconnected
            }
            return
        }
        
        DispatchQueue.main.async {
            self.connectionState = .reconnecting
        }
        reconnectAttempts += 1
        
        let delay = Double(min(30, pow(2.0, Double(reconnectAttempts))))
        
        DispatchQueue.main.asyncAfter(deadline: .now() + delay) { [weak self] in
            self?.connectionState = .disconnected
            self?.connect()
        }
    }
    
    // MARK: - Messaging
    
    private func receiveMessage() {
        webSocketTask?.receive { [weak self] result in
            switch result {
            case .success(let message):
                switch message {
                case .string(let text):
                    self?.handleMessage(text)
                case .data(let data):
                    if let text = String(data: data, encoding: .utf8) {
                        self?.handleMessage(text)
                    }
                @unknown default:
                    break
                }
                self?.receiveMessage()
                
            case .failure(let error):
                print("WebSocket receive error: \(error)")
                DispatchQueue.main.async {
                    self?.connectionState = .disconnected
                }
                self?.reconnect()
            }
        }
    }
    
    private func handleMessage(_ text: String) {
        guard let data = text.data(using: .utf8) else { return }
        
        do {
            if let json = try JSONSerialization.jsonObject(with: data) as? [String: Any],
               let typeStr = json["type"] as? String {
                
                DispatchQueue.main.async { [weak self] in
                    self?.processEvent(type: typeStr, json: json)
                }
            }
        } catch {
            print("JSON parse error: \(error)")
        }
    }
    
    private func processEvent(type: String, json: [String: Any]) {
        let eventData = json["data"] as? [String: Any] ?? [:]
        
        switch type {
        case "connected":
            connectionState = .connected
            connectionId = eventData["connection_id"] as? String
            reconnectAttempts = 0
            
        case "heartbeat":
            break
            
        case "thinking_start":
            isThinking = true
            currentRequestId = json["request_id"] as? String
            thinkingSteps.removeAll()
            thinkingProgress = 0
            
        case "thinking_step":
            if let step = parseThinkingStep(eventData) {
                thinkingSteps.append(step)
                // Ensure progress is a valid number for UI
                let safeProgress = step.progress
                if !safeProgress.isNaN && !safeProgress.isInfinite {
                    thinkingProgress = max(0, min(1, safeProgress))
                }
                
                // Add to collaboration history
                addToCollaborationHistory(eventData)
            }
            
        case "tool_invoked":
            // Tool execution step
            addToolExecutionToHistory(eventData)
            
        case "tool_result":
            // Tool result
            updateToolResultInHistory(eventData)
            
        case "thinking_end":
            thinkingProgress = 1.0
            for i in thinkingSteps.indices {
                thinkingSteps[i].status = .completed
            }
            DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) { [weak self] in
                self?.isThinking = false
            }
            
        case "stream_start":
            isStreaming = true
            streamedContent = ""
            
        case "stream_chunk":
            if let content = eventData["content"] as? String {
                streamedContent += content
            }
            
        case "stream_end":
            isStreaming = false
            
        case "panel_data":
            showPanel = eventData["show_panel"] as? Bool ?? true
            panelType = eventData["panel_type"] as? String
            
        case "error":
            print("WebSocket error: \(eventData["message"] ?? "Unknown")")
            
        default:
            break
        }
    }
    
    private func parseThinkingStep(_ data: [String: Any]) -> ThinkingStep? {
        guard let id = data["id"] as? String,
              let typeStr = data["type"] as? String,
              let type = ThinkingStepType(rawValue: typeStr),
              let title = data["title"] as? String,
              let description = data["description"] as? String else {
            return nil
        }
        
        // Safely parse progress - ensure it's a valid number
        var progress: Double = 0
        if let rawProgress = data["progress"] as? Double {
            progress = rawProgress.isNaN || rawProgress.isInfinite ? 0 : max(0, min(1, rawProgress))
        } else if let intProgress = data["progress"] as? Int {
            progress = max(0, min(1, Double(intProgress)))
        }
        
        return ThinkingStep(
            id: id,
            type: type,
            title: title,
            description: description,
            agent: data["agent"] as? String,
            progress: progress,
            timestamp: Date(),
            status: .inProgress
        )
    }
    
    // MARK: - Send Messages
    
    func send(_ message: [String: Any]) {
        guard let data = try? JSONSerialization.data(withJSONObject: message),
              let text = String(data: data, encoding: .utf8) else {
            return
        }
        
        webSocketTask?.send(.string(text)) { error in
            if let error = error {
                print("WebSocket send error: \(error)")
            }
        }
    }
    
    func sendChat(content: String, conversationId: String? = nil) {
        send([
            "type": "chat",
            "content": content,
            "conversation_id": conversationId ?? UUID().uuidString
        ])
    }
    
    func subscribe(to conversationId: String) {
        send([
            "type": "subscribe",
            "conversation_id": conversationId
        ])
    }
    
    func ping() {
        send(["type": "ping"])
    }
    
    // MARK: - Helpers
    
    private func startPingTimer() {
        DispatchQueue.main.async { [weak self] in
            self?.pingTimer = Timer.scheduledTimer(withTimeInterval: 25, repeats: true) { [weak self] _ in
                self?.ping()
            }
        }
    }
    
    private func stopPingTimer() {
        pingTimer?.invalidate()
        pingTimer = nil
    }
    
    func clearThinkingState() {
        isThinking = false
        thinkingSteps.removeAll()
        thinkingProgress = 0
        currentRequestId = nil
    }
    
    // MARK: - Unified Panel Integration
    
    private func addToCollaborationHistory(_ data: [String: Any]) {
        let panelManager = UnifiedPanelManager.shared
        
        // Start session if needed
        if panelManager.currentSession == nil, let requestId = currentRequestId {
            DispatchQueue.main.async {
                panelManager.startSession(query: "Requête en cours...", requestId: requestId)
            }
        }
        
        // Map thinking step type
        guard let typeStr = data["type"] as? String else { return }
        
        let stepType: AIStep.StepType
        switch typeStr {
        case "analyzing": stepType = .analyzing
        case "routing": stepType = .routing
        case "planning": stepType = .planning
        case "decomposing": stepType = .decomposing
        case "executing": stepType = .executing
        case "collaborating": stepType = .collaboration
        case "reformulating": stepType = .reformulation
        case "validating": stepType = .validation
        case "recovering": stepType = .fallback
        default: stepType = .executing
        }
        
        let step = AIStep(
            type: stepType,
            title: data["title"] as? String ?? typeStr,
            description: data["description"] as? String ?? "",
            agent: data["agent"] as? String
        )
        
        DispatchQueue.main.async {
            panelManager.addStep(step)
        }
    }
    
    private func addToolExecutionToHistory(_ data: [String: Any]) {
        let panelManager = UnifiedPanelManager.shared
        
        guard let toolName = data["tool"] as? String else { return }
        
        // Determine step type
        let stepType: AIStep.StepType
        if toolName.contains("shell") || toolName.contains("command") || toolName.contains("exec") {
            stepType = .shellCommand
        } else if toolName.contains("api") || toolName.contains("http") || toolName.contains("request") {
            stepType = .apiCall
        } else if toolName.contains("scrape") || toolName.contains("crawl") || toolName.contains("fetch") {
            stepType = .webScraping
        } else if toolName.contains("read") || toolName.contains("get") {
            stepType = .documentRead
        } else if toolName.contains("write") || toolName.contains("create") || toolName.contains("update") {
            stepType = .documentWrite
        } else if toolName.contains("search") {
            stepType = .webSearch
        } else if toolName.contains("query") || toolName.contains("sql") || toolName.contains("database") {
            stepType = .databaseQuery
        } else if toolName.contains("code") || toolName.contains("generate") {
            stepType = .codeGeneration
        } else if toolName.contains("run") || toolName.contains("execute") {
            stepType = .codeExecution
        } else {
            stepType = .toolExecution
        }
        
        var step = AIStep(
            type: stepType,
            title: "🛠 \(toolName)",
            description: data["description"] as? String ?? "Exécution de l'outil \(toolName)",
            agent: data["agent"] as? String,
            tool: toolName
        )
        step.parameters = data["parameters"] as? [String: Any]
        
        DispatchQueue.main.async {
            panelManager.addStep(step)
        }
    }
    
    private func updateToolResultInHistory(_ data: [String: Any]) {
        let panelManager = UnifiedPanelManager.shared
        
        guard let invocationId = data["invocation_id"] as? String else { return }
        
        let success = data["success"] as? Bool ?? true
        let output = data["result"] as? String ?? data["output"] as? String
        let error = success ? nil : (data["error"] as? String)
        
        DispatchQueue.main.async {
            panelManager.updateStep(
                id: invocationId,
                status: success ? .completed : .failed,
                output: output,
                error: error
            )
        }
        
        // Send notification for tool failures
        if !success {
            DispatchQueue.main.async {
                NotificationManager.shared.sendToolExecutionNotification(
                    toolName: data["tool"] as? String ?? "Unknown",
                    success: false,
                    result: error ?? output
                )
            }
        }
    }
    
    func endCollaborationSession(withOutput output: String?) {
        DispatchQueue.main.async {
            UnifiedPanelManager.shared.endSession(output: output, success: true)
        }
    }
}
