//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Voice Manager - Speech Recognition & Synthesis              ║
//  ║                    With Automatic Silence Detection                           ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import Speech
import AVFoundation
import Combine

class VoiceManager: NSObject, ObservableObject {
    static let shared = VoiceManager()
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Published State
    // ═══════════════════════════════════════════════════════════════════════════
    
    @Published var isRecording = false
    @Published var isSpeaking = false
    @Published var transcribedText = ""
    @Published var audioLevel: Float = 0
    @Published var hasPermission = false
    @Published var error: String?
    @Published var silenceDetected = false
    @Published var recordingDuration: TimeInterval = 0
    
    // Callbacks
    var onSilenceDetected: (() -> Void)?
    var onAutoStop: ((String) -> Void)?
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Speech Recognition
    // ═══════════════════════════════════════════════════════════════════════════
    
    private let speechRecognizer = SFSpeechRecognizer(locale: Locale(identifier: "fr-FR"))
    private var recognitionRequest: SFSpeechAudioBufferRecognitionRequest?
    private var recognitionTask: SFSpeechRecognitionTask?
    private let audioEngine = AVAudioEngine()
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Silence Detection
    // ═══════════════════════════════════════════════════════════════════════════
    
    private var silenceTimer: Timer?
    private var recordingTimer: Timer?
    private var lastSpeechTime: Date = Date()
    private let silenceThreshold: Float = 0.05
    private let silenceDuration: TimeInterval = 2.0  // 2 seconds of silence
    private let maxRecordingDuration: TimeInterval = 60.0  // Max 60 seconds
    private var autoStopEnabled = true
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Audio Level History for smoothing
    // ═══════════════════════════════════════════════════════════════════════════
    
    private var audioLevelHistory: [Float] = []
    private let audioLevelHistorySize = 10
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Text-to-Speech
    // ═══════════════════════════════════════════════════════════════════════════
    
    private let synthesizer = AVSpeechSynthesizer()
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Initialization
    // ═══════════════════════════════════════════════════════════════════════════
    
    override init() {
        super.init()
        synthesizer.delegate = self
        requestPermissions()
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Permissions
    // ═══════════════════════════════════════════════════════════════════════════
    
    func requestPermissions() {
        SFSpeechRecognizer.requestAuthorization { [weak self] status in
            DispatchQueue.main.async {
                switch status {
                case .authorized:
                    self?.hasPermission = true
                case .denied, .restricted, .notDetermined:
                    self?.hasPermission = false
                    self?.error = "Permission de reconnaissance vocale refusée"
                @unknown default:
                    self?.hasPermission = false
                }
            }
        }
        
        AVAudioApplication.requestRecordPermission { [weak self] granted in
            DispatchQueue.main.async {
                if !granted {
                    self?.hasPermission = false
                    self?.error = "Permission microphone refusée"
                }
            }
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Speech Recognition
    // ═══════════════════════════════════════════════════════════════════════════
    
    func startRecording(autoStop: Bool = true) {
        guard hasPermission else {
            requestPermissions()
            return
        }
        
        autoStopEnabled = autoStop
        
        // Cancel any previous task
        if recognitionTask != nil {
            recognitionTask?.cancel()
            recognitionTask = nil
        }
        
        // Reset state
        silenceDetected = false
        recordingDuration = 0
        lastSpeechTime = Date()
        audioLevelHistory.removeAll()
        
        // Configure audio session
        let audioSession = AVAudioSession.sharedInstance()
        do {
            try audioSession.setCategory(.record, mode: .measurement, options: .duckOthers)
            try audioSession.setActive(true, options: .notifyOthersOnDeactivation)
        } catch {
            self.error = "Erreur configuration audio: \(error.localizedDescription)"
            return
        }
        
        // Create recognition request
        recognitionRequest = SFSpeechAudioBufferRecognitionRequest()
        
        guard let recognitionRequest = recognitionRequest else {
            self.error = "Impossible de créer la requête de reconnaissance"
            return
        }
        
        recognitionRequest.shouldReportPartialResults = true
        recognitionRequest.requiresOnDeviceRecognition = false
        
        // Configure audio input
        let inputNode = audioEngine.inputNode
        
        // Start recognition task
        recognitionTask = speechRecognizer?.recognitionTask(with: recognitionRequest) { [weak self] result, error in
            DispatchQueue.main.async {
                if let result = result {
                    let newText = result.bestTranscription.formattedString
                    if newText != self?.transcribedText {
                        self?.transcribedText = newText
                        self?.lastSpeechTime = Date()
                        self?.silenceDetected = false
                    }
                }
                
                if let error = error {
                    // Ignore certain errors that are expected
                    let nsError = error as NSError
                    if nsError.domain != "kAFAssistantErrorDomain" {
                        self?.error = error.localizedDescription
                    }
                    self?.stopRecording()
                }
            }
        }
        
        // Install tap on input node
        let recordingFormat = inputNode.outputFormat(forBus: 0)
        inputNode.installTap(onBus: 0, bufferSize: 1024, format: recordingFormat) { [weak self] buffer, _ in
            self?.recognitionRequest?.append(buffer)
            
            // Calculate audio level with smoothing
            let level = self?.calculateSmoothedAudioLevel(buffer: buffer) ?? 0
            DispatchQueue.main.async {
                self?.audioLevel = level
                self?.checkForSilence(level: level)
            }
        }
        
        // Start audio engine
        audioEngine.prepare()
        
        do {
            try audioEngine.start()
            DispatchQueue.main.async {
                self.isRecording = true
                self.transcribedText = ""
                self.startTimers()
            }
        } catch {
            self.error = "Erreur démarrage audio: \(error.localizedDescription)"
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Silence Detection
    // ═══════════════════════════════════════════════════════════════════════════
    
    private func startTimers() {
        // Recording duration timer
        recordingTimer = Timer.scheduledTimer(withTimeInterval: 0.1, repeats: true) { [weak self] _ in
            guard let self = self, self.isRecording else { return }
            
            DispatchQueue.main.async {
                self.recordingDuration += 0.1
                
                // Check max duration
                if self.recordingDuration >= self.maxRecordingDuration {
                    self.autoStopRecording()
                }
            }
        }
        
        // Silence detection timer
        silenceTimer = Timer.scheduledTimer(withTimeInterval: 0.5, repeats: true) { [weak self] _ in
            guard let self = self, self.isRecording, self.autoStopEnabled else { return }
            
            let silenceTime = Date().timeIntervalSince(self.lastSpeechTime)
            
            DispatchQueue.main.async {
                // Only trigger if we have some transcribed text and silence is detected
                if silenceTime >= self.silenceDuration && !self.transcribedText.isEmpty {
                    self.silenceDetected = true
                    self.onSilenceDetected?()
                    
                    // Auto-stop after additional delay
                    if silenceTime >= self.silenceDuration + 1.0 {
                        self.autoStopRecording()
                    }
                }
            }
        }
    }
    
    private func checkForSilence(level: Float) {
        if level > silenceThreshold {
            lastSpeechTime = Date()
            silenceDetected = false
        }
    }
    
    private func autoStopRecording() {
        let text = transcribedText
        stopRecording()
        
        if !text.isEmpty {
            onAutoStop?(text)
        }
    }
    
    private func calculateSmoothedAudioLevel(buffer: AVAudioPCMBuffer) -> Float {
        let rawLevel = calculateAudioLevel(buffer: buffer)
        
        // Add to history
        audioLevelHistory.append(rawLevel)
        if audioLevelHistory.count > audioLevelHistorySize {
            audioLevelHistory.removeFirst()
        }
        
        // Return smoothed average
        return audioLevelHistory.reduce(0, +) / Float(audioLevelHistory.count)
    }
    
    func stopRecording() {
        // Stop timers
        silenceTimer?.invalidate()
        silenceTimer = nil
        recordingTimer?.invalidate()
        recordingTimer = nil
        
        // Stop audio engine
        audioEngine.stop()
        audioEngine.inputNode.removeTap(onBus: 0)
        
        recognitionRequest?.endAudio()
        recognitionRequest = nil
        
        recognitionTask?.cancel()
        recognitionTask = nil
        
        DispatchQueue.main.async {
            self.isRecording = false
            self.audioLevel = 0
            self.silenceDetected = false
        }
    }
    
    func pauseRecording() {
        if isRecording {
            audioEngine.pause()
            silenceTimer?.invalidate()
        }
    }
    
    func resumeRecording() {
        if !audioEngine.isRunning && isRecording {
            try? audioEngine.start()
            startTimers()
        }
    }
    
    private func calculateAudioLevel(buffer: AVAudioPCMBuffer) -> Float {
        guard let channelData = buffer.floatChannelData else { return 0 }
        
        let channelDataValue = channelData.pointee
        let channelDataValueArray = stride(from: 0, to: Int(buffer.frameLength), by: buffer.stride).map { channelDataValue[$0] }
        
        let rms = sqrt(channelDataValueArray.map { $0 * $0 }.reduce(0, +) / Float(buffer.frameLength))
        let avgPower = 20 * log10(rms)
        let normalizedValue = (avgPower + 50) / 50 // Normalize to 0-1 range
        
        return max(0, min(1, normalizedValue))
    }
    
    // ═══════════════════════════════════════════════════════════════════════════
    // Text-to-Speech
    // ═══════════════════════════════════════════════════════════════════════════
    
    func speak(_ text: String, rate: Float = 0.5) {
        // Stop any current speech
        if synthesizer.isSpeaking {
            synthesizer.stopSpeaking(at: .immediate)
        }
        
        let utterance = AVSpeechUtterance(string: text)
        utterance.voice = AVSpeechSynthesisVoice(language: "fr-FR")
        utterance.rate = rate
        utterance.pitchMultiplier = 1.0
        utterance.volume = 1.0
        
        // Configure audio session for playback
        do {
            try AVAudioSession.sharedInstance().setCategory(.playback, mode: .default)
            try AVAudioSession.sharedInstance().setActive(true)
        } catch {
            self.error = "Erreur configuration audio: \(error.localizedDescription)"
            return
        }
        
        DispatchQueue.main.async {
            self.isSpeaking = true
        }
        
        synthesizer.speak(utterance)
    }
    
    func stopSpeaking() {
        if synthesizer.isSpeaking {
            synthesizer.stopSpeaking(at: .immediate)
        }
        
        DispatchQueue.main.async {
            self.isSpeaking = false
        }
    }
}

// MARK: - AVSpeechSynthesizerDelegate

extension VoiceManager: AVSpeechSynthesizerDelegate {
    func speechSynthesizer(_ synthesizer: AVSpeechSynthesizer, didFinish utterance: AVSpeechUtterance) {
        DispatchQueue.main.async {
            self.isSpeaking = false
        }
    }
    
    func speechSynthesizer(_ synthesizer: AVSpeechSynthesizer, didCancel utterance: AVSpeechUtterance) {
        DispatchQueue.main.async {
            self.isSpeaking = false
        }
    }
}

// MARK: - Voice Button Component

struct VoiceButton: View {
    @StateObject private var voiceManager = VoiceManager.shared
    @EnvironmentObject var theme: ThemeManager
    @EnvironmentObject var chatManager: ChatManager
    
    @State private var isPulsing = false
    
    var body: some View {
        Button(action: toggleRecording) {
            ZStack {
                // Pulse animation when recording
                if voiceManager.isRecording {
                    ForEach(0..<3, id: \.self) { i in
                        Circle()
                            .stroke(theme.accentColor.opacity(0.5 - Double(i) * 0.15), lineWidth: 2)
                            .scaleEffect(isPulsing ? 1 + CGFloat(i) * 0.3 : 1)
                            .opacity(isPulsing ? 0 : 1)
                            .animation(
                                Animation.easeOut(duration: 1)
                                    .repeatForever(autoreverses: false)
                                    .delay(Double(i) * 0.2),
                                value: isPulsing
                            )
                    }
                }
                
                // Main button
                Circle()
                    .fill(
                        voiceManager.isRecording
                        ? theme.accentGlow
                        : theme.surfaceElevated
                    )
                    .frame(width: 56, height: 56)
                    .shadow(
                        color: voiceManager.isRecording ? theme.accentGlow.opacity(0.5) : .clear,
                        radius: 10
                    )
                
                // Audio level indicator
                if voiceManager.isRecording {
                    Circle()
                        .stroke(theme.accentColor, lineWidth: 3)
                        .frame(width: 56 + CGFloat(voiceManager.audioLevel) * 20,
                               height: 56 + CGFloat(voiceManager.audioLevel) * 20)
                        .animation(.easeOut(duration: 0.1), value: voiceManager.audioLevel)
                }
                
                // Icon
                Image(systemName: voiceManager.isRecording ? "waveform" : "mic.fill")
                    .font(.system(size: 22))
                    .foregroundColor(voiceManager.isRecording ? .white : theme.textPrimary)
            }
        }
        .onAppear {
            if voiceManager.isRecording {
                isPulsing = true
            }
        }
        .onChange(of: voiceManager.isRecording) { _, newValue in
            isPulsing = newValue
        }
    }
    
    private func toggleRecording() {
        if voiceManager.isRecording {
            voiceManager.stopRecording()
            
            // Send transcribed text
            if !voiceManager.transcribedText.isEmpty {
                Task {
                    await chatManager.sendMessage(voiceManager.transcribedText, isVoice: true)
                }
            }
        } else {
            voiceManager.startRecording()
        }
    }
}

// MARK: - Audio Waveform View

struct AudioWaveform: View {
    @EnvironmentObject var theme: ThemeManager
    let level: Float
    
    var body: some View {
        HStack(spacing: 3) {
            ForEach(0..<5, id: \.self) { i in
                Capsule()
                    .fill(theme.accentColor)
                    .frame(width: 3, height: CGFloat(10 + Float(i * 5) * level))
                    .animation(.easeInOut(duration: 0.1), value: level)
            }
            ForEach((0..<5).reversed(), id: \.self) { i in
                Capsule()
                    .fill(theme.accentColor)
                    .frame(width: 3, height: CGFloat(10 + Float(i * 5) * level))
                    .animation(.easeInOut(duration: 0.1), value: level)
            }
        }
    }
}
