//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Notification Manager                                        ║
//  ║         Sophisticated Event-Based Push Notification System                    ║
//  ║         Thread-Safe & Multi-Event Support                                     ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import UserNotifications

// MARK: - Notification Event Types

enum NotificationEventType: String {
    case assistantResponse = "assistant_response"
    case taskStarted = "task_started"
    case taskCompleted = "task_completed"
    case taskFailed = "task_failed"
    case actionRequired = "action_required"
    case sensitiveAction = "sensitive_action"
    case toolExecution = "tool_execution"
    case infrastructureAlert = "infrastructure_alert"
    case syncComplete = "sync_complete"
    case connectionStatus = "connection_status"
    
    var category: String {
        switch self {
        case .assistantResponse: return "ASSISTANT_RESPONSE"
        case .taskStarted, .taskCompleted, .taskFailed: return "TASK_STATUS"
        case .actionRequired, .sensitiveAction: return "ACTION_CONFIRMATION"
        case .toolExecution: return "TOOL_STATUS"
        case .infrastructureAlert: return "INFRA_ALERT"
        case .syncComplete: return "SYNC_STATUS"
        case .connectionStatus: return "CONNECTION"
        }
    }
    
    var defaultSound: UNNotificationSound {
        switch self {
        case .sensitiveAction, .infrastructureAlert:
            return .defaultCritical
        case .taskFailed:
            return .default
        default:
            return .default
        }
    }
    
    var interruptionLevel: UNNotificationInterruptionLevel {
        switch self {
        case .sensitiveAction, .infrastructureAlert:
            return .critical
        case .actionRequired:
            return .timeSensitive
        default:
            return .active
        }
    }
}

// MARK: - Notification Manager

@MainActor
class NotificationManager: ObservableObject {
    static let shared = NotificationManager()
    
    @Published var pendingNotifications: [String] = []
    @Published var isAuthorized = false
    @Published var notificationCount = 0
    
    private var lastNotificationTime: [NotificationEventType: Date] = [:]
    private let minimumInterval: TimeInterval = 2.0 // Prevent spam
    
    private init() {
        checkAuthorization()
    }
    
    // MARK: - Authorization
    
    func checkAuthorization() {
        UNUserNotificationCenter.current().getNotificationSettings { [weak self] settings in
            DispatchQueue.main.async {
                self?.isAuthorized = settings.authorizationStatus == .authorized
            }
        }
    }
    
    func requestAuthorization() async -> Bool {
        do {
            let granted = try await UNUserNotificationCenter.current().requestAuthorization(
                options: [.alert, .sound, .badge, .criticalAlert, .providesAppNotificationSettings]
            )
            await MainActor.run {
                self.isAuthorized = granted
            }
            return granted
        } catch {
            print("[NotificationManager] Authorization error: \(error)")
            return false
        }
    }
    
    // MARK: - Send Notifications
    
    func sendAssistantResponseNotification(title: String, message: String, toolsUsed: [String]?) {
        // Check if app is in background
        let appState = UIApplication.shared.applicationState
        print("[NotificationManager] App state: \(appState.rawValue)")
        
        // Only send if in background or inactive
        guard appState != .active else {
            print("[NotificationManager] App is active, skipping notification")
            return
        }
        
        sendNotification(
            type: .assistantResponse,
            title: title,
            body: message,
            subtitle: toolsUsed.map { "🛠 \($0.prefix(3).joined(separator: ", "))" },
            userInfo: ["tools": toolsUsed ?? []]
        )
    }
    
    func sendTaskNotification(status: NotificationEventType, taskName: String, details: String? = nil) {
        let title: String
        let body: String
        
        switch status {
        case .taskStarted:
            title = "🚀 Tâche démarrée"
            body = taskName
        case .taskCompleted:
            title = "✅ Tâche terminée"
            body = taskName + (details.map { "\n\($0)" } ?? "")
        case .taskFailed:
            title = "❌ Échec de la tâche"
            body = taskName + (details.map { "\n\($0)" } ?? "")
        default:
            return
        }
        
        sendNotification(type: status, title: title, body: body)
    }
    
    func sendActionRequiredNotification(actionId: String, model: String, method: String, sensitivityLevel: String) {
        let title = sensitivityLevel == "critical" ? "⚠️ Action Critique" : "⚡ Confirmation Requise"
        let body = "Action \(method) sur \(model) en attente de confirmation"
        
        sendNotification(
            type: sensitivityLevel == "critical" ? .sensitiveAction : .actionRequired,
            title: title,
            body: body,
            userInfo: [
                "action_id": actionId,
                "model": model,
                "method": method,
                "sensitivity": sensitivityLevel
            ]
        )
    }
    
    func sendInfrastructureAlert(service: String, status: String, message: String) {
        sendNotification(
            type: .infrastructureAlert,
            title: "🐳 Alerte Infrastructure",
            body: "\(service): \(message)",
            subtitle: status,
            userInfo: ["service": service, "status": status]
        )
    }
    
    func sendToolExecutionNotification(toolName: String, success: Bool, result: String?) {
        // Only notify for failures or long-running tools
        guard !success else { return }
        
        sendNotification(
            type: .toolExecution,
            title: "🛠 Outil: \(toolName)",
            body: success ? "Exécution réussie" : "Échec: \(result ?? "Erreur inconnue")",
            userInfo: ["tool": toolName, "success": success]
        )
    }
    
    // MARK: - Core Send Method
    
    private func sendNotification(
        type: NotificationEventType,
        title: String,
        body: String,
        subtitle: String? = nil,
        userInfo: [String: Any] = [:]
    ) {
        // Rate limiting
        if let lastTime = lastNotificationTime[type],
           Date().timeIntervalSince(lastTime) < minimumInterval {
            print("[NotificationManager] Rate limited: \(type.rawValue)")
            return
        }
        
        guard isAuthorized else {
            print("[NotificationManager] Not authorized")
            return
        }
        
        let content = UNMutableNotificationContent()
        content.title = title
        content.body = body
        content.sound = type.defaultSound
        content.categoryIdentifier = type.category
        content.interruptionLevel = type.interruptionLevel
        
        if let subtitle = subtitle {
            content.subtitle = subtitle
        }
        
        // Badge
        notificationCount += 1
        content.badge = NSNumber(value: notificationCount)
        
        // User info
        var finalUserInfo = userInfo
        finalUserInfo["type"] = type.rawValue
        finalUserInfo["timestamp"] = Date().timeIntervalSince1970
        content.userInfo = finalUserInfo
        
        // Thread identifier for grouping
        content.threadIdentifier = type.category
        
        // Create request
        let identifier = "\(type.rawValue)_\(UUID().uuidString.prefix(8))"
        let trigger = UNTimeIntervalNotificationTrigger(timeInterval: 0.5, repeats: false)
        let request = UNNotificationRequest(identifier: identifier, content: content, trigger: trigger)
        
        // Schedule
        UNUserNotificationCenter.current().add(request) { [weak self] error in
            DispatchQueue.main.async {
                if let error = error {
                    print("[NotificationManager] ❌ Error: \(error.localizedDescription)")
                } else {
                    print("[NotificationManager] ✅ Scheduled: \(identifier)")
                    self?.lastNotificationTime[type] = Date()
                    self?.pendingNotifications.append(identifier)
                }
            }
        }
    }
    
    // MARK: - Badge Management
    
    func clearBadge() {
        notificationCount = 0
        UNUserNotificationCenter.current().setBadgeCount(0)
    }
    
    func clearAllNotifications() {
        UNUserNotificationCenter.current().removeAllPendingNotificationRequests()
        UNUserNotificationCenter.current().removeAllDeliveredNotifications()
        pendingNotifications.removeAll()
        clearBadge()
    }
}
