//
//  LanciioManager.swift
//  PersonnelAssistant
//
//  Lanciio VIP MCP Hub Integration
//  - OAuth2 authentication flow
//  - Tool synchronization
//  - VIP workspace management
//

import Foundation
import AuthenticationServices
import Combine

// MARK: - Models

struct LanciioUser: Codable, Identifiable {
    var id: String { email }
    let email: String
    let name: String?
    let isVip: Bool
    
    enum CodingKeys: String, CodingKey {
        case email
        case name
        case isVip = "is_vip"
    }
}

struct LanciioWorkspace: Codable, Identifiable {
    let id: String
    let name: String
    let description: String?
    let toolsCount: Int
    let servers: [String]
    
    enum CodingKeys: String, CodingKey {
        case id
        case name
        case description
        case toolsCount = "tools_count"
        case servers
    }
}

struct LanciioTool: Codable, Identifiable {
    var id: String { fullIdentifier }
    let name: String
    let description: String
    let workspaceId: String
    let workspaceName: String
    let serverName: String
    let fullIdentifier: String
    
    enum CodingKeys: String, CodingKey {
        case name
        case description
        case workspaceId = "workspace_id"
        case workspaceName = "workspace_name"
        case serverName = "server_name"
        case fullIdentifier = "full_identifier"
    }
}

struct LanciioStatus: Codable {
    let connected: Bool
    let userEmail: String?
    let workspacesCount: Int
    let toolsCount: Int
    let isVip: Bool
    
    enum CodingKeys: String, CodingKey {
        case connected
        case userEmail = "user_email"
        case workspacesCount = "workspaces_count"
        case toolsCount = "tools_count"
        case isVip = "is_vip"
    }
}

struct LanciioAuthResponse: Codable {
    let authorizationUrl: String
    let state: String
    
    enum CodingKeys: String, CodingKey {
        case authorizationUrl = "authorization_url"
        case state
    }
}

struct LanciioSyncResponse: Codable {
    let success: Bool
    let toolsSynced: Int?
    let message: String?
    
    enum CodingKeys: String, CodingKey {
        case success
        case toolsSynced = "tools_synced"
        case message
    }
}

// MARK: - LanciioManager

@MainActor
class LanciioManager: NSObject, ObservableObject {
    
    // MARK: - Singleton
    static let shared = LanciioManager()
    
    // MARK: - Published Properties
    @Published var isConnected: Bool = false
    @Published var isConnecting: Bool = false
    @Published var currentUser: LanciioUser?
    @Published var workspaces: [LanciioWorkspace] = []
    @Published var tools: [LanciioTool] = []
    @Published var syncedToolsCount: Int = 0
    @Published var mcpServersCount: Int = 0
    @Published var lastSyncDate: Date?
    @Published var errorMessage: String?
    
    // MARK: - Private Properties
    private let baseURL: String
    private var currentState: String?
    private var authSession: ASWebAuthenticationSession?
    private var cancellables = Set<AnyCancellable>()
    
    // User ID from AuthManager
    private var userId: String {
        AuthManager.shared.currentUser?.id ?? "default_user"
    }
    
    // MARK: - Init
    
    private override init() {
        self.baseURL = "https://assistant.dmagh.me/api/v1/oauth/lanciio"
        super.init()
        
        // Check initial status
        Task {
            await checkConnectionStatus()
        }
    }
    
    // MARK: - Public Methods
    
    /// Initie le flux OAuth2 avec Lanciio
    func connect() async {
        guard !isConnecting else { return }
        
        isConnecting = true
        errorMessage = nil
        
        do {
            // 1. Get authorization URL from backend
            let authResponse = try await getAuthorizationURL()
            currentState = authResponse.state
            
            // 2. Open OAuth flow
            guard let url = URL(string: authResponse.authorizationUrl) else {
                throw LanciioError.invalidURL
            }
            
            await startOAuthFlow(url: url)
            
        } catch {
            errorMessage = error.localizedDescription
            isConnecting = false
        }
    }
    
    /// Déconnecte de Lanciio
    func disconnect() async {
        do {
            let url = URL(string: "\(baseURL)/auth/disconnect")!
            var request = URLRequest(url: url)
            request.httpMethod = "POST"
            request.setValue("application/json", forHTTPHeaderField: "Content-Type")
            request.httpBody = try JSONEncoder().encode(["user_id": userId])
            
            let (_, response) = try await URLSession.shared.data(for: request)
            
            if let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 {
                await MainActor.run {
                    self.isConnected = false
                    self.currentUser = nil
                    self.workspaces = []
                    self.tools = []
                    self.syncedToolsCount = 0
                    self.mcpServersCount = 0
                }
            }
        } catch {
            errorMessage = error.localizedDescription
        }
    }
    
    /// Force la synchronisation des outils
    func syncTools() async {
        do {
            let url = URL(string: "\(baseURL)/sync")!
            var request = URLRequest(url: url)
            request.httpMethod = "POST"
            request.setValue("application/json", forHTTPHeaderField: "Content-Type")
            request.httpBody = try JSONEncoder().encode(["user_id": userId])
            
            let (data, response) = try await URLSession.shared.data(for: request)
            
            if let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 {
                let syncResponse = try JSONDecoder().decode(LanciioSyncResponse.self, from: data)
                
                await MainActor.run {
                    self.syncedToolsCount = syncResponse.toolsSynced ?? 0
                    self.lastSyncDate = Date()
                }
                
                // Refresh tools list
                await fetchTools()
            }
        } catch {
            errorMessage = error.localizedDescription
        }
    }
    
    /// Vérifie le statut de connexion
    func checkConnectionStatus() async {
        do {
            let url = URL(string: "\(baseURL)/status/\(userId)")!
            let (data, response) = try await URLSession.shared.data(from: url)
            
            if let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 {
                let status = try JSONDecoder().decode(LanciioStatus.self, from: data)
                
                await MainActor.run {
                    self.isConnected = status.connected
                    self.syncedToolsCount = status.toolsCount
                    self.mcpServersCount = status.workspacesCount
                }
                
                if status.connected {
                    await fetchWorkspaces()
                    await fetchTools()
                }
            }
        } catch {
            // Silent fail - just means not connected
            isConnected = false
        }
    }
    
    /// Récupère les workspaces
    func fetchWorkspaces() async {
        do {
            let url = URL(string: "\(baseURL)/workspaces/\(userId)")!
            let (data, response) = try await URLSession.shared.data(from: url)
            
            if let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 {
                struct WorkspacesResponse: Codable {
                    let workspaces: [LanciioWorkspace]
                }
                
                let result = try JSONDecoder().decode(WorkspacesResponse.self, from: data)
                
                await MainActor.run {
                    self.workspaces = result.workspaces
                    self.mcpServersCount = result.workspaces.reduce(0) { $0 + $1.servers.count }
                }
            }
        } catch {
            // Silent fail
        }
    }
    
    /// Récupère les outils
    func fetchTools() async {
        do {
            let url = URL(string: "\(baseURL)/tools/\(userId)")!
            let (data, response) = try await URLSession.shared.data(from: url)
            
            if let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 {
                struct ToolsResponse: Codable {
                    let total: Int
                    let byWorkspace: [String: [LanciioTool]]
                    
                    enum CodingKeys: String, CodingKey {
                        case total
                        case byWorkspace = "by_workspace"
                    }
                }
                
                let result = try JSONDecoder().decode(ToolsResponse.self, from: data)
                
                await MainActor.run {
                    self.tools = result.byWorkspace.values.flatMap { $0 }
                    self.syncedToolsCount = result.total
                }
            }
        } catch {
            // Silent fail
        }
    }
    
    /// Recherche des outils
    func searchTools(query: String) async -> [LanciioTool] {
        guard !query.isEmpty else { return [] }
        
        do {
            let encodedQuery = query.addingPercentEncoding(withAllowedCharacters: .urlQueryAllowed) ?? query
            let url = URL(string: "\(baseURL)/tools/\(userId)/search?q=\(encodedQuery)")!
            let (data, response) = try await URLSession.shared.data(from: url)
            
            if let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 {
                struct SearchResponse: Codable {
                    let results: [SearchResult]
                }
                struct SearchResult: Codable {
                    let tool: LanciioTool
                    let score: Int
                }
                
                let result = try JSONDecoder().decode(SearchResponse.self, from: data)
                return result.results.map { $0.tool }
            }
        } catch {
            // Silent fail
        }
        
        return []
    }
    
    // MARK: - Private Methods
    
    private func getAuthorizationURL() async throws -> LanciioAuthResponse {
        let url = URL(string: "\(baseURL)/auth/authorize")!
        var request = URLRequest(url: url)
        request.httpMethod = "POST"
        request.setValue("application/json", forHTTPHeaderField: "Content-Type")
        
        let body = ["user_id": userId]
        request.httpBody = try JSONEncoder().encode(body)
        
        let (data, response) = try await URLSession.shared.data(for: request)
        
        guard let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 200 else {
            throw LanciioError.authorizationFailed
        }
        
        return try JSONDecoder().decode(LanciioAuthResponse.self, from: data)
    }
    
    private func startOAuthFlow(url: URL) async {
        await MainActor.run {
            let callbackURLScheme = "personnelassistant"
            
            authSession = ASWebAuthenticationSession(
                url: url,
                callbackURLScheme: callbackURLScheme
            ) { [weak self] callbackURL, error in
                guard let self = self else { return }
                
                Task { @MainActor in
                    self.isConnecting = false
                    
                    if let error = error {
                        if (error as NSError).code != ASWebAuthenticationSessionError.canceledLogin.rawValue {
                            self.errorMessage = error.localizedDescription
                        }
                        return
                    }
                    
                    // OAuth success - callback was handled by backend
                    // Just refresh status
                    await self.checkConnectionStatus()
                    
                    if self.isConnected {
                        await self.syncTools()
                    }
                }
            }
            
            authSession?.presentationContextProvider = self
            authSession?.prefersEphemeralWebBrowserSession = false
            authSession?.start()
        }
    }
}

// MARK: - ASWebAuthenticationPresentationContextProviding

extension LanciioManager: ASWebAuthenticationPresentationContextProviding {
    func presentationAnchor(for session: ASWebAuthenticationSession) -> ASPresentationAnchor {
        guard let windowScene = UIApplication.shared.connectedScenes.first as? UIWindowScene,
              let window = windowScene.windows.first else {
            return ASPresentationAnchor()
        }
        return window
    }
}

// MARK: - Errors

enum LanciioError: LocalizedError {
    case invalidURL
    case authorizationFailed
    case networkError(Error)
    case decodingError(Error)
    
    var errorDescription: String? {
        switch self {
        case .invalidURL:
            return "URL invalide"
        case .authorizationFailed:
            return "Échec de l'autorisation"
        case .networkError(let error):
            return "Erreur réseau: \(error.localizedDescription)"
        case .decodingError(let error):
            return "Erreur de décodage: \(error.localizedDescription)"
        }
    }
}
