//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Enhanced Chat Manager                                        ║
//  ║         Communication with Collaborative Multi-Agent Backend                   ║
//  ║         Supports Rich Responses with Attachments & Panels                      ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import Combine
import UserNotifications

// MARK: - Chat Message Model

struct ChatMessage: Identifiable, Codable, Equatable {
    let id: String
    let content: String
    let role: MessageRole
    let contentType: ContentType
    let provider: AIProvider?
    let toolsUsed: [String]?
    let agentsUsed: [String]?
    let processingTimeMs: Int?
    let createdAt: Date
    let isVoice: Bool
    
    // Rich content support
    let attachments: [MessageAttachment]?
    let showPanel: Bool
    let panelType: String?
    let suggestedActions: [SuggestedAction]?
    let complexity: String?
    let qualityScore: Double?
    
    init(
        id: String = UUID().uuidString,
        content: String,
        role: MessageRole,
        contentType: ContentType = .text,
        provider: AIProvider? = nil,
        toolsUsed: [String]? = nil,
        agentsUsed: [String]? = nil,
        processingTimeMs: Int? = nil,
        createdAt: Date = Date(),
        isVoice: Bool = false,
        attachments: [MessageAttachment]? = nil,
        showPanel: Bool = false,
        panelType: String? = nil,
        suggestedActions: [SuggestedAction]? = nil,
        complexity: String? = nil,
        qualityScore: Double? = nil
    ) {
        self.id = id
        self.content = content
        self.role = role
        self.contentType = contentType
        self.provider = provider
        self.toolsUsed = toolsUsed
        self.agentsUsed = agentsUsed
        self.processingTimeMs = processingTimeMs
        self.createdAt = createdAt
        self.isVoice = isVoice
        self.attachments = attachments
        self.showPanel = showPanel
        self.panelType = panelType
        self.suggestedActions = suggestedActions
        self.complexity = complexity
        self.qualityScore = qualityScore
    }
    
    static func == (lhs: ChatMessage, rhs: ChatMessage) -> Bool {
        lhs.id == rhs.id
    }
}

enum MessageRole: String, Codable {
    case user
    case assistant
    case system
    case tool
}

enum ContentType: String, Codable {
    case text
    case markdown
    case json
    case code
    case table
    case graph
    case dashboard
    case timeline
    case mixed
    
    init(from decoder: Decoder) throws {
        let container = try decoder.singleValueContainer()
        let rawValue = try container.decode(String.self)
        self = ContentType(rawValue: rawValue) ?? .text
    }
}

// MARK: - Message Attachment

struct MessageAttachment: Identifiable, Codable, Equatable {
    let id: String
    let type: String
    let title: String
    let data: AttachmentData
    let mimeType: String?
    let interactive: Bool
    
    static func == (lhs: MessageAttachment, rhs: MessageAttachment) -> Bool {
        lhs.id == rhs.id
    }
}

struct AttachmentData: Codable, Equatable {
    let rawValue: AnyCodableValue
    
    init(_ value: Any) {
        self.rawValue = AnyCodableValue(value)
    }
    
    init(from decoder: Decoder) throws {
        let container = try decoder.singleValueContainer()
        self.rawValue = try container.decode(AnyCodableValue.self)
    }
    
    func encode(to encoder: Encoder) throws {
        var container = encoder.singleValueContainer()
        try container.encode(rawValue)
    }
    
    static func == (lhs: AttachmentData, rhs: AttachmentData) -> Bool {
        // Simple equality check
        return true
    }
    
    var dictionary: [String: Any]? {
        return rawValue.value as? [String: Any]
    }
}

struct AnyCodableValue: Codable, Equatable {
    let value: Any
    
    init(_ value: Any) {
        self.value = value
    }
    
    init(from decoder: Decoder) throws {
        let container = try decoder.singleValueContainer()
        
        if let dict = try? container.decode([String: AnyCodableValue].self) {
            value = dict.mapValues { $0.value }
        } else if let array = try? container.decode([AnyCodableValue].self) {
            value = array.map { $0.value }
        } else if let string = try? container.decode(String.self) {
            value = string
        } else if let int = try? container.decode(Int.self) {
            value = int
        } else if let double = try? container.decode(Double.self) {
            value = double
        } else if let bool = try? container.decode(Bool.self) {
            value = bool
        } else {
            value = NSNull()
        }
    }
    
    func encode(to encoder: Encoder) throws {
        var container = encoder.singleValueContainer()
        if let dict = value as? [String: Any] {
            try container.encode(dict.mapValues { AnyCodableValue($0) })
        } else if let array = value as? [Any] {
            try container.encode(array.map { AnyCodableValue($0) })
        } else if let string = value as? String {
            try container.encode(string)
        } else if let int = value as? Int {
            try container.encode(int)
        } else if let double = value as? Double {
            try container.encode(double)
        } else if let bool = value as? Bool {
            try container.encode(bool)
        }
    }
    
    static func == (lhs: AnyCodableValue, rhs: AnyCodableValue) -> Bool {
        return true
    }
}

// MARK: - Suggested Action

struct SuggestedAction: Identifiable, Codable, Equatable {
    var id: String { action }
    let action: String
    let command: String
}

// MARK: - API Response Model

struct ChatAPIResponse: Codable {
    let content: String
    let contentType: String?
    let provider: String?
    let model: String?
    let toolsUsed: [String]?
    let agentsUsed: [String]?
    let processingTimeMs: Int?
    let intent: String?
    let complexity: String?
    let qualityScore: Double?
    let showPanel: Bool?
    let panelType: String?
    let attachments: [APIAttachment]?
    let suggestedActions: [SuggestedAction]?
    let relatedInfo: [RelatedInfo]?
    let sources: [String]?
    let error: String?
    
    enum CodingKeys: String, CodingKey {
        case content
        case contentType = "content_type"
        case provider
        case model
        case toolsUsed = "tools_used"
        case agentsUsed = "agents_used"
        case processingTimeMs = "processing_time_ms"
        case intent
        case complexity
        case qualityScore = "quality_score"
        case showPanel = "show_panel"
        case panelType = "panel_type"
        case attachments
        case suggestedActions = "suggested_actions"
        case relatedInfo = "related_info"
        case sources
        case error
    }
}

struct APIAttachment: Codable {
    let id: String
    let type: String
    let title: String
    let data: AnyCodableValue
    let mimeType: String?
    let interactive: Bool
    
    enum CodingKeys: String, CodingKey {
        case id, type, title, data
        case mimeType = "mime_type"
        case interactive
    }
}

struct RelatedInfo: Codable {
    let title: String
    let description: String
}

// MARK: - Chat Manager

@MainActor
class ChatManager: ObservableObject {
    static let shared = ChatManager()
    
    // State
    @Published var messages: [ChatMessage] = []
    @Published var isLoading = false
    @Published var isTyping = false
    @Published var streamingContent = ""
    @Published var selectedProvider: AIProvider = .auto
    @Published var error: String?
    
    // Panel state
    @Published var showContentPanel = false
    @Published var currentPanelAttachments: [PanelAttachment] = []
    @Published var lastSuggestedActions: [SuggestedAction] = []
    
    // Attachments pending to be sent
    @Published var pendingAttachments: [Attachment] = []
    
    // Configuration
    private let baseURL = "https://assistant.dmagh.me/api/v1"
    private var currentConversationId: String = UUID().uuidString
    
    private init() {}
    
    // MARK: - Send Message
    
    func sendMessage(_ content: String, isVoice: Bool = false) async {
        guard !content.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty else { return }
        
        // Add user message
        let userMessage = ChatMessage(
            content: content,
            role: .user,
            contentType: .text,
            isVoice: isVoice
        )
        messages.append(userMessage)
        
        // Start loading
        isLoading = true
        isTyping = true
        error = nil
        streamingContent = ""
        
        // Simulate thinking stream (visual feedback) while waiting for backend
        Task { @MainActor in
            await simulateThinkingSteps(for: content)
        }
        
        do {
            let response = try await sendToBackend(content)
            
            // Stop thinking simulation
            await MainActor.run {
                WebSocketManager.shared.isThinking = false
                WebSocketManager.shared.thinkingSteps.removeAll()
            }
            
            // Parse attachments
            let attachments: [MessageAttachment]? = response.attachments?.map { apiAtt in
                MessageAttachment(
                    id: apiAtt.id,
                    type: apiAtt.type,
                    title: apiAtt.title,
                    data: AttachmentData(apiAtt.data.value),
                    mimeType: apiAtt.mimeType,
                    interactive: apiAtt.interactive
                )
            }
            
            // Create assistant message
            let assistantMessage = ChatMessage(
                content: response.content,
                role: .assistant,
                contentType: ContentType(rawValue: response.contentType ?? "text") ?? .text,
                provider: AIProvider(rawValue: response.provider ?? "auto"),
                toolsUsed: response.toolsUsed,
                agentsUsed: response.agentsUsed,
                processingTimeMs: response.processingTimeMs,
                attachments: attachments,
                showPanel: response.showPanel ?? false,
                panelType: response.panelType,
                suggestedActions: response.suggestedActions,
                complexity: response.complexity,
                qualityScore: response.qualityScore
            )
            
            messages.append(assistantMessage)
            
            // Handle panel display
            if response.showPanel == true, let atts = response.attachments, !atts.isEmpty {
                showPanel(with: atts)
            }
            
            // Store suggested actions
            if let actions = response.suggestedActions {
                lastSuggestedActions = actions
            }
            
            // If voice, speak the response
            if isVoice {
                VoiceManager.shared.speak(response.content)
            }
            
            // Send notification when task is complete (if app is in background)
            sendTaskCompletionNotification(
                title: "AYMENOS - Réponse prête",
                message: extractNotificationPreview(from: response.content),
                toolsUsed: response.toolsUsed
            )
            
            // End collaboration session
            WebSocketManager.shared.endCollaborationSession(withOutput: response.content)
            
        } catch {
            self.error = error.localizedDescription
            
            let errorMessage = ChatMessage(
                content: "❌ Erreur: \(error.localizedDescription)",
                role: .assistant,
                contentType: .text
            )
            messages.append(errorMessage)
        }
        
        isLoading = false
        isTyping = false
    }
    
    // MARK: - Sync Send (for Siri)
    
    func sendMessageSync(_ content: String) async -> String {
        await sendMessage(content, isVoice: false)
        return messages.last?.content ?? "Aucune réponse"
    }
    
    // MARK: - Thinking Simulation
    
    @MainActor
    private func simulateThinkingSteps(for content: String) async {
        let wsManager = WebSocketManager.shared
        
        // Only simulate if not already connected via WebSocket
        guard wsManager.connectionState != .connected else { return }
        
        wsManager.isThinking = true
        wsManager.thinkingProgress = 0
        wsManager.thinkingSteps.removeAll()
        
        // Determine which agents might be involved
        let isInfra = content.lowercased().contains("docker") || content.lowercased().contains("service") || content.lowercased().contains("infra")
        let isTicket = content.lowercased().contains("ticket") || content.lowercased().contains("itop") || content.lowercased().contains("incident")
        let isOdoo = content.lowercased().contains("client") || content.lowercased().contains("odoo") || content.lowercased().contains("commande")
        
        // Step 1: Analyzing
        let step1 = ThinkingStep(
            id: UUID().uuidString,
            type: .analyzing,
            title: "🔍 Analyse",
            description: "Analyse de votre demande...",
            agent: nil,
            progress: 0.15,
            timestamp: Date(),
            status: .inProgress
        )
        wsManager.thinkingSteps.append(step1)
        wsManager.thinkingProgress = 0.15
        
        try? await Task.sleep(nanoseconds: 500_000_000) // 0.5s
        wsManager.thinkingSteps[0].status = .completed
        
        // Step 2: Routing
        let step2 = ThinkingStep(
            id: UUID().uuidString,
            type: .routing,
            title: "🔀 Routage",
            description: "Sélection de l'agent spécialisé",
            agent: nil,
            progress: 0.35,
            timestamp: Date(),
            status: .inProgress
        )
        wsManager.thinkingSteps.append(step2)
        wsManager.thinkingProgress = 0.35
        
        try? await Task.sleep(nanoseconds: 400_000_000) // 0.4s
        wsManager.thinkingSteps[1].status = .completed
        
        // Step 3: Agent Execution
        let agentName: String
        let agentDesc: String
        if isInfra {
            agentName = "🐳 Docker Agent"
            agentDesc = "Vérification de l'infrastructure..."
        } else if isTicket {
            agentName = "🎫 iTop Agent"
            agentDesc = "Interrogation du système ITSM..."
        } else if isOdoo {
            agentName = "💼 Odoo Agent"
            agentDesc = "Consultation de l'ERP..."
        } else {
            agentName = "🧠 Supervisor"
            agentDesc = "Traitement de la demande..."
        }
        
        let step3 = ThinkingStep(
            id: UUID().uuidString,
            type: .executing,
            title: agentName,
            description: agentDesc,
            agent: agentName,
            progress: 0.65,
            timestamp: Date(),
            status: .inProgress
        )
        wsManager.thinkingSteps.append(step3)
        wsManager.thinkingProgress = 0.65
        
        try? await Task.sleep(nanoseconds: 600_000_000) // 0.6s
        wsManager.thinkingSteps[2].status = .completed
        
        // Step 4: Reformulating
        let step4 = ThinkingStep(
            id: UUID().uuidString,
            type: .reformulating,
            title: "✨ Reformulation",
            description: "Amélioration de la réponse...",
            agent: nil,
            progress: 0.9,
            timestamp: Date(),
            status: .inProgress
        )
        wsManager.thinkingSteps.append(step4)
        wsManager.thinkingProgress = 0.9
        
        // Note: We don't complete step 4 here as sendToBackend will handle completion
    }
    
    // MARK: - Backend Communication
    
    private func sendToBackend(_ content: String) async throws -> ChatAPIResponse {
        guard let url = URL(string: "\(baseURL)/chat") else {
            throw ChatError.invalidURL
        }
        
        var request = URLRequest(url: url)
        request.httpMethod = "POST"
        request.setValue("application/json", forHTTPHeaderField: "Content-Type")
        request.timeoutInterval = 120 // 2 minutes for complex requests
        
        // Add auth token if available
        if let token = AuthManager.shared.accessToken {
            request.setValue("Bearer \(token)", forHTTPHeaderField: "Authorization")
        }
        
        let body: [String: Any] = [
            "content": content,
            "provider": selectedProvider.rawValue,
            "conversation_id": currentConversationId
        ]
        
        request.httpBody = try JSONSerialization.data(withJSONObject: body)
        
        let (data, response) = try await URLSession.shared.data(for: request)
        
        guard let httpResponse = response as? HTTPURLResponse else {
            throw ChatError.invalidResponse
        }
        
        guard (200...299).contains(httpResponse.statusCode) else {
            throw ChatError.serverError(statusCode: httpResponse.statusCode)
        }
        
        let decoder = JSONDecoder()
        return try decoder.decode(ChatAPIResponse.self, from: data)
    }
    
    // MARK: - Panel Management
    
    func showPanel(with attachments: [APIAttachment]) {
        currentPanelAttachments = attachments.map { apiAtt in
            PanelAttachment(
                id: apiAtt.id,
                type: PanelContentType(rawValue: apiAtt.type) ?? .text,
                title: apiAtt.title,
                data: apiAtt.data.value,
                mimeType: apiAtt.mimeType,
                interactive: apiAtt.interactive
            )
        }
        
        withAnimation(.spring()) {
            showContentPanel = true
        }
    }
    
    func hidePanel() {
        withAnimation(.spring()) {
            showContentPanel = false
        }
    }
    
    // MARK: - Execute Suggested Action
    
    func executeSuggestedAction(_ action: SuggestedAction) async {
        await sendMessage(action.command, isVoice: false)
    }
    
    // MARK: - Conversation Management
    
    func clearConversation() {
        messages.removeAll()
        streamingContent = ""
        currentConversationId = UUID().uuidString
        showContentPanel = false
        currentPanelAttachments.removeAll()
        lastSuggestedActions.removeAll()
    }
    
    func newConversation() {
        clearConversation()
    }
    
    // MARK: - Notifications
    
    /// Send a local notification when assistant completes a task
    private func sendTaskCompletionNotification(title: String, message: String, toolsUsed: [String]?) {
        print("[ChatManager] sendTaskCompletionNotification called")
        
        // Dispatch to notification manager on main thread
        DispatchQueue.main.async {
            NotificationManager.shared.sendAssistantResponseNotification(
                title: title,
                message: message,
                toolsUsed: toolsUsed
            )
        }
    }
    
    /// Extract a preview from the response content for notification
    private func extractNotificationPreview(from content: String) -> String {
        // Remove markdown formatting
        var preview = content
            .replacingOccurrences(of: "**", with: "")
            .replacingOccurrences(of: "__", with: "")
            .replacingOccurrences(of: "##", with: "")
            .replacingOccurrences(of: "#", with: "")
            .replacingOccurrences(of: "```", with: "")
            .replacingOccurrences(of: "`", with: "")
            .replacingOccurrences(of: "\n", with: " ")
        
        // Trim and limit length
        preview = preview.trimmingCharacters(in: .whitespacesAndNewlines)
        
        if preview.count > 100 {
            let index = preview.index(preview.startIndex, offsetBy: 97)
            preview = String(preview[..<index]) + "..."
        }
        
        return preview.isEmpty ? "Réponse de l'assistant prête" : preview
    }
}

// MARK: - Chat Errors

enum ChatError: LocalizedError {
    case invalidURL
    case invalidResponse
    case serverError(statusCode: Int)
    case decodingError
    
    var errorDescription: String? {
        switch self {
        case .invalidURL:
            return "URL invalide"
        case .invalidResponse:
            return "Réponse invalide du serveur"
        case .serverError(let code):
            return "Erreur serveur (\(code))"
        case .decodingError:
            return "Erreur de décodage"
        }
    }
}

// AIProvider is defined in Models.swift with displayName and icon properties
