//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Background Task Manager                                     ║
//  ║         Handles offline operations, background execution, notifications       ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import Foundation
import BackgroundTasks
import UserNotifications
import SwiftUI

// MARK: - Background Task Manager

class BackgroundTaskManager: ObservableObject {
    static let shared = BackgroundTaskManager()
    
    // Task identifiers
    static let queueProcessingTaskId = "com.aymenos.processQueue"
    static let syncTaskId = "com.aymenos.syncData"
    
    // Published state
    @Published var pendingActions: [QueuedOdooAction] = []
    @Published var actionHistory: [QueuedOdooAction] = []
    @Published var isProcessingQueue = false
    @Published var hasOfflineActions = false
    
    // Network monitoring
    @Published var isOnline = true
    
    private let defaults = UserDefaults.standard
    private let queueKey = "aymenos_action_queue"
    
    init() {
        loadQueueFromStorage()
        setupNetworkMonitoring()
        requestNotificationPermission()
    }
    
    // MARK: - Setup
    
    /// Register background tasks with the system
    static func registerBackgroundTasks() {
        // Register queue processing task
        BGTaskScheduler.shared.register(
            forTaskWithIdentifier: queueProcessingTaskId,
            using: nil
        ) { task in
            BackgroundTaskManager.shared.handleQueueProcessing(task: task as! BGProcessingTask)
        }
        
        // Register sync task
        BGTaskScheduler.shared.register(
            forTaskWithIdentifier: syncTaskId,
            using: nil
        ) { task in
            BackgroundTaskManager.shared.handleSync(task: task as! BGAppRefreshTask)
        }
    }
    
    /// Request notification permission
    private func requestNotificationPermission() {
        UNUserNotificationCenter.current().requestAuthorization(options: [.alert, .sound, .badge]) { granted, error in
            if granted {
                print("[BackgroundTaskManager] Notifications authorized")
            }
        }
    }
    
    /// Setup network monitoring
    private func setupNetworkMonitoring() {
        // In production, use NWPathMonitor
        // For now, assume online
        isOnline = true
    }
    
    // MARK: - Queue Management
    
    /// Add action to queue
    func queueAction(_ action: QueuedOdooAction) {
        DispatchQueue.main.async {
            self.pendingActions.append(action)
            self.hasOfflineActions = true
            self.saveQueueToStorage()
        }
        
        // Send notification if action requires confirmation
        if action.requiresConfirmation {
            sendActionConfirmationNotification(action)
        }
        
        // Schedule background processing
        scheduleQueueProcessing()
    }
    
    /// Confirm an action
    func confirmAction(_ actionId: String) async -> Bool {
        guard let index = pendingActions.firstIndex(where: { $0.id == actionId }) else {
            return false
        }
        
        var action = pendingActions[index]
        action.status = "confirmed"
        action.confirmedAt = Date()
        
        DispatchQueue.main.async {
            self.pendingActions[index] = action
            self.saveQueueToStorage()
        }
        
        // If online, execute immediately
        if isOnline {
            return await executeAction(action)
        }
        
        return true
    }
    
    /// Cancel an action
    func cancelAction(_ actionId: String) {
        DispatchQueue.main.async {
            self.pendingActions.removeAll { $0.id == actionId }
            self.hasOfflineActions = !self.pendingActions.isEmpty
            self.saveQueueToStorage()
        }
    }
    
    /// Execute a single action
    func executeAction(_ action: QueuedOdooAction) async -> Bool {
        guard action.status == "confirmed" else { return false }
        
        do {
            let result = try await executeQueuedActionAPI(actionId: action.id)
            
            var completedAction = action
            completedAction.status = result.success ? "completed" : "failed"
            completedAction.executedAt = Date()
            completedAction.error = result.error
            
            DispatchQueue.main.async {
                self.pendingActions.removeAll { $0.id == action.id }
                self.actionHistory.insert(completedAction, at: 0)
                self.hasOfflineActions = !self.pendingActions.isEmpty
                self.saveQueueToStorage()
            }
            
            // Send completion notification
            sendActionCompletionNotification(completedAction)
            
            return result.success
        } catch {
            print("[BackgroundTaskManager] Execute failed: \(error)")
            return false
        }
    }
    
    /// Process all confirmed actions in queue
    func processQueue() async {
        guard !isProcessingQueue else { return }
        
        DispatchQueue.main.async {
            self.isProcessingQueue = true
        }
        
        let confirmedActions = pendingActions.filter { $0.status == "confirmed" }
        
        for action in confirmedActions {
            _ = await executeAction(action)
        }
        
        DispatchQueue.main.async {
            self.isProcessingQueue = false
        }
    }
    
    // MARK: - Background Task Handlers
    
    private func handleQueueProcessing(task: BGProcessingTask) {
        // Schedule next processing
        scheduleQueueProcessing()
        
        // Create task for queue processing
        let processingTask = Task {
            await processQueue()
        }
        
        task.expirationHandler = {
            processingTask.cancel()
        }
        
        Task {
            await processingTask.value
            task.setTaskCompleted(success: true)
        }
    }
    
    private func handleSync(task: BGAppRefreshTask) {
        // Schedule next sync
        scheduleSync()
        
        Task {
            // Refresh pending actions from server
            await refreshPendingActions()
            task.setTaskCompleted(success: true)
        }
    }
    
    /// Schedule queue processing for background
    func scheduleQueueProcessing() {
        let request = BGProcessingTaskRequest(identifier: Self.queueProcessingTaskId)
        request.requiresNetworkConnectivity = true
        request.requiresExternalPower = false
        request.earliestBeginDate = Date(timeIntervalSinceNow: 60) // 1 minute
        
        do {
            try BGTaskScheduler.shared.submit(request)
        } catch {
            print("[BackgroundTaskManager] Could not schedule processing: \(error)")
        }
    }
    
    /// Schedule sync task
    func scheduleSync() {
        let request = BGAppRefreshTaskRequest(identifier: Self.syncTaskId)
        request.earliestBeginDate = Date(timeIntervalSinceNow: 15 * 60) // 15 minutes
        
        do {
            try BGTaskScheduler.shared.submit(request)
        } catch {
            print("[BackgroundTaskManager] Could not schedule sync: \(error)")
        }
    }
    
    // MARK: - Notifications
    
    /// Send notification for action requiring confirmation
    private func sendActionConfirmationNotification(_ action: QueuedOdooAction) {
        let content = UNMutableNotificationContent()
        content.title = "⚡ Action en attente"
        content.body = "L'action \(action.method) sur \(action.model) nécessite votre confirmation."
        content.sound = .default
        content.categoryIdentifier = "ACTION_CONFIRMATION"
        content.userInfo = ["action_id": action.id]
        
        // Add actions
        let confirmAction = UNNotificationAction(
            identifier: "CONFIRM_ACTION",
            title: "✓ Confirmer",
            options: [.foreground]
        )
        let cancelAction = UNNotificationAction(
            identifier: "CANCEL_ACTION",
            title: "✗ Annuler",
            options: [.destructive]
        )
        let category = UNNotificationCategory(
            identifier: "ACTION_CONFIRMATION",
            actions: [confirmAction, cancelAction],
            intentIdentifiers: [],
            options: []
        )
        UNUserNotificationCenter.current().setNotificationCategories([category])
        
        let request = UNNotificationRequest(
            identifier: "action_\(action.id)",
            content: content,
            trigger: nil // Immediate
        )
        
        UNUserNotificationCenter.current().add(request)
    }
    
    /// Send notification for action completion
    private func sendActionCompletionNotification(_ action: QueuedOdooAction) {
        let content = UNMutableNotificationContent()
        
        if action.status == "completed" {
            content.title = "✅ Action terminée"
            content.body = "\(action.method) sur \(action.model) a été exécuté avec succès."
            content.sound = .default
        } else {
            content.title = "❌ Action échouée"
            content.body = "\(action.method) sur \(action.model) a échoué: \(action.error ?? "Erreur inconnue")"
            content.sound = UNNotificationSound.defaultCritical
        }
        
        let request = UNNotificationRequest(
            identifier: "completion_\(action.id)",
            content: content,
            trigger: nil
        )
        
        UNUserNotificationCenter.current().add(request)
    }
    
    // MARK: - Persistence
    
    private func saveQueueToStorage() {
        if let encoded = try? JSONEncoder().encode(pendingActions) {
            defaults.set(encoded, forKey: queueKey)
        }
        if let encodedHistory = try? JSONEncoder().encode(Array(actionHistory.prefix(100))) {
            defaults.set(encodedHistory, forKey: "\(queueKey)_history")
        }
    }
    
    private func loadQueueFromStorage() {
        if let data = defaults.data(forKey: queueKey),
           let actions = try? JSONDecoder().decode([QueuedOdooAction].self, from: data) {
            pendingActions = actions
            hasOfflineActions = !actions.isEmpty
        }
        if let historyData = defaults.data(forKey: "\(queueKey)_history"),
           let history = try? JSONDecoder().decode([QueuedOdooAction].self, from: historyData) {
            actionHistory = history
        }
    }
    
    // MARK: - API
    
    private func refreshPendingActions() async {
        // Sync with backend - future implementation
    }
    
    private func executeQueuedActionAPI(actionId: String) async throws -> (success: Bool, error: String?) {
        let baseURL = "https://assistant.dmagh.me"
        let endpoint = "/api/v1/odoo/queue/execute/\(actionId)"
        
        guard let url = URL(string: "\(baseURL)\(endpoint)") else {
            return (false, "Invalid URL")
        }
        
        var request = URLRequest(url: url)
        request.httpMethod = "POST"
        request.setValue("application/json", forHTTPHeaderField: "Content-Type")
        
        let (data, response) = try await URLSession.shared.data(for: request)
        
        guard let httpResponse = response as? HTTPURLResponse,
              httpResponse.statusCode == 200 else {
            return (false, "Request failed")
        }
        
        struct ExecuteResponse: Decodable {
            let success: Bool
            let error: String?
        }
        
        let result = try JSONDecoder().decode(ExecuteResponse.self, from: data)
        return (result.success, result.error)
    }
}

// MARK: - Queued Odoo Action Model

struct QueuedOdooAction: Identifiable, Codable {
    let id: String
    var userId: String
    var model: String
    var method: String
    var status: String // pending, confirmed, executing, completed, failed
    var createdAt: Date
    var confirmedAt: Date?
    var executedAt: Date?
    var error: String?
    var requiresConfirmation: Bool
    var sensitivityLevel: String
    
    var isPending: Bool { status == "pending" }
    var isConfirmed: Bool { status == "confirmed" }
    var isCompleted: Bool { status == "completed" }
    var isFailed: Bool { status == "failed" }
    
    var sensitivityColor: Color {
        switch sensitivityLevel {
        case "critical": return .red
        case "high": return .orange
        case "moderate": return .yellow
        default: return .green
        }
    }
    
    init(id: String = UUID().uuidString,
         userId: String,
         model: String,
         method: String,
         status: String = "pending",
         requiresConfirmation: Bool = false,
         sensitivityLevel: String = "safe") {
        self.id = id
        self.userId = userId
        self.model = model
        self.method = method
        self.status = status
        self.createdAt = Date()
        self.requiresConfirmation = requiresConfirmation
        self.sensitivityLevel = sensitivityLevel
    }
}

// MARK: - Odoo Action Classification

struct OdooActionClassification: Codable {
    let level: String
    let model: String
    let method: String
    let requiresConfirmation: Bool
    let warningMessage: String?
    let isDestructive: Bool
    let isWrite: Bool
}
