//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║           🤖 AYMENOS - Personal AI Assistant                                   ║
//  ║           Dark Blood Theme with Sophisticated Animations                       ║
//  ║           Widget, Siri, Apple Watch & Background Tasks Support                 ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI
import WatchConnectivity
import BackgroundTasks
import UserNotifications

@main
struct PersonnelAssistantApp: App {
    @UIApplicationDelegateAdaptor(AppDelegate.self) var appDelegate
    
    @StateObject private var authManager = AuthManager.shared
    @StateObject private var themeManager = ThemeManager.shared
    @StateObject private var chatManager = ChatManager.shared
    @StateObject private var appState = AppState.shared
    @StateObject private var watchConnector = WatchConnector.shared
    @StateObject private var backgroundTaskManager = BackgroundTaskManager.shared
    
    var body: some Scene {
        WindowGroup {
            RootView()
                .environmentObject(authManager)
                .environmentObject(themeManager)
                .environmentObject(chatManager)
                .environmentObject(appState)
                .environmentObject(backgroundTaskManager)
                .preferredColorScheme(.dark)
                .onOpenURL { url in
                    handleDeepLink(url)
                }
                .onReceive(NotificationCenter.default.publisher(for: Notification.Name("StartVoiceRecording"))) { _ in
                    appState.triggerVoiceInput = true
                }
                .onReceive(NotificationCenter.default.publisher(for: Notification.Name("SendChatMessage"))) { notification in
                    if let message = notification.userInfo?["message"] as? String {
                        Task {
                            await chatManager.sendMessage(message, isVoice: false)
                        }
                    }
                }
        }
    }
    
    private func handleDeepLink(_ url: URL) {
        guard let host = url.host else { return }
        
        switch host {
        case "voice":
            appState.selectedTab = 0
            appState.triggerVoiceInput = true
            
        case "chat":
            appState.selectedTab = 0
            
        case "infrastructure":
            appState.selectedTab = 1
            
        case "tools":
            appState.selectedTab = 2
            
        case "settings":
            appState.selectedTab = 3
            
        case "home":
            appState.selectedTab = 0
            
        default:
            break
        }
    }
}

// MARK: - App State

class AppState: ObservableObject {
    static let shared = AppState()
    
    @Published var selectedTab: Int = 0
    @Published var triggerVoiceInput: Bool = false
    @Published var pendingMessage: String?
}

// MARK: - Watch Connector

class WatchConnector: NSObject, ObservableObject, WCSessionDelegate {
    static let shared = WatchConnector()
    
    @Published var isWatchReachable = false
    
    private var session: WCSession?
    
    override init() {
        super.init()
        if WCSession.isSupported() {
            session = WCSession.default
            session?.delegate = self
            session?.activate()
        }
    }
    
    // Send message to Watch
    func sendToWatch(_ message: [String: Any]) {
        guard let session = session, session.isReachable else { return }
        session.sendMessage(message, replyHandler: nil, errorHandler: nil)
    }
    
    // Send infrastructure status to Watch
    func sendInfraStatus() {
        let status: [String: Bool] = [
            "docker": true,
            "vpn": true,
            "sso": true,
            "api": true
        ]
        sendToWatch(["infraStatus": status])
    }
    
    // WCSessionDelegate
    func session(_ session: WCSession, activationDidCompleteWith activationState: WCSessionActivationState, error: Error?) {
        DispatchQueue.main.async {
            self.isWatchReachable = session.isReachable
        }
    }
    
    func sessionDidBecomeInactive(_ session: WCSession) {}
    
    func sessionDidDeactivate(_ session: WCSession) {
        session.activate()
    }
    
    func sessionReachabilityDidChange(_ session: WCSession) {
        DispatchQueue.main.async {
            self.isWatchReachable = session.isReachable
        }
    }
    
    func session(_ session: WCSession, didReceiveMessage message: [String : Any], replyHandler: @escaping ([String : Any]) -> Void) {
        DispatchQueue.main.async {
            if let action = message["action"] as? String {
                switch action {
                case "chat":
                    if let text = message["message"] as? String {
                        Task {
                            await ChatManager.shared.sendMessage(text, isVoice: false)
                            replyHandler(["reply": "Message envoyé"])
                        }
                    }
                    
                case "startVoice":
                    AppState.shared.triggerVoiceInput = true
                    replyHandler(["status": "ok"])
                    
                case "infraStatus":
                    replyHandler([
                        "status": [
                            "docker": true,
                            "vpn": true,
                            "sso": true,
                            "api": true
                        ]
                    ])
                    
                default:
                    replyHandler(["error": "Unknown action"])
                }
            }
        }
    }
}

// MARK: - Root View

struct RootView: View {
    @EnvironmentObject var authManager: AuthManager
    @EnvironmentObject var themeManager: ThemeManager
    @EnvironmentObject var appState: AppState
    
    var body: some View {
        ZStack {
            // Background gradient
            themeManager.backgroundGradient
                .ignoresSafeArea()
            
            // Content
            if authManager.isAuthenticated {
                MainTabView()
                    .transition(.asymmetric(
                        insertion: .opacity.combined(with: .scale(scale: 0.95)),
                        removal: .opacity
                    ))
            } else {
                LoginView()
                    .transition(.opacity)
            }
        }
        .animation(.easeInOut(duration: 0.5), value: authManager.isAuthenticated)
    }
}

// MARK: - Main Tab View

struct MainTabView: View {
    @EnvironmentObject var themeManager: ThemeManager
    @EnvironmentObject var appState: AppState
    
    var body: some View {
        TabView(selection: $appState.selectedTab) {
            ChatView()
                .tabItem {
                    Label("Chat", systemImage: "bubble.left.and.bubble.right.fill")
                }
                .tag(0)
            
            InfrastructureView()
                .tabItem {
                    Label("Infra", systemImage: "server.rack")
                }
                .tag(1)
            
            ToolsView()
                .tabItem {
                    Label("Tools", systemImage: "wrench.and.screwdriver.fill")
                }
                .tag(2)
            
            SettingsView()
                .tabItem {
                    Label("Settings", systemImage: "gearshape.fill")
                }
                .tag(3)
        }
        .tint(themeManager.accentColor)
        .onChange(of: appState.triggerVoiceInput) { _, newValue in
            if newValue {
                // The ChatView will handle this
                DispatchQueue.main.asyncAfter(deadline: .now() + 0.1) {
                    appState.triggerVoiceInput = false
                }
            }
        }
    }
}

// MARK: - App Delegate for Background Tasks & Notifications

class AppDelegate: NSObject, UIApplicationDelegate, UNUserNotificationCenterDelegate {
    
    func application(
        _ application: UIApplication,
        didFinishLaunchingWithOptions launchOptions: [UIApplication.LaunchOptionsKey: Any]? = nil
    ) -> Bool {
        // Register background tasks
        BackgroundTaskManager.registerBackgroundTasks()
        
        // Setup notification delegate
        UNUserNotificationCenter.current().delegate = self
        
        // Request notification authorization
        requestNotificationPermission()
        
        print("[AppDelegate] AYMENOS initialized with background task support")
        
        return true
    }
    
    private func requestNotificationPermission() {
        UNUserNotificationCenter.current().requestAuthorization(
            options: [.alert, .sound, .badge, .providesAppNotificationSettings]
        ) { granted, error in
            if granted {
                print("[AppDelegate] Notification permission granted")
                DispatchQueue.main.async {
                    UIApplication.shared.registerForRemoteNotifications()
                }
                // Register notification categories
                self.registerNotificationCategories()
            }
        }
    }
    
    private func registerNotificationCategories() {
        // Action Confirmation Category
        let confirmAction = UNNotificationAction(
            identifier: "CONFIRM_ACTION",
            title: "✓ Confirmer",
            options: [.foreground]
        )
        let cancelAction = UNNotificationAction(
            identifier: "CANCEL_ACTION",
            title: "✗ Annuler",
            options: [.destructive]
        )
        let actionCategory = UNNotificationCategory(
            identifier: "ACTION_CONFIRMATION",
            actions: [confirmAction, cancelAction],
            intentIdentifiers: [],
            options: []
        )
        
        // Assistant Response Category
        let viewAction = UNNotificationAction(
            identifier: "VIEW_RESPONSE",
            title: "Voir la réponse",
            options: [.foreground]
        )
        let responseCategory = UNNotificationCategory(
            identifier: "ASSISTANT_RESPONSE",
            actions: [viewAction],
            intentIdentifiers: [],
            options: [.hiddenPreviewsShowTitle]
        )
        
        // Register all categories
        UNUserNotificationCenter.current().setNotificationCategories([actionCategory, responseCategory])
        print("[AppDelegate] Notification categories registered")
    }
    
    // Handle notification when app is in foreground
    func userNotificationCenter(
        _ center: UNUserNotificationCenter,
        willPresent notification: UNNotification,
        withCompletionHandler completionHandler: @escaping (UNNotificationPresentationOptions) -> Void
    ) {
        // Show notification even when app is in foreground
        completionHandler([.banner, .sound, .badge])
    }
    
    // Handle notification action
    func userNotificationCenter(
        _ center: UNUserNotificationCenter,
        didReceive response: UNNotificationResponse,
        withCompletionHandler completionHandler: @escaping () -> Void
    ) {
        let categoryId = response.notification.request.content.categoryIdentifier
        let actionId = response.actionIdentifier
        let userInfo = response.notification.request.content.userInfo
        
        // Clear badge
        DispatchQueue.main.async {
            UIApplication.shared.applicationIconBadgeNumber = 0
        }
        
        switch categoryId {
        case "ASSISTANT_RESPONSE":
            // Open chat tab
            DispatchQueue.main.async {
                AppState.shared.selectedTab = 0
            }
            
        case "ACTION_CONFIRMATION":
            if let actionQueueId = userInfo["action_id"] as? String {
                switch actionId {
                case "CONFIRM_ACTION":
                    Task {
                        await BackgroundTaskManager.shared.confirmAction(actionQueueId)
                    }
                    
                case "CANCEL_ACTION":
                    BackgroundTaskManager.shared.cancelAction(actionQueueId)
                    
                default:
                    // Default tap - open app to action details
                    DispatchQueue.main.async {
                        AppState.shared.selectedTab = 2 // Tools tab
                    }
                }
            }
            
        default:
            // Default - open chat
            DispatchQueue.main.async {
                AppState.shared.selectedTab = 0
            }
        }
        
        completionHandler()
    }
    
    // Background fetch
    func application(
        _ application: UIApplication,
        performFetchWithCompletionHandler completionHandler: @escaping (UIBackgroundFetchResult) -> Void
    ) {
        Task {
            await BackgroundTaskManager.shared.processQueue()
            completionHandler(.newData)
        }
    }
    
    // Handle remote notifications
    func application(
        _ application: UIApplication,
        didReceiveRemoteNotification userInfo: [AnyHashable: Any],
        fetchCompletionHandler completionHandler: @escaping (UIBackgroundFetchResult) -> Void
    ) {
        // Process any pending queue items
        Task {
            await BackgroundTaskManager.shared.processQueue()
            completionHandler(.newData)
        }
    }
}
