//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Data Models - Personnel Assistant                           ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import Foundation

// MARK: - User

struct User: Codable, Identifiable {
    let id: String
    let username: String
    let email: String
    let name: String
    let groups: [String]
    let avatar: String?
    
    enum CodingKeys: String, CodingKey {
        case id, username, email, name, groups
        case avatar = "avatar_url"
    }
    
    init(id: String, username: String, email: String, name: String, avatar: String?, groups: [String]) {
        self.id = id
        self.username = username
        self.email = email
        self.name = name
        self.avatar = avatar
        self.groups = groups
    }
}

// MARK: - Auth

struct AuthConfig: Codable {
    let authorizationUrl: String
    let tokenUrl: String
    let userinfoUrl: String
    let clientId: String
    let scope: String
    let responseType: String
    
    enum CodingKeys: String, CodingKey {
        case authorizationUrl = "authorization_url"
        case tokenUrl = "token_url"
        case userinfoUrl = "userinfo_url"
        case clientId = "client_id"
        case scope
        case responseType = "response_type"
    }
}

struct TokenResponse: Codable {
    let accessToken: String
    let tokenType: String
    let expiresIn: Int
    let refreshToken: String?
    let user: User?
    
    enum CodingKeys: String, CodingKey {
        case accessToken = "access_token"
        case tokenType = "token_type"
        case expiresIn = "expires_in"
        case refreshToken = "refresh_token"
        case user
    }
}

// MARK: - AI Provider (Centralized Definition)

enum AIProvider: String, Codable, CaseIterable {
    case openai
    case anthropic
    case ollama
    case auto
    case system
    case mock
    
    var displayName: String {
        switch self {
        case .openai: return "OpenAI GPT-4"
        case .anthropic: return "Claude"
        case .ollama: return "Ollama Local"
        case .auto: return "Auto (Optimal)"
        case .system: return "Système"
        case .mock: return "Mode Démo"
        }
    }
    
    var icon: String {
        switch self {
        case .openai: return "brain"
        case .anthropic: return "sparkles"
        case .ollama: return "server.rack"
        case .auto: return "wand.and.stars"
        case .system: return "terminal"
        case .mock: return "exclamationmark.triangle"
        }
    }
}

// MARK: - Tool Usage

struct ToolUsage: Codable, Identifiable {
    var id: String { name }
    let name: String
    let result: String?
    let status: String?
}

// MARK: - Conversation

struct Conversation: Identifiable, Codable {
    let id: String
    let title: String?
    let lastMessage: String?
    let messageCount: Int
    let createdAt: Date
    let updatedAt: Date
    let pinned: Bool
    
    enum CodingKeys: String, CodingKey {
        case id, title
        case lastMessage = "last_message"
        case messageCount = "message_count"
        case createdAt = "created_at"
        case updatedAt = "updated_at"
        case pinned
    }
}

// MARK: - Infrastructure

struct InfrastructureStatus: Codable {
    let status: String
    let components: [String: ComponentStatus]
    let timestamp: Date
}

struct ComponentStatus: Codable {
    let status: String
}

struct Tool: Identifiable, Codable {
    var id: String { name }
    let name: String
    let description: String
    let category: String
    let parameters: [String: Any]?
    
    enum CodingKeys: String, CodingKey {
        case name, description, category, parameters
    }
    
    init(from decoder: Decoder) throws {
        let container = try decoder.container(keyedBy: CodingKeys.self)
        name = try container.decode(String.self, forKey: .name)
        description = try container.decode(String.self, forKey: .description)
        category = try container.decode(String.self, forKey: .category)
        parameters = nil
    }
    
    func encode(to encoder: Encoder) throws {
        var container = encoder.container(keyedBy: CodingKeys.self)
        try container.encode(name, forKey: .name)
        try container.encode(description, forKey: .description)
        try container.encode(category, forKey: .category)
    }
}

// MARK: - WebSocket

struct WSMessage: Codable {
    let type: WSMessageType
    let payload: [String: String]?
    let timestamp: Date?
}

enum WSMessageType: String, Codable {
    case chat
    case ping
    case pong
    case status
    case error
    case thinking
    case chunk
    case complete
}

// MARK: - Attachments

struct Attachment: Identifiable {
    let id = UUID()
    let type: AttachmentType
    let name: String
    let data: Data
    let mimeType: String
}

enum AttachmentType: String {
    case pdf
    case image
    case document
    case spreadsheet
    case presentation
    case json
    case code
}

// MARK: - Stream Chunk

struct StreamChunk: Codable {
    let type: String
    let content: String
    let metadata: [String: String]?
}
