//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Personnel Assistant - Siri Intents                          ║
//  ║                    Voice Activation: "dmagh"                                   ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import AppIntents
import SwiftUI

// MARK: - App Shortcuts Provider

struct AssistantShortcuts: AppShortcutsProvider {
    static var appShortcuts: [AppShortcut] {
        AppShortcut(
            intent: DmaghVoiceIntent(),
            phrases: [
                "Ouvre \(.applicationName)",
                "Lance \(.applicationName)",
                "Démarre \(.applicationName)",
                "\(.applicationName) aide-moi",
                "\(.applicationName) écoute"
            ],
            shortTitle: "Ouvrir",
            systemImageName: "brain.head.profile"
        )
        
        AppShortcut(
            intent: StartVoiceCommandIntent(),
            phrases: [
                "Commande vocale avec \(.applicationName)",
                "Parler à \(.applicationName)",
                "Demander à \(.applicationName)",
                "\(.applicationName) écoute-moi"
            ],
            shortTitle: "Commande Vocale",
            systemImageName: "mic.fill"
        )
        
        AppShortcut(
            intent: CheckInfrastructureIntent(),
            phrases: [
                "État infrastructure avec \(.applicationName)",
                "Statut services \(.applicationName)",
                "\(.applicationName) statut infrastructure"
            ],
            shortTitle: "Infrastructure",
            systemImageName: "server.rack"
        )
        
        AppShortcut(
            intent: QuickChatIntent(),
            phrases: [
                "Ouvrir le chat \(.applicationName)",
                "Discuter avec \(.applicationName)",
                "Message à \(.applicationName)"
            ],
            shortTitle: "Chat",
            systemImageName: "message.fill"
        )
    }
}

// MARK: - Dmagh Voice Intent (Main Activation)

struct DmaghVoiceIntent: AppIntent {
    static var title: LocalizedStringResource = "Activer Personnel Assistant"
    static var description = IntentDescription("Activez votre assistant personnel en disant 'dmagh'")
    static var openAppWhenRun: Bool = true
    
    @MainActor
    func perform() async throws -> some IntentResult & ProvidesDialog & ShowsSnippetView {
        // Open app with voice mode
        await openVoiceMode()
        
        return .result(
            dialog: "Personnel Assistant activé. Je vous écoute...",
            view: DmaghActivationView()
        )
    }
    
    @MainActor
    private func openVoiceMode() async {
        // Post notification to start voice recording
        NotificationCenter.default.post(
            name: Notification.Name("StartVoiceRecording"),
            object: nil
        )
    }
}

struct DmaghActivationView: View {
    var body: some View {
        HStack(spacing: 12) {
            ZStack {
                Circle()
                    .fill(Color(hex: "DC143C").opacity(0.3))
                    .frame(width: 50, height: 50)
                
                Image(systemName: "waveform")
                    .font(.system(size: 24))
                    .foregroundColor(Color(hex: "DC143C"))
            }
            
            VStack(alignment: .leading, spacing: 4) {
                Text("Personnel Assistant")
                    .font(.headline)
                    .foregroundColor(.white)
                
                Text("Mode écoute activé...")
                    .font(.subheadline)
                    .foregroundColor(.gray)
            }
        }
        .padding()
        .background(Color(hex: "1A0A0A"))
        .cornerRadius(16)
    }
}

// MARK: - Start Voice Command Intent

struct StartVoiceCommandIntent: AppIntent {
    static var title: LocalizedStringResource = "Commande Vocale"
    static var description = IntentDescription("Démarrer une commande vocale avec Personnel Assistant")
    static var openAppWhenRun: Bool = true
    
    @MainActor
    func perform() async throws -> some IntentResult & ProvidesDialog {
        NotificationCenter.default.post(
            name: Notification.Name("StartVoiceRecording"),
            object: nil
        )
        
        return .result(dialog: "Mode écoute activé. Parlez maintenant...")
    }
}

// MARK: - Check Infrastructure Intent

struct CheckInfrastructureIntent: AppIntent {
    static var title: LocalizedStringResource = "Vérifier Infrastructure"
    static var description = IntentDescription("Vérifier l'état de l'infrastructure CodinCloud")
    static var openAppWhenRun: Bool = false
    
    @MainActor
    func perform() async throws -> some IntentResult & ProvidesDialog & ShowsSnippetView {
        // Fetch infrastructure status
        let status = await fetchInfrastructureStatus()
        
        return .result(
            dialog: "Infrastructure: \(status.summary)",
            view: InfrastructureStatusView(status: status)
        )
    }
    
    private func fetchInfrastructureStatus() async -> InfraStatus {
        // In production, this would call the API
        return InfraStatus(
            dockerServices: 24,
            dockerHealthy: 24,
            vpnNodes: 5,
            vpnConnected: 4,
            ssoStatus: true,
            apiStatus: true
        )
    }
}

struct InfraStatus {
    let dockerServices: Int
    let dockerHealthy: Int
    let vpnNodes: Int
    let vpnConnected: Int
    let ssoStatus: Bool
    let apiStatus: Bool
    
    var summary: String {
        if dockerHealthy == dockerServices && ssoStatus && apiStatus {
            return "Tous les systèmes opérationnels"
        } else {
            return "Certains services nécessitent attention"
        }
    }
}

struct InfrastructureStatusView: View {
    let status: InfraStatus
    
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            HStack {
                Image(systemName: "server.rack")
                    .foregroundColor(Color(hex: "DC143C"))
                Text("Infrastructure CodinCloud")
                    .font(.headline)
                    .foregroundColor(.white)
            }
            
            HStack(spacing: 16) {
                InfraStatusBadge(
                    icon: "cube.box.fill",
                    label: "Docker",
                    value: "\(status.dockerHealthy)/\(status.dockerServices)",
                    isHealthy: status.dockerHealthy == status.dockerServices
                )
                
                InfraStatusBadge(
                    icon: "network",
                    label: "VPN",
                    value: "\(status.vpnConnected)/\(status.vpnNodes)",
                    isHealthy: status.vpnConnected > 0
                )
                
                InfraStatusBadge(
                    icon: "lock.shield.fill",
                    label: "SSO",
                    value: status.ssoStatus ? "OK" : "Down",
                    isHealthy: status.ssoStatus
                )
            }
        }
        .padding()
        .background(Color(hex: "1A0A0A"))
        .cornerRadius(16)
    }
}

struct InfraStatusBadge: View {
    let icon: String
    let label: String
    let value: String
    let isHealthy: Bool
    
    var body: some View {
        VStack(spacing: 4) {
            Image(systemName: icon)
                .foregroundColor(isHealthy ? Color(hex: "228B22") : Color(hex: "FF4444"))
            Text(label)
                .font(.caption2)
                .foregroundColor(.gray)
            Text(value)
                .font(.caption)
                .fontWeight(.semibold)
                .foregroundColor(isHealthy ? Color(hex: "228B22") : Color(hex: "FF4444"))
        }
    }
}

// MARK: - Quick Chat Intent

struct QuickChatIntent: AppIntent {
    static var title: LocalizedStringResource = "Chat Rapide"
    static var description = IntentDescription("Ouvrir le chat Personnel Assistant")
    static var openAppWhenRun: Bool = true
    
    @Parameter(title: "Message", description: "Message à envoyer (optionnel)")
    var message: String?
    
    @MainActor
    func perform() async throws -> some IntentResult & ProvidesDialog {
        if let message = message {
            // Send the message via notification
            NotificationCenter.default.post(
                name: Notification.Name("SendChatMessage"),
                object: nil,
                userInfo: ["message": message]
            )
            return .result(dialog: "Message envoyé: \(message)")
        }
        
        return .result(dialog: "Chat ouvert")
    }
}

// MARK: - Ask Assistant Intent (for Siri conversational queries)

struct AskAssistantIntent: AppIntent {
    static var title: LocalizedStringResource = "Demander à l'Assistant"
    static var description = IntentDescription("Poser une question à Personnel Assistant")
    
    @Parameter(title: "Question")
    var question: String
    
    @MainActor
    func perform() async throws -> some IntentResult & ProvidesDialog {
        // In production, this would call the AI API
        let response = await processQuestion(question)
        
        return .result(dialog: IntentDialog(stringLiteral: response))
    }
    
    private func processQuestion(_ question: String) async -> String {
        // Simulate AI response - in production, call the backend API
        return "Je traite votre demande: \"\(question)\". Ouvrez l'application pour une réponse complète."
    }
}

// Note: Color.init(hex:) is defined in Theme.swift
