//
//  ╔═══════════════════════════════════════════════════════════════════════════════╗
//  ║                    Sophisticated Animations                                    ║
//  ╚═══════════════════════════════════════════════════════════════════════════════╝
//

import SwiftUI

// MARK: - Pulse Animation

struct PulseAnimation: View {
    @State private var isAnimating = false
    let color: Color
    let size: CGFloat
    
    var body: some View {
        ZStack {
            ForEach(0..<3, id: \.self) { index in
                Circle()
                    .stroke(color.opacity(0.5 - Double(index) * 0.15), lineWidth: 2)
                    .frame(width: size, height: size)
                    .scaleEffect(isAnimating ? 1.0 + CGFloat(index) * 0.3 : 0.8)
                    .opacity(isAnimating ? 0 : 1)
                    .animation(
                        Animation
                            .easeOut(duration: 1.5)
                            .repeatForever(autoreverses: false)
                            .delay(Double(index) * 0.3),
                        value: isAnimating
                    )
            }
            
            Circle()
                .fill(color)
                .frame(width: size * 0.3, height: size * 0.3)
                .shadow(color: color.opacity(0.8), radius: 10)
        }
        .onAppear {
            isAnimating = true
        }
    }
}

// MARK: - Blood Drop Animation

struct BloodDropAnimation: View {
    @State private var drops: [BloodDrop] = []
    let color: Color
    
    var body: some View {
        GeometryReader { geometry in
            ZStack {
                ForEach(drops) { drop in
                    Circle()
                        .fill(
                            RadialGradient(
                                colors: [color, color.opacity(0.5)],
                                center: .center,
                                startRadius: 0,
                                endRadius: drop.size
                            )
                        )
                        .frame(width: drop.size, height: drop.size)
                        .position(drop.position)
                        .opacity(drop.opacity)
                        .blur(radius: drop.blur)
                }
            }
            .onAppear {
                startAnimation(in: geometry.size)
            }
        }
    }
    
    func startAnimation(in size: CGSize) {
        Timer.scheduledTimer(withTimeInterval: 0.3, repeats: true) { _ in
            let drop = BloodDrop(
                position: CGPoint(
                    x: CGFloat.random(in: 0...size.width),
                    y: -20
                ),
                size: CGFloat.random(in: 4...12),
                opacity: Double.random(in: 0.3...0.8),
                blur: CGFloat.random(in: 0...2)
            )
            
            withAnimation(.easeIn(duration: 2)) {
                drops.append(drop)
            }
            
            DispatchQueue.main.asyncAfter(deadline: .now() + 2) {
                if let index = drops.firstIndex(where: { $0.id == drop.id }) {
                    drops.remove(at: index)
                }
            }
        }
    }
}

struct BloodDrop: Identifiable {
    let id = UUID()
    var position: CGPoint
    let size: CGFloat
    var opacity: Double
    let blur: CGFloat
}

// MARK: - Thinking Animation

struct ThinkingAnimation: View {
    @State private var isAnimating = false
    @EnvironmentObject var theme: ThemeManager
    
    var body: some View {
        HStack(spacing: 8) {
            ForEach(0..<3, id: \.self) { index in
                Circle()
                    .fill(theme.accentColor)
                    .frame(width: 10, height: 10)
                    .scaleEffect(isAnimating ? 1.2 : 0.8)
                    .opacity(isAnimating ? 1 : 0.5)
                    .animation(
                        Animation
                            .easeInOut(duration: 0.6)
                            .repeatForever(autoreverses: true)
                            .delay(Double(index) * 0.2),
                        value: isAnimating
                    )
            }
        }
        .onAppear {
            isAnimating = true
        }
    }
}

// MARK: - Glow Pulse

struct GlowPulse: View {
    @State private var isGlowing = false
    let color: Color
    let size: CGFloat
    
    var body: some View {
        Circle()
            .fill(color.opacity(0.3))
            .frame(width: size, height: size)
            .overlay(
                Circle()
                    .stroke(color, lineWidth: 2)
            )
            .shadow(color: color.opacity(isGlowing ? 0.8 : 0.3), radius: isGlowing ? 20 : 5)
            .scaleEffect(isGlowing ? 1.1 : 1.0)
            .animation(
                Animation
                    .easeInOut(duration: 1.5)
                    .repeatForever(autoreverses: true),
                value: isGlowing
            )
            .onAppear {
                isGlowing = true
            }
    }
}

// MARK: - Typing Indicator

struct TypingIndicator: View {
    @State private var currentDot = 0
    @EnvironmentObject var theme: ThemeManager
    
    var body: some View {
        HStack(spacing: 4) {
            ForEach(0..<3, id: \.self) { index in
                Circle()
                    .fill(currentDot == index ? theme.accentGlow : theme.primaryColor)
                    .frame(width: 8, height: 8)
                    .shadow(
                        color: currentDot == index ? theme.accentGlow : .clear,
                        radius: 5
                    )
            }
        }
        .padding(.horizontal, 16)
        .padding(.vertical, 12)
        .background(theme.surfaceElevated)
        .cornerRadius(theme.radiusLarge)
        .onAppear {
            Timer.scheduledTimer(withTimeInterval: 0.4, repeats: true) { _ in
                withAnimation(.easeInOut(duration: 0.2)) {
                    currentDot = (currentDot + 1) % 3
                }
            }
        }
    }
}

// MARK: - Message Appear Animation

struct MessageAppearAnimation: ViewModifier {
    @State private var isVisible = false
    let delay: Double
    
    func body(content: Content) -> some View {
        content
            .opacity(isVisible ? 1 : 0)
            .offset(y: isVisible ? 0 : 20)
            .scaleEffect(isVisible ? 1 : 0.9)
            .onAppear {
                withAnimation(.spring(response: 0.5, dampingFraction: 0.7).delay(delay)) {
                    isVisible = true
                }
            }
    }
}

// MARK: - Ripple Effect

struct RippleEffect: View {
    @State private var ripples: [Ripple] = []
    let color: Color
    
    var body: some View {
        ZStack {
            ForEach(ripples) { ripple in
                Circle()
                    .stroke(color.opacity(ripple.opacity), lineWidth: 2)
                    .frame(width: ripple.size, height: ripple.size)
            }
        }
    }
    
    func trigger(at position: CGPoint) {
        let ripple = Ripple(size: 0, opacity: 1)
        ripples.append(ripple)
        
        withAnimation(.easeOut(duration: 0.8)) {
            if let index = ripples.firstIndex(where: { $0.id == ripple.id }) {
                ripples[index].size = 200
                ripples[index].opacity = 0
            }
        }
        
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.8) {
            ripples.removeAll { $0.id == ripple.id }
        }
    }
}

struct Ripple: Identifiable {
    let id = UUID()
    var size: CGFloat
    var opacity: Double
}

// MARK: - Particle System

struct ParticleSystem: View {
    @State private var particles: [Particle] = []
    let color: Color
    let count: Int
    
    var body: some View {
        GeometryReader { geometry in
            ZStack {
                ForEach(particles) { particle in
                    Circle()
                        .fill(color.opacity(particle.opacity))
                        .frame(width: particle.size, height: particle.size)
                        .position(particle.position)
                        .blur(radius: particle.blur)
                }
            }
            .onAppear {
                generateParticles(in: geometry.size)
            }
        }
    }
    
    func generateParticles(in size: CGSize) {
        for _ in 0..<count {
            let particle = Particle(
                position: CGPoint(
                    x: CGFloat.random(in: 0...size.width),
                    y: CGFloat.random(in: 0...size.height)
                ),
                size: CGFloat.random(in: 2...6),
                opacity: Double.random(in: 0.1...0.5),
                blur: CGFloat.random(in: 0...1),
                velocity: CGPoint(
                    x: CGFloat.random(in: -0.5...0.5),
                    y: CGFloat.random(in: -0.3...0.3)
                )
            )
            particles.append(particle)
        }
        
        Timer.scheduledTimer(withTimeInterval: 0.05, repeats: true) { _ in
            for i in particles.indices {
                particles[i].position.x += particles[i].velocity.x
                particles[i].position.y += particles[i].velocity.y
                
                if particles[i].position.x < 0 || particles[i].position.x > size.width {
                    particles[i].velocity.x *= -1
                }
                if particles[i].position.y < 0 || particles[i].position.y > size.height {
                    particles[i].velocity.y *= -1
                }
            }
        }
    }
}

struct Particle: Identifiable {
    let id = UUID()
    var position: CGPoint
    let size: CGFloat
    var opacity: Double
    let blur: CGFloat
    var velocity: CGPoint
}

// MARK: - View Extensions

extension View {
    func messageAppear(delay: Double = 0) -> some View {
        modifier(MessageAppearAnimation(delay: delay))
    }
}
